/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity;

import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.apache.http.Header;

/**
 * Provides the information on OData exceptions.
 */
public class ODataException extends Exception
{
    private static final long serialVersionUID = 1885636855371872127L;
    
    private ODataExceptionType type;
    
    String code;

	private String message;
	
	protected int httpStatusCode;
	
	protected String httpStatusLine;
	
	protected transient Header[] httpHeaders;
	
	protected String httpErrorResponseBody;
	
	public void setHttpErrorResponseBody(String httpErrorResponseBody) {
		this.httpErrorResponseBody = httpErrorResponseBody;
	}

	private static ResourceBundle bundle;
	
	private static final String DEFAULT_BUNDLE_NAME = "odata-exceptions-i18n";

	/** Constructs the ODataException object.
     * @param exceptionType Type of exception. If it is null, then the default exception type is ODataExceptionType.OTHER.
     * @param s String containing the error message
     * @param e The exception object
     */
	public ODataException( ODataExceptionType exceptionType, final String s, final Throwable e )
    {
        super(s, e);
        if(exceptionType == null){
    		type = ODataExceptionType.OTHER;
    		this.message = getMessage(ODataExceptionType.OTHER.name(),Locale.ENGLISH) + ": "  + s;
        }else{
        	type = exceptionType;
        	this.message = getMessage(exceptionType.name(),Locale.ENGLISH) + ": " + s;
        }
    }
    
	/** Constructs the ODataException object.
     * @param exceptionType Type of exception. If it is null, then the default exception type is ODataExceptionType.OTHER.
     * @param String containing the error message
     * @param e The exception object
     */
	public ODataException( ODataExceptionType exceptionType, final String s)
    {
    	if(exceptionType == null){
    		type = ODataExceptionType.OTHER;
    		this.message = getMessage(ODataExceptionType.OTHER.name(),Locale.ENGLISH) + ": "  + s;
        }else{
        	type = exceptionType;
        	this.message = getMessage(exceptionType.name(),Locale.ENGLISH) + ": "  + s;
        }
    }
    
	/** Constructs the ODataException object.
     * @param exceptionType Type of exception. If it is null, then the default exception type is ODataExceptionType.OTHER.
     * @param String containing the error message
     * @param status containing  HttpStatusCode   
     * @param e The exception object  
     */
	public ODataException(ODataExceptionType exceptionType, final String s, int statusCode, final Throwable e){
		super(s, e);
		if(exceptionType == null){
    		type = ODataExceptionType.OTHER;
    		this.message = getMessage(ODataExceptionType.OTHER.name(),Locale.ENGLISH) + ": "  + s;
    		this.httpStatusCode = statusCode;
        }else{
        	type = exceptionType;
        	this.message = getMessage(exceptionType.name(),Locale.ENGLISH) + ": "  + s;
        	this.httpStatusCode = statusCode;
        }	
	}
	
    /**
     * Returns the ODataExceptionType associated with this ODataException.
     * @return ODataExceptionType.
     */
    public ODataExceptionType getODataExceptionType() {
    	return type;
    }
    
    public ODataException() {
    	type = ODataExceptionType.ODATA_OPERATION_EXECUTION_FAILED;
    }
    
    public void setCode(String code) {
    	this.code = code;
    }
    
    public void setMessage(String message) {
    	this.message = message;
    }
    
    @Override
    public String getMessage() {
    	return message;
    }
 
    /**
     * Returns the error code from the backend odata service if any. Note that this will give null if the ODataException is caused by something 
     * other than backend error, such as input serialization failure.
     * @return Error code as a String.
     */
    public String getCode() {
    	return code;
    }
    
    public int getHttpStatusCode() {
		return httpStatusCode;
	}

	public String getHttpStatusLine() {
		return httpStatusLine;
	}

	public Header[] getHttpHeaders() {
		return httpHeaders;
	}

	public String getHttpErrorResponseBody() {
		return httpErrorResponseBody;
	}
	
	/**
	 * Get the message based on the locale.
	 * @param key to get the exception message.
	 * @param locale object.
	 * @return
	 */
	private static String getMessage(String key,Locale locale) {
		try {
			bundle =  ResourceBundle.getBundle(DEFAULT_BUNDLE_NAME, locale);
			return bundle.getString(key);
		} catch (final MissingResourceException e) {
			return null;
		}
    }
}
