/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity;

import org.apache.olingo.odata2.api.edm.EdmLiteralKind;
import org.apache.olingo.odata2.api.edm.EdmSimpleType;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeException;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeKind;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.core.edm.EdmDateTime;
import org.apache.olingo.odata2.core.edm.EdmDateTimeOffset;
import org.joda.time.DateTimeZone;
import org.joda.time.LocalDate;
import org.joda.time.LocalDateTime;
import org.joda.time.LocalTime;
import org.joda.time.ReadableInstant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import lombok.NonNull;
import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor(staticName = "of")
final class ODataTypeValueSerializer {
	private static final boolean KEEP_SECOND_FRACTIONS = false;

	@NonNull
	private final EdmType type;
	private static final Logger logger = LoggerFactory.getLogger(ODataTypeValueSerializer.class);

	public static ODataTypeValueSerializer of(final EdmSimpleTypeKind typeKind) {
		return of(typeKind.getEdmSimpleTypeInstance());
	}

	public String toUri(@NonNull Object value) {
		// if value is a property declaration
		if (value instanceof ODataProperty) {
			return ((ODataProperty<?>) value).getField();
		}

		// Olingo serializes any dates with GMT/UTC timezone, that's why this
		// timezone is forced to the input.
		if (type instanceof EdmDateTime || type instanceof EdmDateTimeOffset) {
			if (value instanceof LocalDate) {
				value = (((LocalDate) value).toDateTime(LocalTime.MIDNIGHT, DateTimeZone.UTC)).getMillis();
			} else if (value instanceof LocalDateTime) {
				value = ((LocalDateTime) value).toDateTime(DateTimeZone.UTC).getMillis();
			} else if (value instanceof ReadableInstant) {
				final long ms = ((ReadableInstant) value).getMillis();
				final long offsetInMilliseconds = ((ReadableInstant) value).getZone().getOffset(ms);
				value = ms + offsetInMilliseconds;
			}

			if (value instanceof Long && !KEEP_SECOND_FRACTIONS) {
				value = (((long) value) / 1000) * 1000;
			}
		}

		try {
			return ((EdmSimpleType) type).valueToString(value, EdmLiteralKind.URI, null);
		} catch (final EdmSimpleTypeException e) {
			//Added log 
			logger.error("Error while processing Date . Java 8 Date Time API are not supported", e);
			return null;
		}
	}
}
