package com.sap.cloud.sdk.odatav2.connectivity.batch;

import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.sap.cloud.sdk.cloudplatform.cache.CacheKey;
import com.sap.cloud.sdk.odatav2.connectivity.ErrorResultHandler;
import com.sap.cloud.sdk.odatav2.connectivity.ODataQuery;
import com.sap.cloud.sdk.odatav2.connectivity.impl.BatchRequestImpl;

/*
 * Represents a builder to create a batch request.
 */
public class BatchRequestBuilder {

  private static Map<String, String> headers = new HashMap<String, String>();

  private final String servicePath;

  private List batch = new ArrayList();

  private CacheKey cacheKey;

  private boolean isCacheRefresh;

  private URL metadataFilePath;

  private boolean cacheMetadata;

  private ErrorResultHandler<?> errorHandler;

  private Map<String, String> destinationRelevantHeaders = new HashMap<>();

  private BatchRequestBuilder(String servicePath) {
    this.servicePath = servicePath;
  }

  /**
   * Builds an BatchRequest from this builder.
   * @return an BatchRequest object.
   */
  public BatchRequest build() {
    return new BatchRequestImpl(errorHandler, destinationRelevantHeaders, cacheMetadata,
        metadataFilePath, cacheKey, headers,
        isCacheRefresh, batch, servicePath);

  }

  /**
   * Adds a query request to the batch.
   * @param queryRequest Represents the query request
   * @return BatchRequestBuilder
   */
  public BatchRequestBuilder addQueryRequest(ODataQuery queryRequest) {
    this.batch.add(queryRequest);
    return this;
  }

  /**
   * Adds a change set request to the batch.
   * @param changeSet Represents the change set
   * @return BatchRequestBuilder
   */
  public BatchRequestBuilder addChangeSet(ChangeSet changeSet) {
    this.batch.add(changeSet);
    return this;
  }

  /**
   * Specifies the service where the batch request will be executed.
   * @param serviceName Name of the OData service
   * @return BatchRequestBuilder
   */
  public static BatchRequestBuilder withService(String serviceName) {
    return new BatchRequestBuilder(serviceName);

  }
  
  
  /**
   * Adds a header to the batch request.
   * @param key name of the header
   * @param value value of the header
   * @return BatchRequestBuilder
   */
  public BatchRequestBuilder withHeader(String key, String value) {
      return withHeader(key, value, false);
    }
    
    /**
     * Adds a header to the batch request and optionally to the metadata request as well depending on the value of the 
     * passInAllRequests parameter.
     * @param key name of the header
     * @param value value of the header
     * @param passInAllRequests boolean indicating whether the header is to be passed in all the requests to the backend like $metadata call, CSRF Token fetch etc. made as part of the Create Request call
     * @return BatchRequestBuilder
     */
  public BatchRequestBuilder withHeader(String key, String value, boolean passInAllRequests) {
    
    if(passInAllRequests)
      destinationRelevantHeaders.put(key, value);//These headers are added to the metadata request.
    
    if(key.equals("SAP-PASSPORT") && !passInAllRequests)
        destinationRelevantHeaders.put(key, value);// The header "SAP-PASSPORT" is added to metadata request even though the 'passInAllRequests' us false.
      
    headers.put(key, value);//All headers must be considered for the actual OData operation.
    return this;
    

  }

  /**
   * Sets an error handler in this BatchRequestBuilder.
   * @param errorHandler
   * @return BatchRequestBuilder
   */
  public BatchRequestBuilder withErrorHandler(ErrorResultHandler<?> errorHandler) {
    this.errorHandler = errorHandler;
    return this;
  }

  /**
   * Enables caching of the metadata of an OData V2 data source. If your application is running on a tenant, then the
   * tenant ID along with the metadata URL is used to form the cache key.
   * @return ODataCreateRequestBuilder
   */
  public BatchRequestBuilder enableMetadataCache() {
    this.cacheMetadata = true;
    return this;
  }

  /**
   * Enables caching of the metadata of an OData V2 data source.
   * @param key {@link com.sap.cloud.sdk.cloudplatform.cache.CacheKey Cache key} containing the ID of the tenant where
   * the application runs. You can also include the user name and other parameters in the cache key.
   * @return BatchRequestBuilder
   */
  public BatchRequestBuilder enableMetadataCache(CacheKey cacheKey) {
    this.cacheKey = cacheKey;
    this.cacheMetadata = true;
    return this;
  }

  /**
   * Replaces the existing metadata in the cache with the latest version from the OData V2 data source.
   * @return BatchRequestBuilder
   */
  public BatchRequestBuilder withCacheRefresh() {
    this.isCacheRefresh = true;
    return this;
  }

  /**
   * Gets the metadata from the specified path.
   * @param metadataFilePath URL pointing to the metadata information
   * @return BatchRequestBuilder A builder for forming the Create
   */
  public BatchRequestBuilder withMetadata(URL metadataFilePath) {
    this.metadataFilePath = metadataFilePath;
    return this;
  }
}
