/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity.filter;

import java.util.Stack;

import com.sap.cloud.sdk.odatav2.connectivity.FilterExpression;
import com.sap.cloud.sdk.odatav2.connectivity.ODataType;
import com.sap.cloud.sdk.odatav2.connectivity.filter.FilterFunctions.FUNCTIONS;
import com.sap.cloud.sdk.service.prov.api.filter.BinaryExpressionNode;
import com.sap.cloud.sdk.service.prov.api.filter.Expression;
import com.sap.cloud.sdk.service.prov.api.filter.ExpressionNode;
import com.sap.cloud.sdk.service.prov.api.filter.ExpressionOperatorTypes.NODE_KIND;
import com.sap.cloud.sdk.service.prov.api.filter.FunctionNode;
import com.sap.cloud.sdk.service.prov.api.filter.LiteralNode;
import com.sap.cloud.sdk.service.prov.api.filter.PropertyNode;
import com.sap.cloud.sdk.service.prov.api.filter.UnaryExpressionNode;

public class FilterExpressionConverter {
	private enum OPERANDKIND {
		PROPERTY, LITERAL, EXPRESSION, FUNCTION
	};

	private enum OPERATORKIND {
		BINARY, UNARY, FUNCTION
	}

	private class OperandNode {
		private Object key;
		private OPERANDKIND kind;

		public OperandNode(Object path, OPERANDKIND opKind) {
			key = path;
			kind = opKind;
		}

		public Object getKey() {
			return key;
		}

		public OPERANDKIND getKind() {
			return kind;
		}
	}

	private class OperatorNode {
		private String operatorValue;
		private OPERATORKIND kind;

		public OperatorNode(String operator, OPERATORKIND opkind) {
			operatorValue = operator;
			kind = opkind;
		}

		public String getOperatorValue() {
			return operatorValue;
		}

		public OPERATORKIND getKind() {
			return kind;
		}
	}

	private static final int BINARYOPERAND = 2;
	private static final int UNARYOPERAND = 1;

	/**
	 * Converts the filter expression retrieved from the QueryRequest object into
	 * the filter expression that can be used with the consumption APIs of the
	 * ODataQueryBuilder object.
	 * 
	 * @param filterExp
	 *            Filter expression from QueryRequest
	 * @return FilterExpression The converted filter expression that can be passed
	 *         into the OData V2 consumption APIs
	 * @throws FilterFunctionException
	 */
	public static FilterExpression convertTo(Expression filterExp) throws FilterFunctionException {
		Stack<OperatorNode> operatorStack = new Stack<OperatorNode>();
		Stack<OperandNode> operandStack = new Stack<OperandNode>();
		return convertTo(filterExp, operatorStack, operandStack);
	}
	
	public static FilterExpression convertTo(Expression filterExp, Stack<OperatorNode> operatorStack, Stack<OperandNode> operandStack) throws FilterFunctionException {
		FilterExpression result = null;
		if (filterExp instanceof ExpressionNode) {
			ExpressionNode filterNode = (ExpressionNode) filterExp;

			if (filterNode.getKind().equals(NODE_KIND.BINARY)) {
				operatorStack
						.push(new FilterExpressionConverter().new OperatorNode(((BinaryExpressionNode) filterNode).getOperator(), OPERATORKIND.BINARY));
				convertTo(((BinaryExpressionNode) filterNode).getFirstChild(), operatorStack, operandStack);
				convertTo(((BinaryExpressionNode) filterNode).getSecondChild(), operatorStack, operandStack);
				FilterExpression fxp = processStacksToCreateExpression(OPERATORKIND.BINARY, BINARYOPERAND, operatorStack, operandStack);
				operandStack.push(new FilterExpressionConverter().new OperandNode(fxp, OPERANDKIND.EXPRESSION));
			} else if (filterNode.getKind() == NODE_KIND.PROPERTY) {
				operandStack.push(new FilterExpressionConverter().new OperandNode(((PropertyNode) filterNode).getPath(), OPERANDKIND.PROPERTY));
			} else if (filterNode.getKind() == NODE_KIND.LITERAL) {
				operandStack.push(new FilterExpressionConverter().new OperandNode(((LiteralNode) filterNode).getValue(), OPERANDKIND.LITERAL));
			} else if (filterNode.getKind() == NODE_KIND.UNARY) {
				operatorStack
						.push(new FilterExpressionConverter().new OperatorNode(((UnaryExpressionNode) filterNode).getOperator(), OPERATORKIND.UNARY));
				convertTo(((UnaryExpressionNode) filterNode).getChild(), operatorStack, operandStack);
				FilterExpression fxp = processStacksToCreateExpression(OPERATORKIND.UNARY, UNARYOPERAND, operatorStack, operandStack);
				operandStack.push(new FilterExpressionConverter().new OperandNode(fxp, OPERANDKIND.EXPRESSION));
			} else if (filterNode.getKind() == NODE_KIND.FUNCTION) {
				FunctionNode fnode = (FunctionNode) filterNode;
				operatorStack.push(new FilterExpressionConverter().new OperatorNode(fnode.getFunctionName(), OPERATORKIND.FUNCTION));
				for (ExpressionNode fnodeParam : fnode.getParameters()) {
					convertTo(fnodeParam, operatorStack, operandStack);
				}
				OperandNode operand;
				String func = fnode.getFunctionName();
				if(func.equalsIgnoreCase(FUNCTIONS.STARTSWITH.name()) || func.equalsIgnoreCase(FUNCTIONS.ENDSWITH.name())||
						func.equalsIgnoreCase(FUNCTIONS.SUBSTRINGOF.name())){
					FilterExpression exp = processStacksToCreateFilterFunctionExp(fnode.getParameters().size(), operatorStack, operandStack);
					operand = new FilterExpressionConverter().new OperandNode(exp, OPERANDKIND.FUNCTION);
				}
				else{
					String filterFunc = processStacksToCreateFilterFunction(fnode.getParameters().size(), operatorStack, operandStack);
					operand = new FilterExpressionConverter().new OperandNode(filterFunc, OPERANDKIND.FUNCTION);
				}
				operandStack.push(operand);
			}

			if (operatorStack.isEmpty() && operandStack.size() == 1) {
				result = (FilterExpression) operandStack.pop().getKey();
			}
		}
		return result;
	}

	private static FilterExpression processStacksToCreateFilterFunctionExp(int size, Stack<OperatorNode> operatorStack, Stack<OperandNode> operandStack) {

		OperandNode[] nodes = new OperandNode[size];
		for (int i = 0; i < size; i++) {
			nodes[i] = operandStack.pop();
		}
		OperatorNode operator = operatorStack.pop();

		String property = "";
		Object op1 = null;
		Object op2 = null;
		if(operator.operatorValue.equalsIgnoreCase(FUNCTIONS.SUBSTRINGOF.name())){
			property = (nodes[0].getKind() == OPERANDKIND.PROPERTY || nodes[0].getKind() == OPERANDKIND.FUNCTION) ? nodes[0].getKey().toString() : null;
			op1 = nodes[1].getKind() == OPERANDKIND.LITERAL ? nodes[1].getKey() : null;
		}
		if(op1 == null){
			property = getOperandAsProperty(nodes);
			op1 = getOperandasFirstOperand(nodes);
			op2 = getOperandAsSecondOperand(nodes);
		}
		if (FilterConverterHelper.isStringFunction(operator.getOperatorValue())) {
			return FilterConverterHelper.createV2StringFunctionExpression(operator.operatorValue, property, op1, op2,
					nodes.length);
		}
		return null;
	}

	private static String processStacksToCreateFilterFunction(int size, Stack<OperatorNode> operatorStack, Stack<OperandNode> operandStack) throws FilterFunctionException {
		OperandNode[] nodes = new OperandNode[size];
		for (int i = 0; i < size; i++) {
			nodes[i] = operandStack.pop();
		}
		OperatorNode operator = operatorStack.pop();
		return createV2FilterFunction(nodes, operator).getStringValue();
	}

	private static FilterExpression processStacksToCreateExpression(OPERATORKIND kind, int operandCount, Stack<OperatorNode> operatorStack, Stack<OperandNode> operandStack) {
		OperandNode node[] = new OperandNode[operandCount];
		OperatorNode operator = null;
		FilterExpression v2Exp = null;
		switch (kind) {
		case BINARY:
			node[0] = operandStack.pop();
			node[1] = operandStack.pop();
			operator = operatorStack.pop();
			v2Exp = createV2FilterExpression(node[0], node[1], operator);
			break;
		case UNARY:
			operatorStack.pop();
			node[0] = operandStack.pop();
			FilterExpression exp = (FilterExpression) node[0].getKey();
			v2Exp = FilterExpression.not(exp);
			break;
		default:
			break;
		}
		return v2Exp;
	}

	private static BaseFilterValue createV2FilterFunction(OperandNode[] nodes, OperatorNode operator)
			throws FilterFunctionException {
		String property = "";
		Object op1 = null;
		Object op2 = null;
		property = getOperandAsProperty(nodes);
		op1 = getOperandasFirstOperand(nodes);
		op2 = getOperandAsSecondOperand(nodes);
		if (FilterConverterHelper.isStringFunction(operator.getOperatorValue())) {
			return FilterConverterHelper.createV2StringFunction(operator.operatorValue, property, op1, op2,
					nodes.length);
		} else if (FilterConverterHelper.isMathFunction(operator.getOperatorValue())) {
			return FilterConverterHelper.createV2MathFunction(operator.operatorValue, property, op1);
		} else if (FilterConverterHelper.isDateFunction(operator.getOperatorValue())) {
			return FilterConverterHelper.createV2DateFunction(operator.operatorValue, property, op1);
		} else {
			throw new FilterFunctionException("Filter function is not supported :" + operator.operatorValue);
		}

	}

	private static String getOperandAsProperty(OperandNode[] nodes) {
		if (nodes.length == 3) {
			return (nodes[2].getKind() == OPERANDKIND.PROPERTY || nodes[2].getKind() == OPERANDKIND.FUNCTION) ? nodes[2].getKey().toString() : null;
		} else if (nodes.length == 2) {
			return (nodes[1].getKind() == OPERANDKIND.PROPERTY || nodes[1].getKind() == OPERANDKIND.FUNCTION) ? nodes[1].getKey().toString() : null;
		} else {
			return (nodes[0].getKind() == OPERANDKIND.PROPERTY || nodes[0].getKind() == OPERANDKIND.FUNCTION) ? nodes[0].getKey().toString() : null;
		}
	}

	private static Object getOperandasFirstOperand(OperandNode[] nodes) {
		if (nodes.length == 3) {
			return (nodes[1].getKind() == OPERANDKIND.LITERAL || nodes[1].getKind() == OPERANDKIND.FUNCTION) ? nodes[1].getKey() : null;
		} else if (nodes.length == 2) {
			return (nodes[0].getKind() == OPERANDKIND.LITERAL || nodes[0].getKind() == OPERANDKIND.FUNCTION) ? nodes[0].getKey() : null;
		} else {
			return null;
		}}

	private static Object getOperandAsSecondOperand(OperandNode[] nodes) {
		if (nodes.length == 3) {
			return nodes[0].getKind() == OPERANDKIND.LITERAL ? nodes[0].getKey() : null;
		} else {
			return null;
		}
	}	

	private static FilterExpression createV2FilterExpression(OperandNode node1, OperandNode node2,
			OperatorNode operator) {
		FilterExpression result = null;
		boolean isRightExpressionNode = false;
		FilterExpression left = null;
		FilterExpression right = null;
		Object literalValue = null;
		String propertyValue = "";
		boolean isFunction = false;
		boolean isMultiFunction = false;
		// more optimization later possible
		switch (node1.getKind()) {
		case EXPRESSION:
			isRightExpressionNode = true;
			right = (FilterExpression) node1.getKey();
			if(node2.getKind() == OPERANDKIND.FUNCTION) {
				isFunction = true;
			}
			break;
		case LITERAL:
			literalValue = node1.getKey();
			break;
		case PROPERTY:
			propertyValue = node1.getKey().toString();
			break;
		case FUNCTION:
			propertyValue = node1.getKey().toString();
			isFunction = true;
			break;
		default:
			break;

		}
		switch (node2.getKind()) {
		case EXPRESSION:
			if(isFunction) {
				right = (FilterExpression) node1.getKey();
				left =  (FilterExpression) node2.getKey();
				isMultiFunction = true;
			}
			else {
				left = (FilterExpression) node2.getKey();
			}
			break;
		case LITERAL:
			literalValue = node2.getKey();
			break;
		case PROPERTY:
			propertyValue = node2.getKey().toString();
			break;
		case FUNCTION:
			if(isFunction) {
				right = (FilterExpression) node1.getKey();
				left =  (FilterExpression) node2.getKey();
				isMultiFunction = true;
			}
			else {
				propertyValue = node2.getKey().toString();
				isFunction = true;
			}
			break;
		default:
			break;

		}
		// currently only leftexpression & right literal or vice versa not supported
		if (isRightExpressionNode && left != null) {
			if (operator.getOperatorValue().equals("AND")) {
				result = left.and(right);
			} else if (operator.getOperatorValue().equals("OR")) {
				result = left.or(right);
			}
		} else {
			if(isFunction) {
				if(isMultiFunction && left != null) {
					if (operator.getOperatorValue().equals("AND")) {
						result = left.and(right);
					} else if (operator.getOperatorValue().equals("OR")) {
						result = left.or(right);
					}
				}
				else {
					if (literalValue == null) {
						result = new FilterExpression(propertyValue, operator.getOperatorValue().toLowerCase(),
								null, true);
					}
					else {
						result = new FilterExpression(propertyValue, operator.getOperatorValue().toLowerCase(),
								ODataType.of(literalValue), true);
					}			
				}	
			}
			else {
				if (literalValue == null) {
					result = new FilterExpression(propertyValue, operator.getOperatorValue().toLowerCase(),
							null);
				}
				else {
					result = new FilterExpression(propertyValue, operator.getOperatorValue().toLowerCase(),
							ODataType.of(literalValue));
				}
			}
			
		}
		return result;
	}
}
