/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.odatav2.connectivity.impl;

import static com.sap.cloud.sdk.odatav2.connectivity.internal.ODataConnectivityUtil.SEPARATOR_PATH;
import static com.sap.cloud.sdk.odatav2.connectivity.internal.ODataConnectivityUtil.withSeparator;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.entity.StringEntity;
import org.apache.olingo.odata2.api.edm.Edm;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmLiteralKind;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.edm.EdmSimpleType;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeException;
import org.apache.olingo.odata2.api.ep.EntityProviderException;
import org.apache.olingo.odata2.api.processor.ODataResponse;
import org.apache.olingo.odata2.client.api.ODataClient;
import org.apache.olingo.odata2.client.api.ep.Entity;
import org.apache.olingo.odata2.client.api.ep.EntitySerializerProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.cloudplatform.cache.CacheKey;
import com.sap.cloud.sdk.cloudplatform.connectivity.Destination;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpClientAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.WithDestinationName;
import com.sap.cloud.sdk.odatav2.connectivity.ErrorResultHandler;
import com.sap.cloud.sdk.odatav2.connectivity.ODataException;
import com.sap.cloud.sdk.odatav2.connectivity.ODataExceptionType;
import com.sap.cloud.sdk.odatav2.connectivity.ODataUpdateRequest;
import com.sap.cloud.sdk.odatav2.connectivity.ODataUpdateResult;
import com.sap.cloud.sdk.odatav2.connectivity.UpdateMethod;
import com.sap.cloud.sdk.odatav2.connectivity.cache.metadata.GuavaMetadataCache;
import com.sap.cloud.sdk.odatav2.connectivity.cache.metadata.MetadataCache;
import com.sap.cloud.sdk.odatav2.connectivity.internal.EdmWithCSRF;
import com.sap.cloud.sdk.odatav2.connectivity.internal.ODataConnectivityUtil;
import com.sap.cloud.sdk.odatav2.connectivity.internal.UrlEncoder;

public class ODataUpdateRequestImpl implements ODataUpdateRequest {
	private static Logger logger = LoggerFactory.getLogger(ODataUpdateRequestImpl.class);
	private static final String UTF_8 = "UTF-8";
	private static final String APPLICATION_JSON = "application/json";
	public static final String AUTHORIZATION_HEADER = "Authorization";
	private static MetadataCache metadataCache = new GuavaMetadataCache();
	/**
	 * Content Type Constants
	 */
	public static final String CONTENT_TYPE = "Content-Type";
	public static final String ACCEPT_HEADER = "Accept";
	public static final String CONTENTTYPE_ATOM_XML = "application/xml";
	private static final String STRING = "String";
	private String serviceName;
	private String entitySetName;
	private Map<String, Object> body;
	private Map<String, Object> keys;
	private ErrorResultHandler<?> errorHandler;
	private Map<String, String> headers;
	private Map<String, String> destinationRelevantHeaders;
	private Boolean cacheMetadata;
	private URL metadataFilePath;
	private CacheKey cacheKey;
	private boolean isCacheRefresh;

	public ODataUpdateRequestImpl(String serviceName, String entitySetName, Map<String, Object> keys,
			Map<String, Object> body, ErrorResultHandler<?> errorHandler, Map<String, String> headers,
			Map<String, String> destinationRelevantHeaders, boolean metadataCache, URL metadataFilePath,CacheKey cacheKey, boolean isCacheRefresh) {
		this.serviceName = serviceName;
		this.entitySetName = entitySetName;
		this.body = body;
		this.keys = keys;
		this.errorHandler = errorHandler;
		this.headers = headers;
		this.destinationRelevantHeaders = destinationRelevantHeaders;
		this.cacheMetadata = metadataCache;
		this.metadataFilePath = metadataFilePath;
		this.cacheKey = cacheKey;
		this.isCacheRefresh = isCacheRefresh;
	}

	@Override
	public ODataUpdateResult execute(UpdateMethod updateMethod, String destinationName) throws ODataException {
		logger.debug("Update Called with Destination Name: " + destinationName);
		return handleExecute(updateMethod, destinationName, null);
	}

	private ODataUpdateResult handleExecute(UpdateMethod updateMethod, String destinationName, HttpClient httpClient) throws ODataException {
		ODataUpdateResult result = null;
		httpClient = destinationName == null ? httpClient : getHttpClient(destinationName);
		if (cacheMetadata) {
			try {
				result = update(httpClient, updateMethod);
			} catch (ODataException e) {
				if (e.getODataExceptionType().equals(ODataExceptionType.OTHER)
						|| e.getODataExceptionType().equals(ODataExceptionType.ODATA_OPERATION_EXECUTION_FAILED)) {
					throw e;
				} else {
					this.isCacheRefresh = true; // set the isCacheRefresh to true so the entry in the cache is removed for the particular cacheKey.
					result = update(httpClient, updateMethod);
				}
			}
		} else {
			result = update(httpClient, updateMethod);
		}
		return result;
	}

	ODataUpdateResult update(HttpClient httpClient, UpdateMethod um) throws ODataException {
		String serviceUri = this.serviceName;
		String entitySetName = this.entitySetName;
		String completeUrl = null;
		HttpResponse httpResponse = null;
		String contentType = APPLICATION_JSON;
		// To get the EdmEntitySet Object
		// Get the edm object first.
		Edm edm = null;
		ODataResponse response = null;
		Entity entity = null;
		EdmWithCSRF edmWithCSRF = null;
		try {
			edmWithCSRF = ODataConnectivityUtil.readMetadataWithCSRF(serviceUri, httpClient, destinationRelevantHeaders,
					errorHandler, cacheMetadata, metadataFilePath,cacheKey,isCacheRefresh);
		} catch (IOException e) {
			throw new ODataException(ODataExceptionType.METADATA_FETCH_FAILED, "Metadata fetch failed!", e);
		}
		if (edmWithCSRF == null || edmWithCSRF.getEdm() == null) {
			throw new ODataException(ODataExceptionType.METADATA_FETCH_FAILED, "Metadata fetch failed!", null);
		}
		edm = edmWithCSRF.getEdm();
		EdmEntitySet entitySet;
		EdmEntityType entityType;
		String keyPredicateString;
		try {
			entitySet = edm.getDefaultEntityContainer().getEntitySet(entitySetName);
			if (entitySet == null)
				throw new ODataException(ODataExceptionType.INVALID_ENTITY_NAME,
						"No entity with name " + entitySetName + " in the OData service", null);
			entityType = edm.getDefaultEntityContainer().getEntitySet(entitySetName).getEntityType();
			keyPredicateString = convertKeyValuesToString(this.keys, entityType);
			entity = ODataConnectivityUtil.addPropertiesToEntity(this.body, entityType);
		} catch (EdmException e) {
			throw new ODataException(ODataExceptionType.METADATA_PARSING_FAILED, "Error while parsing the metadata.",
					e);
		}
		completeUrl = serviceUri + "/" + entitySetName + '(' + keyPredicateString + ')';
		HttpEntityEnclosingRequestBase updateRequest;

		if (um == UpdateMethod.PATCH)
			updateRequest = new HttpPatch(completeUrl);
		else
			updateRequest = new HttpPut(completeUrl);
		// Set CSRF Token Header.
		updateRequest.setHeader(ODataConnectivityUtil.CSRF_HEADER, edmWithCSRF.getCsrfToken());

		// Set Content-Type header
		updateRequest.setHeader(CONTENT_TYPE, contentType);
		updateRequest.setHeader(ACCEPT_HEADER, contentType);
		for (Entry<String, String> header : headers.entrySet()) {
			updateRequest.setHeader(header.getKey(), header.getValue());
		}

		entity.setWriteProperties(EntitySerializerProperties.serviceRoot(URI.create(serviceUri)).build());
		try {
			response = ODataClient.newInstance().createSerializer(contentType).writeEntry(entitySet, entity);
		} catch (EntityProviderException e) {
			throw new ODataException(ODataExceptionType.INPUT_DATA_SERIALIZATION_FAILED,
					"Error during serialization of input payload." + e.getMessage(), e);
		}
		try {
			InputStream inputStream = ODataConnectivityUtil.getObjectStream(response.getEntity());
			String entityString = null;
			if (inputStream != null) {
				entityString = getEntityString(inputStream);
				try {
					inputStream.close();
				} catch (IOException e) {
					logger.error("Error while closing the inputstream", e);
				}
			}
			StringEntity postEntity = new StringEntity(entityString, UTF_8);
			updateRequest.setEntity(postEntity);
			httpResponse = httpClient.execute(updateRequest);
			ODataConnectivityUtil.checkHttpStatus(httpResponse, errorHandler);
		} catch (UnsupportedEncodingException e) {
			throw new ODataException(null, "UnsupportedEncodingException", e);
		} catch (IOException e) {
			throw new ODataException(null, "IOException", e);
		} finally {
			ODataConnectivityUtil.closeQuietly(httpResponse, updateRequest);
		}

		return new ODataUpdateResult(httpResponse);
	}

	private String convertKeyValuesToString(Map<String, Object> keys, EdmEntityType entityType) throws EdmException, ODataException {

		String keyPredicateString = "";
		String convertedValue;
		for (Entry<String, Object> e : keys.entrySet()) {
			if (!keyPredicateString.isEmpty())
				keyPredicateString += ',';
			String key = e.getKey();
			Object value = e.getValue();
			EdmProperty prop = (EdmProperty) entityType.getProperty(key);
			EdmSimpleType type = (EdmSimpleType) (prop.getType());
			try {
				if(STRING.equals(type.getName()) && value != null){
					value = UrlEncoder.encode(value.toString());
				}
				convertedValue = type.toUriLiteral(type.valueToString(value, EdmLiteralKind.DEFAULT, prop.getFacets()));
			} catch (EdmSimpleTypeException e1) {
				throw new ODataException(ODataExceptionType.BAD_KEY_VALUE, e1.getMessage(), e1);
			}
			keyPredicateString += key + '=' + convertedValue;
		}
		return keyPredicateString;
	}

	protected HttpClient getHttpClient(String destinationName) {
		Destination dest = DestinationAccessor.tryGetDestination(destinationName).get();
		return HttpClientAccessor.getHttpClient(dest.asHttp());
	}

	private String getEntityString(InputStream stream) {

		StringBuilder sb = new StringBuilder();
		String line;

		try (BufferedReader br = new BufferedReader(new InputStreamReader(stream, Charset.forName(UTF_8)))) {

			while ((line = br.readLine()) != null) {

				sb.append(line);
				sb.append("\n");
			}

		} catch (IOException e) {

		}

		return sb.toString();
	}

	@Override
	public ODataUpdateResult execute(WithDestinationName withDestinationName) throws ODataException {
		return execute(withDestinationName.getDestinationName());
	}

	@Override
	public ODataUpdateResult execute(String destinationName) throws ODataException {
		return execute(UpdateMethod.PUT, destinationName);
	}

	@Override
	public ODataUpdateResult execute(UpdateMethod updateMethod, WithDestinationName withDestinationName)
			throws ODataException {
		return execute(updateMethod, withDestinationName.getDestinationName());
	}

	@Override
	public ODataUpdateResult execute(UpdateMethod updateMethod, HttpClient providedClient) throws ODataException {
		logger.debug("Update Called with direct URL");
		return handleExecute(updateMethod, null, providedClient);
	}

	String getServiceName() {
    return serviceName;
  }

  String getEntitySetName() {
    return entitySetName;
  }

  Map<String, Object> getBody() {
    return body;
  }

  Map<String, Object> getKeys() {
    return keys;
  }

  Map<String, String> getHeaders() {
    return headers;
  }


	@Override
	public String toString() {
		return "UPDATE " + withSeparator(SEPARATOR_PATH, serviceName, entitySetName) + " with key " + keys.toString();
	}
}

