/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.rt.etag.processor;

import java.util.Arrays;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.olingo.odata2.api.processor.ODataContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ETagProcessor {
	private static Logger log = LoggerFactory.getLogger(ETagProcessor.class);
	public static final String IF_MATCH = "If-Match";
	public static final String IF_NONE_MATCH = "If-None-Match";
	private static final String WEAK_PATTERN = "W/\"(.*)\"";
	private static final String STRONG_PATTERN = "\"(.*)\"";
	private static final String DELIMITER = ",";
	private static final String ASTERISK = "*";
	private static final Pattern strongPattern = Pattern.compile(STRONG_PATTERN);
	private static final Pattern weakPattern = Pattern.compile(WEAK_PATTERN);

	private boolean ifMatchConditionMet;
	private boolean ifNoneMatchConditionMet;
	private boolean asteriskPresentInIfMatch;
	private boolean asteriskPresentInIfNoneMatch;
	private boolean eTagHeaderPresent;
	private List<String> ifMatchHeaderValue;
	private List<String> ifNoneMatchHeaderValue;
	
	public ETagProcessor(ODataContext context){
		if (context != null) {
			if (log.isDebugEnabled()) {log.debug("Initialise ETagProcessor. Request object not null.");};
			ifMatchConditionMet = true;
			ifNoneMatchConditionMet = true;
			eTagHeaderPresent = false;
			asteriskPresentInIfMatch = false;
			asteriskPresentInIfNoneMatch = false;
			this.extractETagHeader(context);
		}
	}
	
	private void extractETagHeader(ODataContext context){
		if (context.getRequestHeader(IF_MATCH)!=null) {
			if (log.isDebugEnabled()) {log.debug("Request contains "+IF_MATCH+" header.");};
			ifMatchHeaderValue = Arrays.asList(context.getRequestHeader(IF_MATCH).split(Pattern.quote(",")));
			asteriskPresentInIfMatch = this.isAsteriskInIfMatchEtag(ifMatchHeaderValue);
			eTagHeaderPresent = true;
		}
		if (context.getRequestHeader(IF_NONE_MATCH)!=null) {
			if (log.isDebugEnabled()) {log.debug("Request contains "+IF_NONE_MATCH+" header.");};
			ifNoneMatchHeaderValue = Arrays.asList(context.getRequestHeader(IF_NONE_MATCH).split(Pattern.quote(",")));
			asteriskPresentInIfNoneMatch = this.isAsteriskInIfNoneMatchEtag(ifNoneMatchHeaderValue);
			eTagHeaderPresent = true;
		}
	}
	
	public boolean ifETagHeaderPresent() {
		return eTagHeaderPresent;
	}
	
	public boolean ifAsteriskPresentInIfMatch() {
		return asteriskPresentInIfMatch;
	}
	
	public boolean ifAsteriskPresentInIfNoneMatch() {
		return asteriskPresentInIfNoneMatch;
	}
	
	public List<String> getIfMatchHeaderValue() {
		return ifMatchHeaderValue;
	}
	
	public List<String> getIfNoneMatchHeaderValue() {
		return ifNoneMatchHeaderValue;
	}
	
	public boolean evaluateETag(String propValue) {
		if (propValue != null) { //Continue with the operation if eTag property value is null
			if (ifMatchHeaderValue != null) {
				ifMatchConditionMet = evaluateIfMatchEtag(propValue);
			}
			if (ifNoneMatchHeaderValue != null) {
				ifNoneMatchConditionMet = evaluateIfNoneMatchEtag(propValue);
			}
		}
		return (ifMatchConditionMet && ifNoneMatchConditionMet);
	}
	
	private boolean evaluateIfMatchEtag(String propValue) {
		if (log.isDebugEnabled()) {log.debug("Evaluating eTag values for "+IF_MATCH+" header.");};
		String eTagValue = null;
		//Perform OR operation to validate if-match ETag values
		for (String headerValue:ifMatchHeaderValue) {
			eTagValue = fetchETagValueFromHeader(headerValue.trim());
			System.out.println(("If Match InDB ::"+propValue+"   , In Header :: "+ eTagValue));
			if (eTagValue != null) {
				if (eTagValue.equals(propValue)||eTagValue.equals(ASTERISK)) {
					return true;
				}
			}
		}
		return false;
	}
	
	private boolean evaluateIfNoneMatchEtag(String propValue) {
		if (log.isDebugEnabled()) {log.debug("Evaluating eTag values for "+IF_NONE_MATCH+" header.");};
		String eTagValue = null;
		//Perform AND operation to validate if-none-match ETag values
		for (String headerValue:ifNoneMatchHeaderValue) {
			eTagValue = fetchETagValueFromHeader(headerValue.trim());
			if (eTagValue != null) {
				if (eTagValue.equals(propValue)||eTagValue.equals(ASTERISK)) {
					return false;
				}
			}
		}
		return true;
	}
	
	private boolean isAsteriskInIfNoneMatchEtag(List<String> headerValues){
		boolean isFound = false;
		String eTagValue = null;		
		for (String headerValue:headerValues) {
			eTagValue = fetchETagValueFromHeader(headerValue.trim());
			if (eTagValue != null && eTagValue.equals(ASTERISK)) {
				if (log.isDebugEnabled()) {log.debug("Asterisk present in if-none-match eTag header "+headerValue+".");};
				isFound = true;
				break;
			}
		}
		return isFound;
	}

	private boolean isAsteriskInIfMatchEtag(List<String> headerValues){
		boolean isFound = false;
		String eTagValue = null;		
		for (String headerValue:headerValues) {
			eTagValue = fetchETagValueFromHeader(headerValue.trim());
			if (eTagValue != null && eTagValue.equals(ASTERISK)) {
				if (log.isDebugEnabled()) {log.debug("Asterisk present in if-match eTag header "+headerValue+".");};
				isFound = true;
				break;
			}
		}
		return isFound;
	}
	
	public String fetchETagValueFromHeader(String headerValue) {
		if (log.isDebugEnabled()) {log.debug("Extracting eTag header value "+headerValue+".");};
		Matcher weakMatcher = weakPattern.matcher(headerValue);
		Matcher strongMatcher = strongPattern.matcher(headerValue);
		if (weakMatcher.matches()) {
			return weakMatcher.group(1);
		} else if (strongMatcher.matches()) {
			return strongMatcher.group(1);
		} else if (headerValue != null && headerValue.equals(ASTERISK)) {
			return headerValue;
		} else {
			return null;
		}
	}
	
	 
	
}
