/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.rt.util;

import static com.sap.cloud.sdk.service.prov.api.internal.SQLMapping.convertToUpperCaseIfRequired;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.olingo.odata2.api.edm.EdmAnnotations;
import org.apache.olingo.odata2.api.edm.EdmCustomizableFeedMappings;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmFacets;
import org.apache.olingo.odata2.api.edm.EdmMapping;
import org.apache.olingo.odata2.api.edm.EdmMultiplicity;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.edm.EdmTyped;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.uri.KeyPredicate;
import org.apache.olingo.odata2.api.uri.NavigationSegment;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.core.uri.KeyPredicateImpl;

import com.sap.cloud.sdk.service.prov.v2.rt.api.DraftFlow;

public class DraftUtilsV2 {

	private DraftUtilsV2() {
	}
	
	public static final String DRAFTS_ADMIN_TABLE_NAME="DraftAdministrativeData";
	public static final String DRAFTS_TABLE_NAME_SUFFIX="_Drafts";
	public static final String DRAFTS_FOREIGN_KEY="DraftAdministrativeData_DraftUUID";
	public static final String DRAFTS_ISACTIVE_ENTITY="IsActiveEntity";
	public static final String DRAFTS_HASACTIVE_ENTITY="HasActiveEntity";
	public static final String DRAFTS_HASDRAFT_ENTITY="HasDraftEntity";
	public static final String DRAFTS_ADMIN_TABLE_KEY="DraftUUID";
	public static final String DRAFTS_SIBLING_ISACTIVEENTITY = "SiblingEntity.IsActiveEntity";
	public static final String DRAFTS_ADMIN_INPROCESSBYUSER = "DraftAdministrativeData.InProcessByUser";
	public static final String DRAFTS_SIBLING_NAVIGATION = "SiblingEntity";
	
	//Query Identifiers
	public static final String DRAFTS_FILTER_ROOT="IsActiveEntity eq true";
	public static final String DRAFTS_FILTER_OWNDRAFT="IsActiveEntity eq false";
	public static final String DRAFTS_FILTER_ALL="IsActiveEntity eq false or SiblingEntity/IsActiveEntity eq null";
	public static final String DRAFTS_FILTER_UNCHANGED="IsActiveEntity eq true and HasDraftEntity eq false";
	public static final String DRAFTS_FILTER_UNSAVED="IsActiveEntity eq true and SiblingEntity/IsActiveEntity eq null and DraftAdministrativeData/InProcessByUser eq ''";
	public static final String DRAFTS_FILTER_LOCKEDBYOTHER="IsActiveEntity eq true and SiblingEntity/IsActiveEntity eq null and DraftAdministrativeData/InProcessByUser ne ''";
	public static final String DRAFTS_FILTER_BOTH="IsActiveEntity eq true and substringof";
	
	public static boolean isUriFromMainTable(UriInfo uriInfo) throws EdmException {
		List<KeyPredicate> keypredicates = uriInfo.getKeyPredicates();
		for(Iterator<KeyPredicate> iter=keypredicates.iterator(); iter.hasNext();) {
			KeyPredicate param = iter.next();
			if(param.getProperty() != null && param.getProperty().getName().equalsIgnoreCase(DRAFTS_ISACTIVE_ENTITY)){
				return "true".equalsIgnoreCase(param.getLiteral());
			}
		}
		return false;
	}
	
	public static List<String> extractKeyPredicateFilterExpressions(UriInfo uriInfo, boolean considerActiveEntityKey) throws EdmException {
		List<KeyPredicate> keypredicates = extractKeyPredicateRefsFromUriInfo(uriInfo, considerActiveEntityKey);
		EdmEntityType entity = uriInfo.getStartEntitySet().getEntityType(); 
		List<String> filters=new ArrayList<>();
		for (int j = 0; j < keypredicates.size(); j++) {
			String filter;
			String kpAlias = keypredicates.get(j).getProperty().getName();
			EdmTyped kpRef = entity.getProperty(kpAlias);
			if(kpRef.getName().equals(DRAFTS_ISACTIVE_ENTITY)) {
				continue;
			}
			String kpNameInFilter = convertToUpperCaseIfRequired(kpRef.getName());
			String keyPredType = kpRef.getType().toString();
			if (keyPredType != null && (keyPredType.equals("Edm.DateTimeOffset")
					|| keyPredType.equals("Edm.DateTime")
					|| keyPredType.equals("Edm.Date")
					|| keyPredType.equals("Edm.Guid")
					)) {
				filter=kpNameInFilter.replace("/", ".") + " = '"+ keypredicates.get(j).getLiteral() + "'";
			} else {
				filter=kpNameInFilter.replace("/", ".") + " = " + keypredicates.get(j).getLiteral();
			}
			filters.add(filter);
		}
		return filters;
	}
	
	public static List<KeyPredicate> extractKeyPredicateRefsFromUriInfo(UriInfo uriInfo, boolean considerActiveEntityKey) throws EdmException {
		List<KeyPredicate> keyPredicates = uriInfo.getKeyPredicates();
		List<KeyPredicate> keyparams = new ArrayList<>();
		if(!considerActiveEntityKey) {
			for(Iterator<KeyPredicate> iter=keyPredicates.iterator();iter.hasNext();) {
				KeyPredicate param = iter.next();
				if(!DRAFTS_ISACTIVE_ENTITY.equalsIgnoreCase(param.getProperty().getName())){
					keyparams.add(param);
				}
			}
		}else {
			keyparams.addAll(keyPredicates);
		}
		return keyparams;
	}	
	
	public static String getUUIDkeyPropertyName(EdmEntitySet entSet) throws EdmException {
		String refKeyName = null;
		EdmEntityType entType = entSet.getEntityType();
		for(EdmProperty refs:entType.getKeyProperties()) {
			if(refs.getType().getName().equalsIgnoreCase("Guid")) {
				refKeyName = refs.getName();
				break;
			}
		}
		return refKeyName;
	}
	
	public static List<KeyPredicate> extractKeyPredicateRefsEntity(Map<String, Object> ent, EdmEntityType entityType) throws ODataException {
		List<KeyPredicate> keyparams=new ArrayList<>();
		for(String key:entityType.getKeyPropertyNames()) {
			if(!key.equalsIgnoreCase(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY)) {
				KeyPredicateImpl keyParam = new KeyPredicateImpl(ent.get(key).toString(), new EdmProperty() {
					@Override
					public EdmAnnotations getAnnotations() throws EdmException {
						return null;
					}
					
					@Override
					public String getName() throws EdmException {
						return entityType.getProperty(key).getName();
					}
					
					@Override
					public EdmType getType() throws EdmException {
						return entityType.getProperty(key).getType();
					}
					
					@Override
					public EdmMultiplicity getMultiplicity() throws EdmException {
						return entityType.getProperty(key).getMultiplicity();
					}
					
					@Override
					public EdmMapping getMapping() throws EdmException {
						return null;
					}
					
					@Override
					public EdmFacets getFacets() throws EdmException {
						return null;
					}
					
					@Override
					public boolean isSimple() {
						return true;
					}
					
					@Override
					public String getMimeType() throws EdmException {
						return null;
					}
					
					@Override
					public EdmCustomizableFeedMappings getCustomizableFeedMappings() throws EdmException {
						return null;
					}
				});
				keyparams.add(keyParam);
			}
		}
		return keyparams;
	}
	
	public static DraftFlow determinedraftQueryType(UriInfo uriInfo) {
		List<KeyPredicate> keys = uriInfo.getKeyPredicates();
		if (uriInfo.getFilter() != null 
				&& uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_OWNDRAFT) 
				&& uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_ROOT) 
				&& !keys.isEmpty()
				&& !checkIfExists(uriInfo.getNavigationSegments())) {
			for (KeyPredicate key : keys) {
				try {
					if (key.getProperty().getName().equalsIgnoreCase(DRAFTS_ISACTIVE_ENTITY) && key.getLiteral().equalsIgnoreCase("true")) {
						return DraftFlow.QUERY_ROOT;
					} else if (key.getProperty().getName().equalsIgnoreCase(DRAFTS_ISACTIVE_ENTITY) && key.getLiteral().equalsIgnoreCase("false")) {
						return DraftFlow.QUERY_NOTSUPPORTED;
					}
				} catch (EdmException e) {
					return DraftFlow.QUERY_NOTSUPPORTED;
				}
			}
		} else if(uriInfo.getFilter() != null) {
			if(uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_UNSAVED)) {
				return DraftFlow.QUERY_UNSAVED;
			}else if(uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_LOCKEDBYOTHER)) {
				return DraftFlow.QUERY_LOCKEDBYOTHER;
			}else if(uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_UNCHANGED)) {
				return DraftFlow.QUERY_UNCHANGED;
			}else if(uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_ALL)) {
				return DraftFlow.QUERY_ALL;
			}else if (uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_BOTH)){
				return DraftFlow.QUERY_ROOT;
			}else if(uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_OWNDRAFT)){
				return DraftFlow.QUERY_OWNDRAFTS;
			}else if(uriInfo.getFilter().getExpressionString().contains(DRAFTS_FILTER_ROOT)){
				return DraftFlow.QUERY_ROOT;
			}else {
				return DraftFlow.QUERY_NOTSUPPORTED;
			}
		}
		return DraftFlow.QUERY_NOTSUPPORTED;
	}
	
	private static boolean checkIfExists(List<NavigationSegment> navigationSegments) {
		List<String> navProperties = new ArrayList<>();
		navProperties.add("SiblingEntity");
		navProperties.add("DraftAdministrativeData");
		for (NavigationSegment navSeg : navigationSegments) {
			try {
				if (navProperties.contains(navSeg.getNavigationProperty().getName())) {
					return true;
				}
			} catch (EdmException e) {
				return true;
			}
		}
		return false;
	}
	
	public static boolean isValidPropertyForMainTable(String columnName) {
		boolean isValid = true;
		if(columnName.equalsIgnoreCase(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY)||
			columnName.equalsIgnoreCase(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY)||
			columnName.equalsIgnoreCase(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY)||
			columnName.equalsIgnoreCase(DraftUtilsV2.DRAFTS_FOREIGN_KEY)) {
			isValid = false;
		}
		return isValid;
	}
	
}
