package com.sap.cloud.sdk.service.prov.rt.hana.sql;


import static com.sap.cloud.sdk.service.prov.api.internal.SQLMapping.convertToUpperCaseIfRequired;
import static com.sap.cloud.sdk.service.prov.api.internal.SQLMapping.quoteIfRequired;

import java.util.List;

import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmLiteral;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeKind;
import org.apache.olingo.odata2.api.edm.EdmTyped;
import org.apache.olingo.odata2.api.uri.expression.BinaryExpression;
import org.apache.olingo.odata2.api.uri.expression.BinaryOperator;
import org.apache.olingo.odata2.api.uri.expression.ExpressionVisitor;
import org.apache.olingo.odata2.api.uri.expression.FilterExpression;
import org.apache.olingo.odata2.api.uri.expression.LiteralExpression;
import org.apache.olingo.odata2.api.uri.expression.MemberExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodOperator;
import org.apache.olingo.odata2.api.uri.expression.OrderByExpression;
import org.apache.olingo.odata2.api.uri.expression.OrderExpression;
import org.apache.olingo.odata2.api.uri.expression.PropertyExpression;
import org.apache.olingo.odata2.api.uri.expression.SortOrder;
import org.apache.olingo.odata2.api.uri.expression.UnaryExpression;
import org.apache.olingo.odata2.api.uri.expression.UnaryOperator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.hana.sql.DateUtils;
import com.sap.cloud.sdk.service.prov.rt.cds.CDSQueryGenerator;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Filter;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Parameter;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.CDSExpressionVisitor;
import com.sap.cloud.sdk.service.prov.v2.rt.util.DraftUtilsV2;

public class HanaExpressionVisitor implements ExpressionVisitor {

	public static final String ERROR_MSG = "UNSUPPORTED_FILTER";
	// Following constant will be used to replace '/' in the URL which represents a
	// Math division
	public static final String ARITHMETIC_DIVISION = "ADIV123ZZ321DIVA";
	private EdmEntitySet targetEntitySet;
	private static final Logger logger = LoggerFactory.getLogger(CDSExpressionVisitor.class);
	private Filter filter;
	private String propName;
	private boolean isDraft = false;

	public HanaExpressionVisitor(EdmEntitySet targetEntitySet, boolean isDraft) {
		/*
		 * Target entity set is used in some cases when we need the namespace and entity
		 * name of a property in order to be able to query the CSN for the persistence
		 * name in the hdbcds mode
		 */
		this.filter = new Filter();
		this.targetEntitySet = targetEntitySet;
		this.isDraft = isDraft;
	}

	@Override
	public Object visitFilterExpression(FilterExpression filterExpression, String expressionString, Object expression) {
		this.filter.setExpression(expression.toString());
		return filter;
	}

	@Override
	public Object visitBinary(BinaryExpression binaryExpression, BinaryOperator operator, Object leftSide,
			Object rightSide) {

		String leftString;
		// Ensure not to replace " / " which is a result of the arithmetic operation
		// division
		StringBuffer buffer;
		String rightString;
		if (rightSide instanceof EdmLiteral) {
			rightString = ((EdmLiteral) rightSide).getLiteral();
		} else {
			rightString = rightSide.toString();
		}

		if (leftSide instanceof EdmLiteral) {
			leftString = ((EdmLiteral) leftSide).getLiteral().toString().replaceAll(" / ", ARITHMETIC_DIVISION);
			leftString = leftString.replaceAll("/", ".");
			leftString = leftString.replaceAll(ARITHMETIC_DIVISION, " / ");
			
			// Trigger only for draft.
			if (this.isDraft) {
				// Ignore construction of where clause for these draft table properties
				if (leftString.trim().contains(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY.toUpperCase())
						|| leftString.trim().contains(DraftUtilsV2.DRAFTS_SIBLING_ISACTIVEENTITY.toUpperCase())
						|| leftString.trim().contains(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY.toUpperCase())
						|| leftString.trim().contains(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY.toUpperCase())
						|| leftString.trim().contains(DraftUtilsV2.DRAFTS_ADMIN_INPROCESSBYUSER.toUpperCase())) {
					return "";
				}

				// If right is blank and left side is an expression return it
				if ("".equals(leftString.trim())) {
					if (rightSide.toString().startsWith("(") && rightSide.toString().endsWith(")")) {
						return rightSide;
					} else if (!rightSide.toString().isEmpty() && !this.filter.getParameters().isEmpty()) {
						this.filter.getParameters().removeLast();
						return "";
					} else {
						return "";
					}
				}

				// If left is blank and right side is an expression return it
				if ("".equals(rightSide.toString().trim())) {
					if (leftString.startsWith("(") && leftString.endsWith(")")) {
						return leftString;
					} else if (leftString.endsWith(")")) {
						return leftString;
					} else {
						return "";
					}
				}
			}
			
			buffer = new StringBuffer("( ?");
		} else {
			leftString = leftSide.toString().replaceAll(" / ", ARITHMETIC_DIVISION);
			leftString = leftString.replaceAll("/", ".");
			leftString = leftString.replaceAll(ARITHMETIC_DIVISION, " / ");
			
			// Trigger only for draft.
			if (this.isDraft) {
				// Ignore construction of where clause for these draft table properties
				if (leftString.trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY.toUpperCase())
						|| leftString.trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_SIBLING_ISACTIVEENTITY.toUpperCase())
						|| leftString.trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY.toUpperCase())
						|| leftString.trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY.toUpperCase())
						|| leftString.trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_ADMIN_INPROCESSBYUSER.toUpperCase())) {
					return "";
				}

				// If right is blank and left side is an expression return it
				if ("".equals(leftString.trim())) {
					if (rightSide.toString().startsWith("(") && rightSide.toString().endsWith(")")) {
						return rightSide;
					} else if (!rightSide.toString().isEmpty() && !this.filter.getParameters().isEmpty()) {
						this.filter.getParameters().removeLast();
						return "";
					} else {
						return "";
					}
					
				}

				// If left is blank and right side is an expression return it
				if ("".equals(rightSide.toString().trim())) {
					if (leftString.startsWith("(") && leftString.endsWith(")")) {
						return leftString;
					} else if (leftString.trim().endsWith(")")) {
						return leftString;
					} else {
						return "";
					}
				}
			}
			
			buffer = new StringBuffer("( " + leftString);
		}

		
		boolean caseMod = false;
		switch (operator) {
		// Arithmetic operation
		case ADD:
			buffer.append(' ').append("+").append(' ');
			break;
		case SUB:
			buffer.append(' ').append("-").append(' ');
			break;
		case MUL:
			buffer.append(' ').append("*").append(' ');
			break;
		case DIV:
			buffer.append(' ').append("/").append(' ');
			break;
		case MODULO:
			caseMod = true;
			break;
		// Logical operations
		case GT:
			buffer.append(' ').append('>').append(' ');
			break;
		case LT:
			buffer.append(' ').append('<').append(' ');
			break;
		case EQ:
			if ("null".equals(rightString)) {
				buffer.append(" is null ");
			} else if (buffer.indexOf("LIKE") < 0) {
				buffer.append(' ').append('=').append(' ');
			}

			if (buffer.indexOf("LOCATE") > 0) {
				String odataIndex = this.filter.getParameters().getLast().getValue();
				Integer hanaIndex = Integer.parseInt(odataIndex) + 1;
				if (!this.filter.getParameters().isEmpty() && this.filter.getParameters().getLast() != null) {
					this.filter.getParameters().getLast().setValue(hanaIndex.toString());
				}
			}
			break;
		case NE:
			if ("null".equals(rightString)) {
				buffer.append(" is not null ");
			} else {
				buffer.append(' ').append("<>").append(' ');
			}
			break;
		case LE:
			buffer.append(' ').append("<=").append(' ');
			break;
		case GE:
			buffer.append(' ').append(">=").append(' ');
			break;
		case AND:
		case OR:
			buffer.append(' ').append(operator.toUriLiteral()).append(' ').append(rightString);
			buffer.append(" )");
			return buffer.toString();
		default:
			buffer.append(' ').append(operator.toUriLiteral()).append(' ');
		}

		if (rightSide instanceof EdmLiteral) {
			if ("null".equals(rightString) && !this.filter.getParameters().isEmpty()) {
				this.filter.getParameters().removeLast();
			} else if (buffer.indexOf("LIKE") < 0)
				buffer.append('?');
			else if (rightString.equals("false")) {
				String newBuffer = buffer.toString().replaceFirst(" LIKE ", " NOT LIKE ");
				buffer = new StringBuffer(newBuffer);
				if(!this.filter.getParameters().isEmpty())
				this.filter.getParameters().removeLast();
			} else if (rightString.equals("true") && !this.filter.getParameters().isEmpty()) {
				this.filter.getParameters().removeLast();
			} else {
				buffer.append('?');
			}
		} else {
			buffer.append(rightString);
		}

		if (caseMod) {
			buffer.delete(1, buffer.length());
			buffer.append("MOD (" + leftSide.toString() + " , " + rightString + ")");
		}

		buffer.append(" )");

		return buffer.toString();
	}

	@Override
	public Object visitLiteral(LiteralExpression literal, EdmLiteral edmLiteral) {

		switch(edmLiteral.getType().toString()) {
		case "Edm.Decimal":
			filter.setParameter(new Parameter(edmLiteral.getLiteral() , "Edm.Decimal" , this.propName));
			break;
		case "Edm.Double":
			filter.setParameter(new Parameter(edmLiteral.getLiteral() , "Edm.Double", this.propName));
			break;
		case "Edm.Single":
			filter.setParameter(new Parameter(edmLiteral.getLiteral() , "Edm.Single", this.propName));
			break;
		case "Edm.Int64":
			filter.setParameter(new Parameter(edmLiteral.getLiteral() , "Edm.Int64" , this.propName));
			break;
		case "Edm.Int32":
		case "Edm.Int16":
		case "System.Uint7":
		case "Edm.Byte":
		case "Edm.SByte":
			filter.setParameter(new Parameter(edmLiteral.getLiteral() , "Edm.Int" , this.propName));
			break;
		case "Edm.Binary":
			filter.setParameter(new Parameter(edmLiteral.getLiteral() , "Edm.Binary" , this.propName));
			break;
		case "Edm.Guid":
			filter.setParameter(new Parameter((edmLiteral.getLiteral()).toLowerCase() , "Edm.Guid" , this.propName));
			break;
		case "Edm.String":
			filter.setParameter(new Parameter(edmLiteral.getLiteral() , "Edm.String" , this.propName));
			break;
		case "Edm.DateTime":
		case "Edm.Time":
			filter.setParameter(new Parameter(DateUtils.dateLiteralresolver("datetime\'" + edmLiteral.getLiteral() + "\'").replaceAll("^\'|\'$", "") , "Edm.Date" , this.propName));
			break;
		case "Edm.DateTimeOffset":
			filter.setParameter(new Parameter(DateUtils.dateLiteralresolver("datetimeoffset\'" + edmLiteral.getLiteral() + "\'").replaceAll("^\'|\'$", "") , "Edm.Date" ,this.propName));
			break;
	    default:
		filter.setParameter(new Parameter(edmLiteral.getLiteral() , "Edm.String" , this.propName));
		}
		
		return edmLiteral;
	}

	@Override
	public Object visitMethod(MethodExpression methodExpression, MethodOperator method, List<Object> parameters) {
		if ("ENDSWITH".equals(method.name())) {
			String value = "?";
			String fExpression = "UPPER(" + parameters.get(0) + ")" + " LIKE " + "UPPER(" + value + ")";
			if(!this.filter.getParameters().isEmpty() && this.filter.getParameters().getLast() != null) {
				this.filter.getParameters().getLast().setValue("%"+this.filter.getParameters().getLast().getValue());
			}
			return fExpression;
		}
		if ("STARTSWITH".equals(method.name())) {
			String value = "?";
			String fExpression = "UPPER(" + parameters.get(0) + ")" + " LIKE " + "UPPER(" + value + ")";
			if(!this.filter.getParameters().isEmpty() && this.filter.getParameters().getLast() != null) {
				this.filter.getParameters().getLast().setValue(this.filter.getParameters().getLast().getValue()+"%");
			}
			return fExpression;
		}
		if ("SUBSTRINGOF".equals(method.name())) {
			/*
			 * In Case of Substring the order of parameters from odata is reversed. This is
			 * the standard
			 * http://www.odata.org/documentation/odata-version-2-0/uri-conventions/
			 */
			String value = "?";
			String fExpression = "UPPER(" + parameters.get(1) + ")" + " LIKE " + "UPPER(" + value + ")";
			if(!this.filter.getParameters().isEmpty() && this.filter.getParameters().getLast() != null) {
				this.filter.getParameters().getLast().setValue("%"+this.filter.getParameters().getLast().getValue()+"%");
			}
			return fExpression;
		}
		if ("TOUPPER".equals(method.name())) {
		  if(parameters.get(0) instanceof EdmLiteral) {
        EdmLiteral edmLiteral = ((EdmLiteral) parameters.get(0));
        removeParameterWithLiteral(edmLiteral);
        Parameter parameter = new Parameter(edmLiteral.getLiteral().toUpperCase(), "Edm.String" , this.propName);
        filter.getParameters().add(parameter);
        return new EdmLiteral(EdmSimpleTypeKind.String.getEdmSimpleTypeInstance(), edmLiteral.getLiteral().toUpperCase());
      }
			return "UCASE(" + parameters.get(0) + ")";
		}
		if ("TOLOWER".equals(method.name())) {
		  if(parameters.get(0) instanceof EdmLiteral) {
		    EdmLiteral edmLiteral = ((EdmLiteral) parameters.get(0));
		    removeParameterWithLiteral(edmLiteral);
		    Parameter parameter = new Parameter(edmLiteral.getLiteral().toLowerCase(), "Edm.String" , this.propName);
		    filter.getParameters().add(parameter);
		    return new EdmLiteral(EdmSimpleTypeKind.String.getEdmSimpleTypeInstance(), edmLiteral.getLiteral().toLowerCase());
      }
			return "LCASE(" + parameters.get(0) + ")";
		}
		if("LENGTH".equals(method.name())){
			String fExpression = "LENGTH("+parameters.get(0)+")";
			return fExpression;
		}
		if("INDEXOF".equals(method.name())){
			String value = "?";
			String fExpression = "LOCATE("+parameters.get(0)+" , "+ value +")";
			return fExpression;
		}
		if("SUBSTRING".equals(method.name())){
			String fExpression;
			String value = "?";
			String odataIndex = ((EdmLiteral) parameters.get(1)).getLiteral();
			Integer hanaIndex =  Integer.parseInt(odataIndex) + 1;
			if(parameters.size() == 2) {
				fExpression = "SUBSTRING("+parameters.get(0)+" , "+ value +")";
				if(!this.filter.getParameters().isEmpty() && this.filter.getParameters().getFirst() != null){
					this.filter.getParameters().getLast().setValue(hanaIndex.toString());
				}
			}else {
				fExpression = "SUBSTRING("+parameters.get(0)+" , "+ value + " ," + value +")";
				if(!this.filter.getParameters().isEmpty() && this.filter.getParameters().getFirst() != null){
					this.filter.getParameters().getFirst().setValue(hanaIndex.toString());
				}
			}
			return fExpression;
		}
		if("TRIM".equals(method.name())) {
			String fExpression = "TRIM("+parameters.get(0)+")";
			return fExpression;
		}
		if("CONCAT".equals(method.name())) {
			String value;
			if(parameters.get(1).toString().contains("ROOT_ENTITY_SELECT_ALIAS.PATH_FROM_ROOT")){
				value = parameters.get(1).toString();
			}
			else{
				value = "?";
			}
			String fExpression = "CONCAT("+parameters.get(0)+" ,"+ value +")";
			return fExpression;
		}
		return "";
	}

  /**
   * @param edmLiteral
   */
  private void removeParameterWithLiteral(EdmLiteral edmLiteral) {
    for (Parameter param : filter.getParameters()) {
      if (param.getValue().equals(edmLiteral.getLiteral()) && param.getName().equals(this.propName)) {
        filter.getParameters().remove(param);
      }
    }
  }

	/**
	 * This method executed for expression tree for any member operator ("/") which
	 * is used to reference a property of an complex type or entity type.
	 */
	@Override
	public Object visitMember(MemberExpression memberExpression, Object path, Object property) {
		// String value = path.toString() + "/" + property.toString();
		if(!path.toString().isEmpty()) {
			String pathString = convertToUpperCaseIfRequired(path.toString());
			String propertyName = convertToUpperCaseIfRequired(property.toString());
			return "ROOT_ENTITY_SELECT_ALIAS" + "." + "PATH_FROM_ROOT."
					+ pathString.replaceAll("ROOT_ENTITY_SELECT_ALIAS.", "").replaceAll("PATH_FROM_ROOT.", "") + "."
					+ propertyName.replaceAll("ROOT_ENTITY_SELECT_ALIAS.", "").replaceAll("PATH_FROM_ROOT.", "");
		} else {
			return "";
		}
		
	}

	@Override
	public Object visitProperty(PropertyExpression propertyExpression, String uriLiteral, EdmTyped edmProperty) {
		// parseToWhereExpression difference 1 in SQLFilterExpressionVisitor
		// this.tableAlias + "." + fieldName; difference 2 in SQLFilterExpressionVisitor
		boolean skipDraftProperties = false;
		if (this.isDraft) {
			try {
				// Ignore construction of where clause for these draft table properties
				if (edmProperty.getName().trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY.toUpperCase())
						|| edmProperty.getName().trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_SIBLING_ISACTIVEENTITY.toUpperCase())
						|| edmProperty.getName().trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY.toUpperCase())
						|| edmProperty.getName().trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_ADMIN_INPROCESSBYUSER.toUpperCase())
						|| edmProperty.getName().trim().toUpperCase().contains(DraftUtilsV2.DRAFTS_SIBLING_NAVIGATION.toUpperCase())) {
					skipDraftProperties = true;
				}
			} catch (EdmException e) {
				logger.error("Error while processing filter", e);
			}
			
		}
		if(!skipDraftProperties) {
			String persistenceName = uriLiteral;
			//Todo: Handle exception
			try {
				this.propName = edmProperty.getName();
				
				if(!this.filter.getParameters().isEmpty() && this.filter.getParameters().getLast() != null && this.filter.getParameters().getLast().getName() == null) {
					this.filter.getParameters().getLast().setName(this.propName);
				}
			} catch (EdmException e) {
				logger.error("Error while processing filter", e);
			}
			if (targetEntitySet != null) {
				try {
					String namespace = targetEntitySet.getEntityType().getNamespace();
					String fqName = namespace + "." + targetEntitySet.getName();
					persistenceName = CDSQueryGenerator.getPersistenceName(fqName, uriLiteral);
				} catch (EdmException e) {
					logger.error("Error while processing filter", e);
				}
			}
			String pathPrefixedName = "ROOT_ENTITY_SELECT_ALIAS" + "." + "PATH_FROM_ROOT."
					+ quoteIfRequired(persistenceName);
			return pathPrefixedName;
		} else {
			return "";
		}
		
	}

	@Override
	public Object visitUnary(UnaryExpression unaryExpression, UnaryOperator operator, Object operand) {
		switch (operator) {
		case MINUS:
			return "-" + operand;
		case NOT:
			return " NOT(" + operand + ") ";
		}
		return null;
	}

	@Override
	public Object visitOrderByExpression(OrderByExpression orderByExpression, String expressionString,
			List<Object> orders) {
		return null;
	}

	@Override
	public Object visitOrder(OrderExpression orderExpression, Object filterResult, SortOrder sortOrder) {
		return null;
	}

}
