/**
 * This class extends BaseDataProvider .Execute OData V2 Query & Returns  IDataProviderResponse
 *
 * 
 * @version 1.0
 * @since   2016-10-27 
 */
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.commons.InlineCount;
import org.apache.olingo.odata2.api.edm.EdmAnnotationAttribute;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmNavigationProperty;
import org.apache.olingo.odata2.api.ep.EntityProviderReadProperties;
import org.apache.olingo.odata2.api.ep.EntityProviderWriteProperties.ODataEntityProviderPropertiesBuilder;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.exception.ODataApplicationException;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.processor.ODataContext;
import org.apache.olingo.odata2.api.processor.ODataRequest;
import org.apache.olingo.odata2.api.uri.KeyPredicate;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.info.DeleteUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntitySetCountUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntitySetUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntityUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetFunctionImportUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetMediaResourceUriInfo;
import org.apache.olingo.odata2.api.uri.info.PostUriInfo;
import org.apache.olingo.odata2.api.uri.info.PutMergePatchUriInfo;
import org.apache.olingo.odata2.core.edm.EdmDateTime;
import org.apache.olingo.odata2.core.edm.EdmDateTimeOffset;
import org.apache.olingo.odata2.core.edm.EdmDecimal;
import org.apache.olingo.odata2.core.edm.EdmDouble;
import org.apache.olingo.odata2.core.edm.EdmGuid;
import org.apache.olingo.odata2.core.edm.EdmInt64;
import org.apache.olingo.odata2.core.edm.EdmSingle;
import org.apache.olingo.odata2.core.edm.EdmString;
import org.apache.olingo.odata2.core.uri.UriInfoImpl;
import org.apache.olingo.odata2.core.uri.UriParserImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.annotations.VisibleForTesting;
import com.sap.cloud.sdk.service.prov.api.ExtensionHelper;
import com.sap.cloud.sdk.service.prov.api.connection.DataSourceParams;
import com.sap.cloud.sdk.service.prov.api.connection.ThreadSafeObjectStore;
import com.sap.cloud.sdk.service.prov.api.internal.AdminDataAnnotation;
import com.sap.cloud.sdk.service.prov.api.internal.CSNNotSetException;
import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;
import com.sap.cloud.sdk.service.prov.api.locale.impl.LocaleTypeResolver;
import com.sap.cloud.sdk.service.prov.api.statistics.SAPStatistics;
import com.sap.cloud.sdk.service.prov.api.transaction.impl.TransactionHandler;
import com.sap.cloud.sdk.service.prov.rt.cds.CDSHandler;
import com.sap.cloud.sdk.service.prov.rt.cds.ExtensionHelperImpl;
import com.sap.cloud.sdk.service.prov.rt.cds.HANADMLExecutor;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.CreateEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DeleteEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.EntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.UpdateEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.wrapper.CDSConnectionWrapper;
import com.sap.cloud.sdk.service.prov.rt.cds.wrapper.CDSTransactionInitializer;
import com.sap.cloud.sdk.service.prov.v2.rt.api.internal.ODataOperationValidator;
import com.sap.cloud.sdk.service.prov.v2.rt.api.internal.TransactionParams;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException.MessageKeys;
import com.sap.cloud.sdk.service.prov.v2.rt.csn.AcceptsCSN;
import com.sap.cloud.sdk.service.prov.v2.rt.data.provider.DataProvider;
import com.sap.cloud.sdk.service.prov.v2.rt.etag.processor.ETagProcessor;
import com.sap.cloud.sdk.service.prov.v2.rt.util.LocaleUtil;
import com.sap.gateway.core.api.provider.data.BaseDataProviderResponse;
import com.sap.gateway.core.api.provider.data.IDataProviderResponse;
import com.sap.gateway.core.api.srvrepo.IServiceInfo;

public class CDSDataProvider extends DataProvider implements AcceptsCSN, ODataOperationValidator {

	final static Logger logger = LoggerFactory.getLogger(CDSDataProvider.class);
	Connection conn = null;
	JsonNode csn;
	ODataToCDSProcessor cdsprocessor;
	private Long pageTop;
	private Long pageSkip;
	private Long serverPageSize;
	private Long currSkipTokenValue = (long) 0;
	private Long topValOriginal = (long) 0;
	private CDSDataSourceParam cdsDSTparam;
	boolean isBatchRequest = false;
	private SAPStatistics timings = new SAPStatistics();
	boolean showSapStat = false;
	private boolean isDraftFlow = false;
	private static final String NO_ENTITY_FOUND = "No entity found";
	// persisting isBatchFlow info, so that in case if DS become null, can be used to instantiate with persisted value.
	private boolean isBatchFlow = false;
	private int inlineCount; //For calculating inline count draft flow
	private static final String DRAFT_ROOT = "isDraftRoot";
	/**
	 * Initialize the ODataToCDSProcessor and the TransactionManager,decides whether transaction manager will be local transaction manager
	 * or provided transaction manager.
	 * @param service
	 */
	public CDSDataProvider(IServiceInfo service) {
		super(service);
		this.cdsprocessor = new ODataToCDSProcessor();
		CDSTransactionInitializer.initializeTransactionManager();
	}
	
	/*
	 * This method specifies what an EDM & CSN Type should be mapped to java
	 * type. This Data Provider has the CSN information and also the edmx
	 * information from the URI Info. This makes the data provider to find out
	 * the corresponding java type mapping.
	 * 
	 * Currently deep insert scenarios are not handled. It should be written
	 * with the information from the navigation property
	 */
	@SuppressWarnings("rawtypes")
	@Override
	public Map<String, Object> getTypeMapping(UriInfo uriInfo, EntityProviderReadProperties readProperties,
			EdmNavigationProperty navigationProperty) throws ODataException {
		HashMap<String, Object> propertiesTypeMapping = new HashMap<String, Object>();
		List<String> propertNames = uriInfo.getTargetEntitySet().getEntityType().getPropertyNames();
		for (String propertyName : propertNames) {
			Class mappedType = CDSHANATypeMapping.getMappedType(null,
					uriInfo.getTargetEntitySet().getEntityType().getProperty(propertyName).getType().toString());
			if (mappedType != null)
				propertiesTypeMapping.put(propertyName, mappedType);
		}
		return propertiesTypeMapping;
	}
	
	

	/**
	 * ReadEntity Flow .
	 */
	@Override
	public IDataProviderResponse readEntity(GetEntityUriInfo uriInfo, ODataContext context, boolean cascadeDelete) throws ODataException {
		Connection connection=null;
		timings = new SAPStatistics();	
		try {
			connection=getConnection(context,uriInfo.getCustomQueryOptions());
			BaseDataProviderResponse response = new BaseDataProviderResponse();
			/******check if the entity is annotated only with @autoexposed all direct requests are rejected ****/
		
			if (uriInfo.getNavigationSegments().size()<1 && CSNUtil.isAutoExposed(uriInfo.getTargetEntitySet().getName(),uriInfo.getTargetEntitySet().getEntityType().getNamespace()) && !cascadeDelete ){
			      autoExposecheck();
			}
			List<Map<String, Object>> ec=cdsprocessor.handleAndExecute((UriInfo) uriInfo, context, csn, connection, timings);
			if (ec != null&&!ec.isEmpty()) {
				response.setResultEntity(ec.get(0));
			}else {
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, null);
			}
			setStatisticsInContext(context, response, timings);
			return response;
		}catch(SQLException e){
			if(e.getErrorCode() != 404)
				logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(), LocaleUtil.getLocaleforException(),HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		}finally {
			closeConnection();
		}
		
	}

	/**
	 * ReadEntityCollection Flow
	 */
	@Override
	public IDataProviderResponse readEntitySet(GetEntitySetUriInfo uriInfo, ODataContext context)
			throws ODataException {
		Connection connection=null;
		timings = new SAPStatistics();
		String rawUri = context.getPathInfo().getRequestUri().toString();
		/******check if the entity is annotated only with @autoexposed all direct requests are rejected ****/
		
		if ( uriInfo.getNavigationSegments().size()<1 &&  CSNUtil.isAutoExposed(uriInfo.getTargetEntitySet().getName(),uriInfo.getTargetEntitySet().getEntityType().getNamespace())){
			
			 autoExposecheck();
		}
		try {
			connection=getConnection(context,uriInfo.getCustomQueryOptions());
			BaseDataProviderResponse response = new BaseDataProviderResponse();
			setPageParameters((UriInfo) uriInfo);
			List<Map<String, Object>> ec=cdsprocessor.handleAndExecute((UriInfo) uriInfo, context, csn, connection, timings);
			response.setResultEntities(ec);
			if (getPageSize(uriInfo) != null && !ec.isEmpty() && ec.size() == serverPageSize) {
				setUriParams(uriInfo, response, rawUri);
			}
			cleanPageParameters();
			if (uriInfo.getInlineCount() != null && uriInfo.getInlineCount().name().equals(InlineCount.ALLPAGES.toString())) {
				response.setInlineCount(getInlineCount(uriInfo, context, csn,connection,false));
			}
			setStatisticsInContext(context, response, timings);
			return response;
		}catch(SQLException e){
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(), LocaleUtil.getLocaleforException(),HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		}finally {
			closeConnection();
		}
	}	
	
	@Override
	public IDataProviderResponse readEntityMedia(GetMediaResourceUriInfo uriInfo, ODataContext context)
			throws ODataException {
		timings = new SAPStatistics();
		Connection connection = null;
		String entitySetName = uriInfo.getStartEntitySet().getEntityType().getName();
		Map<String, String> mediaDetails = MediaUtils.getMediaAnnotationFromCSN(
				uriInfo.getStartEntitySet().getEntityType().getNamespace(), entitySetName, csn);
		String mediaColumn = mediaDetails.get(MediaUtils.mediaColumn);
		String mimeTypeColumn = mediaDetails.get(MediaUtils.mimeTypeColumn);
		String mimeType = mediaDetails.get(MediaUtils.mimeType);
		Map<String, Object> resultEntity = null;
		try {
			BaseDataProviderResponse response = new BaseDataProviderResponse();
			connection = getConnection(context,uriInfo.getCustomQueryOptions());
			List<Map<String, Object>> ec = cdsprocessor.handleAndExecute((UriInfo) uriInfo, context, csn, connection,
					timings, mediaColumn);
			if (ec != null && !ec.isEmpty()) {
				resultEntity = ec.get(0);
				if (resultEntity.get(mediaColumn) != null) {
					MediaUtils.populateMediaDetails(resultEntity, response, mediaColumn, mimeTypeColumn, mimeType);
					return response;
				} else {
					// throw exception if media is null/ does not exist
					throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.READ_NO_MEDIA_FOUND,
							"Unable to read the media because the media does not exist.",
							LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, null);
				}
			} else {
				// throw exception if entity does not exist
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.READMEDIA_NO_ENTITY_FOUND,
						"Unable to read the media because the entity does not exist.",
						LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, null);
			}
		} catch (SQLException e) {
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),
					LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		} finally {
			closeConnection();
		}
	}
	
	/*
	 * $Count Flow
	 */
	
	
	@Override
	public IDataProviderResponse countEntitySet(GetEntitySetCountUriInfo uriInfo, ODataContext context)
			throws ODataException {
		Connection connection=null;
		timings = new SAPStatistics();	
		try {
			connection=getConnection(context,uriInfo.getCustomQueryOptions());
			BaseDataProviderResponse response = new BaseDataProviderResponse();
			Integer count=cdsprocessor.handleAndExecute((UriInfo) uriInfo, context, csn, connection, timings,true,false);
			response.setCount(String.valueOf(count));
			setStatisticsInContext(context, response, timings);
			return response;
		}catch(SQLException e){
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),LocaleUtil.getLocaleforException(),HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		}finally {
			closeConnection();
		}
	}
	
	private int getInlineCount(GetEntitySetUriInfo uriInfo, ODataContext context, JsonNode csn,Connection connection,boolean mainTable)throws ODataException, SQLException {
		Integer count=cdsprocessor.handleAndExecute((UriInfo) uriInfo, context, csn, connection, timings,true,mainTable);
		return count;
	}
	
	private void initializeConnection() {
		this.conn = new CDSConnectionWrapper(ThreadSafeObjectStore.getConnection());
	}

	public CDSHandler getHandler(ODataRequest request, String namespace) {
		try {
			if(conn==null || conn.isClosed())
				initializeConnection();
			if (conn != null) {
				setLocale(conn, request);
				setUserContextInfo(conn);
			}
		} catch (SQLException e) {
			logger.error("Error checking status of connection - closed or open?",e);
		}
		return new CDSHandler(conn, namespace);
	}
	
	@Override
	public ExtensionHelper getExtensionHelper(ODataRequest request, String namespace) {
		CDSHandler handler = getHandler(request, namespace);
		return new ExtensionHelperImpl(handler, TransactionHandler::isActive, handler::getConnection);
	}

	@Override
	public DataSourceParams getDataSource() {
		if (cdsDSTparam == null) {
			cdsDSTparam = new CDSDataSourceParam(new CDSConnectionWrapper(ThreadSafeObjectStore.getConnection()));
			cdsDSTparam.setIsBatchFlow(this.isBatchFlow);
		}
		return cdsDSTparam;
	}

	@Override
	public final DataSourceParams getDSParams() {
		if (cdsDSTparam == null) {
			cdsDSTparam = new CDSDataSourceParam(new CDSConnectionWrapper(ThreadSafeObjectStore.getConnection()));
			cdsDSTparam.setIsBatchFlow(this.isBatchFlow);
		}
		return cdsDSTparam;
	}

	@Override
	public final TransactionParams getTSParams() {
		if (cdsDSTparam == null) {
			cdsDSTparam = new CDSDataSourceParam(new CDSConnectionWrapper(ThreadSafeObjectStore.getConnection()));
			cdsDSTparam.setIsBatchFlow(this.isBatchFlow);
		}
		return cdsDSTparam;
	}


	
	/*
	 * DML Logic for create update and delete
	 * 
	 * 
	 */
	
	

	@Override
	public IDataProviderResponse createEntity(PostUriInfo uriInfo, ODataEntry content, String requestContentType,ODataEntityProviderPropertiesBuilder providerPropertiesBuilder, ODataContext context)throws ODataException {
		timings = new SAPStatistics();
		BaseDataProviderResponse response = new BaseDataProviderResponse();
		if (!isCreatable(uriInfo.getTargetEntitySet().getEntityType().getNamespace(), uriInfo.getTargetEntitySet().getName())) {
			throw new ODataApplicationException("Method not allowed on entity set.",LocaleUtil.getLocaleforException(),
					HttpStatusCodes.METHOD_NOT_ALLOWED);
		}
		
	/******check if the entity is annotated only with @autoexposed all direct requests are rejected ****/
		
		if (uriInfo.getNavigationSegments().size()<1 && CSNUtil.isAutoExposed(uriInfo.getTargetEntitySet().getName(),uriInfo.getTargetEntitySet().getEntityType().getNamespace())){
			
			 autoExposecheck();
		}
		
		/******check if the entity is annotated with @autoexposed and @autoexpose all direct write are rejected ****/
		
		if (uriInfo.getNavigationSegments().size()<1 && CSNUtil.isAutoExposedAndAutoExpose(uriInfo.getTargetEntitySet().getName(),uriInfo.getTargetEntitySet().getEntityType().getNamespace())){
			
			 autoExposecheck();
		}
		
		Connection connection=null;
		// Get Connection and setLocal
		try {
			connection=getConnection(context,uriInfo.getCustomQueryOptions());
			Map<String, AdminDataAnnotation> adminDataMapForCreate = CSNUtil.getAdminDataMapForCreate(uriInfo.getTargetEntitySet().getEntityType().getNamespace(),uriInfo.getTargetEntitySet().getName(),(connection != null && connection.getClientInfo(APPLICATIONUSER) != null));
			CreateEntityInfo ceInfo = cdsprocessor.transormToCreateEntityInfo(uriInfo, content, adminDataMapForCreate);
			Map<String, Object> returnMap = cdsprocessor.executeInsert(uriInfo, ceInfo, connection, timings);
			if (returnMap != null && returnMap.get(HANADMLExecutor.ROWS_AFFECTED) != null&& (Integer) returnMap.get(HANADMLExecutor.ROWS_AFFECTED) > 0) {
				//Get data from main draft table
				if(this.isDraftFlow()){
					response.setResultEntity(content.getProperties());
					return response;
				}else{
					List<Map<String, Object>> ec=cdsprocessor.handleAndExecute((UriInfo) uriInfo, context,content, csn, connection, timings);
					if (ec != null&&!ec.isEmpty()) {
						response.setResultEntity(ec.get(0));
					}
				}
			}
			setStatisticsInContext(context, response, timings);
			return response;
		} catch (ODataApplicationException e) {
			throw e;
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.QUERY_FAILED, e.getMessage(),LocaleUtil.getLocaleforException(),
					HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		} finally {
			closeConnection();
		}

	}	
	
	@Override
	public IDataProviderResponse createEntityMedia(PostUriInfo uriInfo, InputStream content, String requestContentType,
			String slugType, ODataContext context) throws ODataException {		
		timings = new SAPStatistics();
		BaseDataProviderResponse response = new BaseDataProviderResponse();
		// Media is creatable only if the parent entity is creatable
		if (!isCreatable(uriInfo.getTargetEntitySet().getEntityType().getNamespace(),
				uriInfo.getTargetEntitySet().getName())) {
			throw new ODataApplicationException("Method not allowed on entity set.", LocaleUtil.getLocaleforException(),
					HttpStatusCodes.FORBIDDEN);
		}
		Connection connection = null;
		// Create media should be done as an update call to DB and hence we have to form
		// the key predicates list from the slug header
		UriInfoImpl info = (UriInfoImpl) uriInfo;
		// Forming the entityLink as expected for using the method
		// getKeyPredicatesFromEntityLink to get the list of key predicates
		String entityLink = info.getTargetEntitySet().getName() + "(" + slugType + ")";
		List<KeyPredicate> keyPredicates = UriParserImpl.getKeyPredicatesFromEntityLink(info.getTargetEntitySet(),
				entityLink, null);
		info.setKeyPredicates(keyPredicates);		
		// Get Connection and setLocal
		try {
			connection = getConnection(context,uriInfo.getCustomQueryOptions());
			// We are creating AdminData for Update since it is an update call to DB
			Map<String, AdminDataAnnotation> adminDataMapForUpdate = CSNUtil.getAdminDataMapForUpdate(
					uriInfo.getTargetEntitySet().getEntityType().getNamespace(), uriInfo.getTargetEntitySet().getName(),
					(connection != null && connection.getClientInfo(APPLICATIONUSER) != null));
			String entitySetName = uriInfo.getStartEntitySet().getEntityType().getName();
			Map<String, String> mediaDetails = MediaUtils.getMediaAnnotationFromCSN(
					uriInfo.getStartEntitySet().getEntityType().getNamespace(), entitySetName, csn);
			String mediaColumn = mediaDetails.get(MediaUtils.mediaColumn);
			String mimeTypeColumn = mediaDetails.get(MediaUtils.mimeTypeColumn);
			String mimeType = mediaDetails.get(MediaUtils.mimeType);
			Map<String, Object> properties = new HashMap<>();
			properties.put(mediaColumn, content);
			if (mimeTypeColumn != null)
				properties.put(mimeTypeColumn, requestContentType);
			else if (mimeType != null && !(mimeType.equals(requestContentType))) {
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.CREATE_MIMETYPE_MISMATCH,
						"Unable to create the media because the media type specified in the header does not match with the type specified in the model.",
						Locale.ENGLISH, HttpStatusCodes.BAD_REQUEST);
			}
			Map<String, Object> resultEntity = null;			
			CreateEntityInfo cInfo = cdsprocessor.transormToCreateEntityMediaInfo(uriInfo,properties,adminDataMapForUpdate);			
			cdsprocessor.executeInsert(uriInfo, cInfo, connection, timings);
			List<Map<String, Object>> ec = cdsprocessor.handleAndExecute((UriInfo) uriInfo, context, csn, connection, timings, mediaColumn);
			if (ec != null && !ec.isEmpty()) {
				resultEntity = ec.get(0);
				// Update the media details to the response
				if (resultEntity.get(mediaColumn) != null) {
					MediaUtils.populateMediaDetails(resultEntity, response, mediaColumn, mimeTypeColumn, mimeType);
				}
			}
		} catch (ODataApplicationException e) {
			throw e;
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.QUERY_FAILED, e.getMessage(),
					LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		} finally {
			closeConnection();
		}
		return response;
	}
 
	@Override
	public boolean checkEtagStatusPass(UriInfo uriInfo, ODataContext context, BaseDataProviderResponse response)
			throws ODataException {
		String processedEtagValue = null;
		String eTagPropName = HANADMLHelperV2.getEtagPropertyName(uriInfo.getTargetEntitySet().getEntityType());
		Boolean eTagStatusPass = true;
		// Code to Get Etag Value and lock Dsource
		if (eTagPropName != null) {
			eTagStatusPass = false;
			Object eTagPropReadValue = checkEtagValidations(response, (UriInfo) uriInfo, context);
			if (eTagPropReadValue != null) {
				processedEtagValue = HANADMLHelperV2
						.processEtagValue((UriInfo) uriInfo, eTagPropName, eTagPropReadValue).toString();
				eTagStatusPass = compareEtagValues(context, processedEtagValue);
			}
		}
		return eTagStatusPass;

	}

	@Override
	public void populateUpdatedEntityinResponse(UriInfo uriInfo, ODataContext context,BaseDataProviderResponse response, ODataEntry content) throws ODataException {
		try {
			String eTagPropName = HANADMLHelperV2.getEtagPropertyName(uriInfo.getTargetEntitySet().getEntityType());
			if ((null != context.getRequestHeader(ETagProcessor.IF_MATCH)|| null != context.getRequestHeader(ETagProcessor.IF_NONE_MATCH)) && eTagPropName != null) {
				List<Map<String, Object>> ec=cdsprocessor.handleAndExecute((UriInfo) uriInfo, context,content, csn, getConnection(context, uriInfo.getCustomQueryOptions()), timings);
				if (ec != null&&!ec.isEmpty()) {
					response.setResultEntity(ec.get(0));
				}
			}
		} catch (SQLException e) {
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.QUERY_FAILED, e.getMessage(),LocaleUtil.getLocaleforException(),HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		}
	}

	@Override
	public void populateEtagInHeaderFromResponse(UriInfo uriInfo, BaseDataProviderResponse response)
			throws ODataException {
		String eTagPropName = HANADMLHelperV2.getEtagPropertyName(uriInfo.getTargetEntitySet().getEntityType());
		if (eTagPropName != null && response.getResultEntity() != null) {
			String processedEtagValue = HANADMLHelperV2
					.processEtagValue((UriInfo) uriInfo, eTagPropName, response.getResultEntity().get(eTagPropName))
					.toString();
			populateResponseHeaderEtag(response, processedEtagValue, eTagPropName, null);
		}
	}

	@Override
	public IDataProviderResponse updateEntity(PutMergePatchUriInfo uriInfo, ODataEntry content,
			String requestContentType, boolean merge, ODataContext context) throws ODataException {
		timings = new SAPStatistics();
		BaseDataProviderResponse response = null;
		Connection connection=null;
		
/******check if the entity is annotated only with @autoexposed all direct requests are rejected ****/
		
		if (uriInfo.getNavigationSegments().size()<1 && CSNUtil.isAutoExposed(uriInfo.getTargetEntitySet().getName(),uriInfo.getTargetEntitySet().getEntityType().getNamespace())){
			 autoExposecheck();			
		}
		
		/******check if the entity is annotated with @autoexposed and @autoexpose all direct write are rejected ****/
		
			if (uriInfo.getNavigationSegments().size()<1 && CSNUtil.isAutoExposedAndAutoExpose(uriInfo.getTargetEntitySet().getName(),uriInfo.getTargetEntitySet().getEntityType().getNamespace())){
				 autoExposecheck();
			}
		try {
			if (!isUpdatable(uriInfo.getTargetEntitySet().getEntityType().getNamespace(), uriInfo.getTargetEntitySet().getName())) {
				throw new ODataApplicationException("Method not allowed on entity set.",LocaleUtil.getLocaleforException(),HttpStatusCodes.METHOD_NOT_ALLOWED);
			}
			response = new BaseDataProviderResponse();
			// Get Connection and setLocal
			connection=getConnection(context,uriInfo.getCustomQueryOptions());
			Map<String, AdminDataAnnotation> adminDataMapForUpdate = CSNUtil.getAdminDataMapForUpdate(uriInfo.getTargetEntitySet().getEntityType().getNamespace(),uriInfo.getTargetEntitySet().getName(),(connection != null && connection.getClientInfo(APPLICATIONUSER) != null));
			UpdateEntityInfo uEinfo = cdsprocessor.transormToUpdateEntityInfo(uriInfo, content, merge,adminDataMapForUpdate);
			cdsprocessor.executeUpdate(uriInfo, uEinfo, connection, timings);
			//Draft needs result entries back.
			if(this.isDraftFlow()){
				response.setResultEntity(content.getProperties());
			}
		}catch (SQLException|IOException e) {
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),
					Locale.ENGLISH, HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		} finally {
			closeConnection();
		}
		setStatisticsInContext(context, response, timings);
		return response;
	}
	
	@Override
	public IDataProviderResponse updateEntityMedia(PutMergePatchUriInfo uriInfo, InputStream content,
			String requestContentType, ODataContext context) throws ODataException {

		timings = new SAPStatistics();
		BaseDataProviderResponse response = null;
		Connection connection = null;
		try {
			// Media is updatable only if the parent entity is updatable
			if (!isUpdatable(uriInfo.getTargetEntitySet().getEntityType().getNamespace(),
					uriInfo.getTargetEntitySet().getName())) {
				throw new ODataApplicationException("Method not allowed on entity set.",
						LocaleUtil.getLocaleforException(), HttpStatusCodes.FORBIDDEN);
			}
			response = new BaseDataProviderResponse();
			// Get Connection and setLocal
			connection = getConnection(context,uriInfo.getCustomQueryOptions());
			Map<String, AdminDataAnnotation> adminDataMapForUpdate = CSNUtil.getAdminDataMapForUpdate(
					uriInfo.getTargetEntitySet().getEntityType().getNamespace(), uriInfo.getTargetEntitySet().getName(),
					(connection != null && connection.getClientInfo(APPLICATIONUSER) != null));
			String entitySetName = uriInfo.getStartEntitySet().getEntityType().getName();
			Map<String, String> mediaDetails = MediaUtils.getMediaAnnotationFromCSN(
					uriInfo.getStartEntitySet().getEntityType().getNamespace(), entitySetName, csn);
			String mediaColumn = mediaDetails.get(MediaUtils.mediaColumn);
			String mimeTypeColumn = mediaDetails.get(MediaUtils.mimeTypeColumn);
			String mimeType = mediaDetails.get(MediaUtils.mimeType);
			Map<String, Object> properties = new HashMap<>();
			properties.put(mediaColumn, content);
			if (mimeTypeColumn != null)
				properties.put(mimeTypeColumn, requestContentType);
			else if (mimeType != null && !(mimeType.equals(requestContentType))) {
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.UPDATE_MIMETYPE_MISMATCH,
						"Unable to update the media because the media type specified in the header does not match with the type specified in the model.",
						Locale.ENGLISH, HttpStatusCodes.BAD_REQUEST);
			}
			List<Map<String, Object>> ec = cdsprocessor.handleAndExecute((UriInfo) uriInfo, context, csn, connection,
					timings, mediaColumn);
			Map<String, Object> resultEntity = null;
			if (ec != null && !ec.isEmpty()) {
				resultEntity = ec.get(0);
				// throw exception if media is null/ does not exist
				if (resultEntity.get(mediaColumn) == null) {
					throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.UPDATE_NO_MEDIA_FOUND,
							"Unable to update the media because the media does not exist.", LocaleUtil.getLocaleforException(),
							HttpStatusCodes.NOT_FOUND, null);
				}
			} else {
				// throw exception if entity does not exist
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.UPDATEMEDIA_NO_ENTITY_FOUND,
						"Unable to update the media because the entity does not exist.",
						LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, null);
			}
			UpdateEntityInfo uEinfo = cdsprocessor.transormToUpdateEntityMediaInfo(uriInfo, properties,
					adminDataMapForUpdate);
			cdsprocessor.executeUpdate(uriInfo, uEinfo, connection, timings);
		} catch (SQLException | IOException e) {
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),
					LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		} finally {
			closeConnection();
		}
		setStatisticsInContext(context, response, timings);
		return response;
	}

	@Override
	public IDataProviderResponse deleteEntity(DeleteUriInfo uriInfo, ODataContext context,boolean cascadeDelete) throws ODataException {
		timings = new SAPStatistics();
		Connection connection = null;
		
		BaseDataProviderResponse response = null;
		/******check if the entity is annotated only with @autoexposed all direct requests are rejected ****/
		
		if (uriInfo.getNavigationSegments().size()<1 && CSNUtil.isAutoExposed(uriInfo.getTargetEntitySet().getName(),uriInfo.getTargetEntitySet().getEntityType().getNamespace()) && !cascadeDelete){
			 autoExposecheck();
		}
		
		/******check if the entity is annotated with @autoexposed and @autoexpose all direct write are rejected ****/
		
		if (uriInfo.getNavigationSegments().size()<1 && CSNUtil.isAutoExposedAndAutoExpose(uriInfo.getTargetEntitySet().getName(),uriInfo.getTargetEntitySet().getEntityType().getNamespace())&& !cascadeDelete){
			 autoExposecheck();
		}
		try {
			
			if (!isDeletable(uriInfo.getTargetEntitySet().getEntityType().getNamespace(), uriInfo.getTargetEntitySet().getName())) {
				throw new ODataApplicationException("Method not allowed on entity set.",LocaleUtil.getLocaleforException(),HttpStatusCodes.METHOD_NOT_ALLOWED);
			}
			response = new BaseDataProviderResponse();
			DeleteEntityInfo dEinfo = cdsprocessor.transormToDeleteEntityInfo(uriInfo);
			try {
				// Get Connection and setLocale
				connection = getConnection(context,uriInfo.getCustomQueryOptions());
				cdsprocessor.executeDelete(uriInfo, dEinfo, connection, timings);
			} catch (ODataApplicationException e) {
				logger.error("DELETE FAILED", e);
				throw e;
			}catch (SQLException|IOException e) {
				logger.error(e.getMessage(), e);
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),
						Locale.ENGLISH, HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
			} 
		} finally {
			closeConnection();
		}
		setStatisticsInContext(context, response, timings);
		return response;
	}
	
	@Override
	public IDataProviderResponse deleteEntityMedia(DeleteUriInfo uriInfo, ODataContext context) throws ODataException {
		timings = new SAPStatistics();
		BaseDataProviderResponse response = null;
		Connection connection = null;
		try {
			// Media is deletable only if the parent entity is deletable
			if (!isDeletable(uriInfo.getTargetEntitySet().getEntityType().getNamespace(),
					uriInfo.getTargetEntitySet().getName())) {
				throw new ODataApplicationException("Method not allowed on entity set.",
						LocaleUtil.getLocaleforException(), HttpStatusCodes.FORBIDDEN);
			}
			response = new BaseDataProviderResponse();
			connection = getConnection(context,uriInfo.getCustomQueryOptions());
			// delete media should be done as an update call to DB
			PutMergePatchUriInfo putUriInfo = (PutMergePatchUriInfo) uriInfo;
			// we are creating AdminData for Update since it is an update call to DB
			Map<String, AdminDataAnnotation> adminDataMapForUpdate = CSNUtil.getAdminDataMapForUpdate(
					uriInfo.getTargetEntitySet().getEntityType().getNamespace(), uriInfo.getTargetEntitySet().getName(),
					(connection != null && connection.getClientInfo(APPLICATIONUSER) != null));
			String entitySetName = uriInfo.getStartEntitySet().getEntityType().getName();
			Map<String, String> mediaDetails = MediaUtils.getMediaAnnotationFromCSN(
					uriInfo.getStartEntitySet().getEntityType().getNamespace(), entitySetName, csn);
			String mediaColumn = mediaDetails.get(MediaUtils.mediaColumn);
			String mimeTypeColumn = mediaDetails.get(MediaUtils.mimeTypeColumn);
			Map<String, Object> properties = new HashMap<>();
			properties.put(mediaColumn, null);
			// if mime type is stored in a DB column
			if (mimeTypeColumn != null)
				properties.put(mimeTypeColumn, null);
			List<Map<String, Object>> ec = cdsprocessor.handleAndExecute((UriInfo) putUriInfo, context, csn, connection,
					timings, mediaColumn);
			Map<String, Object> resultEntity = null;
			if (ec != null && !ec.isEmpty()) {
				resultEntity = ec.get(0);
				// throw exception if media is null/ does not exist
				if (resultEntity.get(mediaColumn) == null) {
					throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.DELETE_NO_MEDIA_FOUND,
							"Unable to delete the media because the media does not exist.",
							LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, null);
				}
			}
			// throw exception if entity does not exist
			else {
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.DELETEMEDIA_NO_ENTITY_FOUND,
						"Unable to delete the media because the entity does not exist.",
						LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, null);
			}
			UpdateEntityInfo uEinfo = cdsprocessor.transormToUpdateEntityMediaInfo(putUriInfo, properties,
					adminDataMapForUpdate);
			cdsprocessor.executeUpdate(putUriInfo, uEinfo, connection, timings);
		} catch (ODataApplicationException e) {
			logger.error("DELETE FAILED", e);
			throw e;
		} catch (SQLException | IOException e) {
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),
					LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		} finally {
			closeConnection();
		}
		setStatisticsInContext(context, response, timings);
		return response;
	}

	@Override
	public void setCSN(JsonNode csn) {
		this.csn = csn;
	}

	
	@Override
	public Object checkEtagValidations(BaseDataProviderResponse response, UriInfo uriInfo, ODataContext context)throws ODataApplicationException {
		Object eTagPropReadValue = null;
		Connection connection=null;
		if (null != context.getRequestHeader(ETagProcessor.IF_MATCH)|| null != context.getRequestHeader(ETagProcessor.IF_NONE_MATCH)) {
			try {
				connection=getConnection(context,uriInfo.getCustomQueryOptions());
				if (connection.getAutoCommit()) {
					connection.setAutoCommit(false);
				}
				EntityInfo updateLockEinfo = cdsprocessor.transormToEntityInfoForLock((UriInfo) uriInfo);
				eTagPropReadValue = cdsprocessor.executeUpdateForLock((UriInfo) uriInfo, updateLockEinfo, connection);
			}catch(ODataApplicationException e){
				throw e;
			}catch (SQLException | ODataException e) {
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),Locale.ENGLISH, HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
			}
		}
		return eTagPropReadValue;
	}
	
	
	@Override
	public void startTransaction() throws ODataApplicationException {
		getTSParams().setIsBatchFlow(true);
	}

	@Override
	public void commitTransaction() throws ODataApplicationException {
		try {
			if (null != conn && conn.isClosed()) {
				conn = null;
				cdsDSTparam = null;
			}
		} catch (SQLException e) {
			throw new ODataApplicationException("Commit transaction failed", LocaleUtil.getLocaleforException(), e);
		}
	}

	@Override
	public void rollbackTransaction() throws ODataApplicationException {
		try {
			if (null != conn && conn.isClosed()) {
				conn = null;
				cdsDSTparam = null;
			}
		} catch (SQLException e) {
			throw new ODataApplicationException("RollBack failed ", LocaleUtil.getLocaleforException(), e);
		}

	}
	

	private void setLocale(Connection conn, ODataContext context,Map<String, String> customQueryOptions)throws SQLClientInfoException {
		/*
		 * We found that setting both LOCALE and LOCALE_SAP variables leads to
		 * uncertain behavior sometimes. Hence setting only the LOCALE and not
		 * LOCALE_SAP.
		 */
        if(customQueryOptions!=null){
        	for (Map.Entry<String,String> entry : customQueryOptions.entrySet()) {
        		if(entry.getKey().compareTo("sap-language")==0) {
        			Locale locale=LocaleTypeResolver.getLocale(entry.getValue());
        			String acceptedLocale = LocaleTypeResolver.getLanguage(locale.toString());
				if (conn != null && acceptedLocale.length() > 1) {
					conn.setClientInfo("LOCALE", acceptedLocale);
				}
				return;
			}
		}
               }
		String acceptedLocale = LocaleTypeResolver.getLanguage(context.getAcceptableLanguages().get(0).toString());
		if (conn != null && acceptedLocale.length() > 1) {
			conn.setClientInfo("LOCALE", acceptedLocale);
		}
	}

	private void setLocale(Connection conn, ODataRequest request) throws SQLClientInfoException{
		/*
		 * We found that setting both LOCALE and LOCALE_SAP variables leads to
		 * uncertain behavior sometimes. Hence setting only the LOCALE and not
		 * LOCALE_SAP.
		 */
		if (request != null) {
			Map<String,String>queryParameters=request.getQueryParameters();
			for (Map.Entry<String,String> entry : queryParameters.entrySet()) {
				if(entry.getKey().compareTo("sap-language")==0) {
        			Locale locale=LocaleTypeResolver.getLocale(entry.getValue());
        			String acceptedLocale = LocaleTypeResolver.getLanguage(locale.toString());
					if (conn != null && acceptedLocale.length() > 1) {
						conn.setClientInfo("LOCALE", acceptedLocale);
					}
					return;
				}
			}
			
			String acceptedLocale = LocaleTypeResolver.getLanguage(request.getAcceptableLanguages().get(0).toString());
			if (conn != null && acceptedLocale.length() > 1) {
				try {
					conn.setClientInfo("LOCALE", acceptedLocale);
				} catch (SQLClientInfoException e) {
					logger.error(e.getMessage(),e);
				}
			}
		}
	}
	
	@VisibleForTesting
	Connection getConnection(ODataContext context, Map<String, String> customQueryOptions) throws ODataApplicationException, SQLClientInfoException{

		try {
			if(conn==null || conn.isClosed())
				initializeConnection();
		} catch (SQLException e) {
			logger.error("Error checking status of connection - closed or open?",e);
			throw new ODataApplicationException("No connection to database", LocaleUtil.getLocaleforException(),
					HttpStatusCodes.INTERNAL_SERVER_ERROR);
		}
		if (conn == null)
			throw new CDSRuntimeException(MessageKeys.NO_CONNECTION, "No Connection to database",Locale.ENGLISH, HttpStatusCodes.INTERNAL_SERVER_ERROR);		
		setLocale(conn, context, customQueryOptions);
		setUserContextInfo(conn);
		return this.conn;
	}
	
	 @VisibleForTesting
	 void closeConnection() throws CDSRuntimeException{
		try {
			if(conn!=null) {
				conn.close();
				if(!cdsDSTparam.isBatchFlow()) {
					conn = null;
					cdsDSTparam = null;
				}
			}
		} catch (SQLException e) {
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),LocaleUtil.getLocaleforException(),HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		}
	}
	 
	 
		@Override
		public void cleanupTransaction() {
			
		}
	 
		@Override
		public void setIsBatchFlow(boolean isBatchFlow) {
			this.isBatchFlow = isBatchFlow;
			if (cdsDSTparam == null)
				cdsDSTparam = new CDSDataSourceParam(new CDSConnectionWrapper(ThreadSafeObjectStore.getConnection()));
			if (cdsDSTparam != null)
				cdsDSTparam.setIsBatchFlow(isBatchFlow);
		}

	@Override
	public boolean isCreatable(String serviceName, String entityName) {
		try {
			return CSNUtil.isCreatable(serviceName, entityName);
		} catch (CSNNotSetException e) {
			return true;
		}
	}

	@Override
	public boolean isUpdatable(String serviceName, String entityName) {
		try {
			return CSNUtil.isUpdatable(serviceName, entityName);
		} catch (CSNNotSetException e) {
			return true;
		}
	}

	@Override
	public boolean isDeletable(String serviceName, String entityName) {
		try {
			return CSNUtil.isDeletable(serviceName, entityName);
		} catch (CSNNotSetException e) {
			return true;
		}
	}
	
	
	/*
	 * Logic Regarding Pagination
	 */
	
	
	private void  setPageParameters(UriInfo uriInfo) {
		if (uriInfo.getTop() != null) {
			cdsprocessor.setPageTop((long) uriInfo.getTop());
		}
		if (uriInfo.getSkip() != null) {
			cdsprocessor.setPageSkip((long) uriInfo.getSkip());
		}
		if (getPageSize(uriInfo) != null) {
			computePageValues(uriInfo);
			if (serverPageSize != null) {
				modTopSkipforPagination();
			}
		}
	}
	private void  cleanPageParameters() {
		cdsprocessor.setPageTop(null);
		cdsprocessor.setPageSkip(null);
	}
	
	private void modTopSkipforPagination() {
		cdsprocessor.setPageTop(pageTop);
		if (pageSkip != null) {
			/*
			 * This is done in order to consider for $skiptoken option. In case
			 * both $skip & $skiptoken is present, $skiptoken takes precedence.
			 * The initialization of the pageSkip variable happens in the
			 * computePageValues method.
			 */
			cdsprocessor.setPageSkip(pageSkip);

		}
	}

	public Long getPageSize(GetEntitySetUriInfo uriInfo) {
		return super.getPageSize(uriInfo);
	}


	public final void computePageValues(UriInfo uriInfo)throws NumberFormatException {
		// Pick ServerPagesize from overwrittten Method
		serverPageSize = getPageSize(uriInfo);
		// Negotiate between user Page Size and Server Page size
		if (serverPageSize != null && serverPageSize > 0) {
			// Modifying top and SKip
			if (uriInfo.getTop() == null && uriInfo.getSkip() == null) {
				pageTop = serverPageSize;
				if (uriInfo.getSkipToken() != null) {
					currSkipTokenValue = Long.parseLong(uriInfo.getSkipToken());
					pageSkip = currSkipTokenValue;
				}
			} else if (uriInfo.getTop() != null && uriInfo.getSkip() == null) {
				if (uriInfo.getTop() < serverPageSize) {
					pageTop = (long) uriInfo.getTop();
					topValOriginal = (long) uriInfo.getTop();
				} else {
					pageTop = serverPageSize;
				}
				if (uriInfo.getSkipToken() != null) {
					currSkipTokenValue = Long.parseLong(uriInfo.getSkipToken());
					pageSkip = currSkipTokenValue;
				}
			} else if (uriInfo.getTop() == null && uriInfo.getSkip() != null) {
				pageTop = serverPageSize;
				if (uriInfo.getSkipToken() != null) {
					currSkipTokenValue = Long.parseLong(uriInfo.getSkipToken());
					pageSkip = currSkipTokenValue + uriInfo.getSkip();
				}
			} else if (uriInfo.getTop() != null && uriInfo.getSkip() != null) {
				if (uriInfo.getTop() < serverPageSize) {
					pageTop = (long) uriInfo.getTop();
					topValOriginal = (long) uriInfo.getTop();
				} else {
					pageTop = serverPageSize;
				}
				if (uriInfo.getSkipToken() != null) {
					currSkipTokenValue = Long.parseLong(uriInfo.getSkipToken());
					pageSkip = currSkipTokenValue + uriInfo.getSkip();
					pageTop = uriInfo.getTop() - currSkipTokenValue;
				}
			}
		}
	}

	private final void setUriParams(GetEntitySetUriInfo uriInfo, BaseDataProviderResponse response, String rawUri) {
		long nextSkip = currSkipTokenValue + serverPageSize;
		if (topValOriginal == 0 || (topValOriginal > 0 && topValOriginal > nextSkip)) {
			response.setSkipToken(Long.toString(nextSkip));
		}
	}
	
	/**
	 * Checks whether the following conditions to execute a function are met: 
	 *	1. If the function is bound to an entity and if the entity is ETag-enabled:
	 *	 	- If the HTTP headers for the ETag value (<code>if-match</code> or <code>if-none-match</code>) are present and the corresponding conditions are fulfilled
	 *			 then the function can be executed.
	 * 		- On the other hand, if the HTTP header for the ETag value is missing or if the corresponding condition is not fulfilled then an ODataException is thrown.
	 * 
	 * 	2. If the function is not bound, then the function can be executed.
	 * 
	 * @param uriInfo Contains information specific to the function from the request 
	 * @param context Contains general information from the request
	 * @throws ODataException If the HTTP header is missing or if the condition specified by it is not met
	 */
	@Override
	public void executeEtagCheckForFunctionImport(GetFunctionImportUriInfo uriInfo, ODataContext context) throws ODataException {
		BaseDataProviderResponse response = new BaseDataProviderResponse();
		Boolean eTagStatusPass = true;
		//get the action-for attribute of function import
		EdmAnnotationAttribute actionForAnnotationAttribute = uriInfo.getFunctionImport().getAnnotations().getAnnotationAttribute("action-for", "http://www.sap.com/Protocols/SAPData");
		if(actionForAnnotationAttribute != null){
			//from the action-for value get the entity which is bound to function import
			String actionForValue = actionForAnnotationAttribute.getText();
			String[] arrOfActionForValue = actionForValue.split("\\.");
			//form the UriInfo with the required data from function import object.
			UriInfoImpl newUriInfo = new UriInfoImpl();
			EdmEntitySet entitySet = null;
			if (arrOfActionForValue.length > 1) {
				String actionForEntity = arrOfActionForValue[arrOfActionForValue.length - 1];
				entitySet = uriInfo.getFunctionImport().getEntityContainer().getEntitySet(actionForEntity);
			}
			newUriInfo.setTargetEntitySet(entitySet);
			newUriInfo.setStartEntitySet(entitySet);
			newUriInfo.setEntityContainer(uriInfo.getFunctionImport().getEntityContainer());
			String eTagPropName = HANADMLHelperV2.getEtagPropertyName(newUriInfo.getTargetEntitySet().getEntityType());
			//check the bound entity is etag enabled.
			if(eTagPropName != null && entitySet != null){
				if (null == context.getRequestHeader(ETagProcessor.IF_MATCH) && null == context.getRequestHeader(ETagProcessor.IF_NONE_MATCH)) {
					throw new ODataApplicationException("Pre-Condition Required",LocaleUtil.getLocaleforException(),
							HttpStatusCodes.PRECONDITION_REQUIRED);
				}
				String entitykeySet = entitySet.getEntityType().getKeyPropertyNames().toString();
				//get the function import parameter values and the set to uriInfo keypredicate.
				String keyPredicate = getKeyPredicate(uriInfo,entitykeySet);
				String entityLink = newUriInfo.getTargetEntitySet().getName() + "(" + keyPredicate + ")";
				List<KeyPredicate> keyp = UriParserImpl.getKeyPredicatesFromEntityLink(newUriInfo.getTargetEntitySet(), entityLink, null);
				newUriInfo.setKeyPredicates(keyp);
				//checks the etag status and does the permutation and combination of Etag scenarios
				eTagStatusPass= checkEtagStatusPass((UriInfo) newUriInfo, context,
						(BaseDataProviderResponse) response);
				if(!eTagStatusPass){
					throw new ODataApplicationException("Pre-Condition failed",LocaleUtil.getLocaleforException(),
							HttpStatusCodes.PRECONDITION_FAILED);
					
				}
			}
		}
	}
	
	private String getKeyPredicate(GetFunctionImportUriInfo uriInfo,String EntitykeySet){
		String keyPredicate = null ;
		for ( String key : uriInfo.getFunctionImportParameters().keySet() ) {
			if(EntitykeySet.contains(key)){
				String literalValue =uriInfo.getFunctionImportParameters().get(key).getLiteral();
				String modifiedliteralValue = getLiteralValue(uriInfo,key,literalValue);
				
				if(keyPredicate != null){
					keyPredicate = keyPredicate + "," + key + "=" + modifiedliteralValue;
				}else{
					keyPredicate = key + "=" + modifiedliteralValue;
				}
			}
		}
		return keyPredicate;
	}
	
	/**
	 * literal Value is modified according to the datatype.
	 * @param uriInfo
	 * @param key
	 * @param literalValue
	 * @return
	 */
	private String getLiteralValue(GetFunctionImportUriInfo uriInfo,String key,String literalValue){
		if(uriInfo.getFunctionImportParameters().get(key).getType() instanceof EdmGuid){
			literalValue = "guid'" + literalValue + "'";
		}
		if(uriInfo.getFunctionImportParameters().get(key).getType() instanceof EdmString){
			literalValue = "'" + literalValue + "'";
		}
		if(uriInfo.getFunctionImportParameters().get(key).getType() instanceof EdmDecimal){
			literalValue = literalValue + "M";
		}
		if(uriInfo.getFunctionImportParameters().get(key).getType() instanceof EdmDouble){
			literalValue = literalValue + "D";
		}
		if(uriInfo.getFunctionImportParameters().get(key).getType() instanceof EdmSingle){
			literalValue = literalValue + "f";
		}
		if(uriInfo.getFunctionImportParameters().get(key).getType() instanceof EdmInt64){
			literalValue = literalValue + "L";
		}
		if(uriInfo.getFunctionImportParameters().get(key).getType() instanceof EdmDateTime){
			literalValue = "datetime'" + literalValue + "'";
		}
		if(uriInfo.getFunctionImportParameters().get(key).getType() instanceof EdmDateTimeOffset){
			literalValue = "datetimeoffset'" + literalValue + "'";
		}
		
		return literalValue;
		
	}

	@Override
	public Boolean isDraftFlow() {
		return isDraftFlow;
	}

	@Override
	public void setIsDraftFlow(boolean draftFlow) {
		this.isDraftFlow=draftFlow;
		
	}
	
	@Override
	public Map<String, Object> readDataFromMainTable(UriInfo uriInfo, ODataContext context, List<String> filters) throws ODataApplicationException{
		Map<String, Object> activeEntity = null;
		try{
			Connection localConn = getConnection(context,uriInfo.getCustomQueryOptions());			
			ResultSet result = cdsprocessor.fetchResultSetFromMainTable(uriInfo, context, filters, localConn, timings, false);
			List<Map<String, Object>> ec = cdsprocessor.processResultSetForMainTable(context, result);

			if (ec.isEmpty()){
				if(logger.isErrorEnabled()){
					StringBuilder msg = new StringBuilder();
					msg.append(NO_ENTITY_FOUND);
					logger.error(msg.toString());
				}
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, 
						LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND);
			}else {
				activeEntity = ec.get(0);
			}
		}catch(Exception oe){
			logger.error("No entity found for CDS query: ", oe);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, 
					LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, oe);
		}
		return activeEntity;
	}
	
	
	@Override
	public List<Map<String, Object>> readDataCollectionFromMainTable(UriInfo uriInfo, ODataContext context, List<String> filters) throws ODataApplicationException {
		List<Map<String, Object>> activeEntity = null;
		try{
			setPageParameters((UriInfo) uriInfo);
			ResultSet result = generateStatementAndExecute(uriInfo, context, filters, true);
			activeEntity = cdsprocessor.processResultSetForMainTable(context, result);
		}catch(Exception oe){
			logger.error("No entity found for CDS query: ", oe);
			if(oe instanceof CDSRuntimeException) {
				throw oe;
			}else {
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, 
						LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, oe);
			}
		}
		return activeEntity;
	}
	
	public List<Map<String, Object>> readDataCollectionFromMainTableForCombinedEntries(UriInfo uriInfo, ODataContext context, List<String> filters, boolean pagination) throws ODataApplicationException {
		List<Map<String, Object>> activeEntity = null;
		try{
			setPageParameters((UriInfo) uriInfo);
			ResultSet result = generateStatementAndExecute(uriInfo, context, filters, pagination);
			activeEntity = cdsprocessor.processResultSetForMainTable(context, result);
		}catch(Exception oe){
			logger.error("No entity found for CDS query: ", oe);
			if(oe instanceof CDSRuntimeException) {
				throw oe;
			}else {
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, 
						LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, oe);
			}
		}
		return activeEntity;
	}
	//$count support on the main table for draft flow
		@Override
	public IDataProviderResponse countDataCollectionFromMainTable(UriInfo uriInfo, ODataContext context, List<String> filters) throws ODataApplicationException {
		BaseDataProviderResponse response = new BaseDataProviderResponse();;
		try{
			ResultSet result = generateStatementAndExecute(uriInfo, context, filters, true);
			int count = 0;
			if (result!=null && result.next()) {
			    count = result.getInt(1);
			}
			response.setCount(String.valueOf(count));
		}catch(Exception oe){
			logger.error("No entity found for CDS query: ", oe);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, 
					LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, oe);
		}
		return response;
	}

	private ResultSet generateStatementAndExecute(UriInfo uriInfo, ODataContext context, List<String> filters, boolean pagination) throws ODataApplicationException {
		try{
			Connection localConn = getConnection(context,uriInfo.getCustomQueryOptions());	
			return cdsprocessor.fetchResultSetFromMainTable(uriInfo, context, filters, localConn, timings, pagination);
		}catch(Exception oe){
			logger.error("No entity found for CDS query: ", oe);
			if(oe instanceof CDSRuntimeException) {
				throw (CDSRuntimeException)oe;
			}else {
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, 
						LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, oe);
			}
		}
	}
	//$inlineCount support for draft flow where inline count = total count of main table + draft table
	@Override
	public int getInlineCount(UriInfo uriInfo, ODataContext context, boolean mainTable)throws ODataApplicationException{
		try {
			if (uriInfo.getInlineCount() != null
					&& uriInfo.getInlineCount().name().equals(InlineCount.ALLPAGES.toString())) {
				if (inlineCount == 0) {
					context.setParameter(DRAFT_ROOT, "true");
					this.inlineCount = getInlineCount(uriInfo, context, csn, getConnection(context, uriInfo.getCustomQueryOptions()),mainTable);
				}
			}
		}catch(Exception oe){
			logger.error("No entity found for CDS query: ", oe);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, 
					LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, oe);
		}
		return inlineCount;
	}
	
	private void autoExposecheck()throws ODataApplicationException{
		
		CDSRuntimeException cdsRuntimeException = new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR,
				"INSERT FAILED FOR AUTOEXPOSED ENTITIES",  LocaleUtil.getLocaleforException(),HttpStatusCodes.INTERNAL_SERVER_ERROR);
		throw cdsRuntimeException ;
		
	}
}
