/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import java.io.IOException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmMultiplicity;
import org.apache.olingo.odata2.api.edm.EdmNavigationProperty;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.exception.ODataApplicationException;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.processor.ODataContext;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.info.PostUriInfo;
import org.apache.olingo.odata2.api.uri.info.PutMergePatchUriInfo;
import org.apache.olingo.odata2.core.ep.entry.ODataEntryImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.api.connection.DataSourceParams;
import com.sap.cloud.sdk.service.prov.api.connection.ThreadSafeObjectStore;
import com.sap.cloud.sdk.service.prov.api.internal.AdminDataAnnotation;
import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;
import com.sap.cloud.sdk.service.prov.api.security.AuthorizationService;
import com.sap.cloud.sdk.service.prov.api.statistics.SAPStatistics;
import com.sap.cloud.sdk.service.prov.rt.cds.HANADMLExecutor;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.CreateEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DeleteEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DraftAdminDataInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.UpdateEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.wrapper.CDSConnectionWrapper;
import com.sap.cloud.sdk.service.prov.v2.rt.api.DraftFlow;
import com.sap.cloud.sdk.service.prov.v2.rt.api.DraftProvider;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException;
import com.sap.cloud.sdk.service.prov.v2.rt.core.util.UriInfoDraftHelperV2;
import com.sap.cloud.sdk.service.prov.v2.rt.core.util.UriInfoDraftHelperV2.UriInfoTransformerBuilder;
import com.sap.cloud.sdk.service.prov.v2.rt.util.DraftUtilsV2;
import com.sap.cloud.sdk.service.prov.v2.rt.util.EDMHelper;
import com.sap.cloud.sdk.service.prov.v2.rt.util.HeaderUtil;
import com.sap.cloud.sdk.service.prov.v2.rt.util.LocaleUtil;
import com.sap.gateway.core.api.provider.data.BaseDataProviderResponse;
import com.sap.gateway.core.api.provider.data.IDataProviderResponse;

public class CDSDraftProvider implements DraftProvider {

	private static final Logger logger = LoggerFactory.getLogger(CDSDraftProvider.class);	
	private static ODataToCDSProcessor cdsprocessor = new ODataToCDSProcessor();
	private SAPStatistics timings = new SAPStatistics();
	public static final String  APPLICATIONUSER="APPLICATIONUSER";
	private static final String INSERT_ERROR_MSG = "INSERT FAILED WITH INTERNAL ERROR";
	private static final String NO_ENTITY_FOUND = "No entity found";
	private static final String NO_ENTITY_MSG = "No entity found for CDS query:";
	private int inlineCount = -1;
	
	@Override
	public UUID createEntityForDraftAdminData(PostUriInfo uriInfo, ODataContext context, 
			DataSourceParams dsParams) throws ODataApplicationException {
		LocaleUtil.setLocaleForRequest(context);
		UUID draftAdminUUID = null;
		// write logic for extracting username from header for testing purposes
		String userName= AuthorizationService.getUserName();
		if(userName == null || userName.isEmpty())
		userName = context.getRequestHeader("username");		
		Connection connection = getConnection(dsParams);
		try{
			CreateEntityInfo ceInfo = DraftProcessor.createInfoForDraftAdminData(userName);
			draftAdminUUID = (UUID) ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_TABLE_KEY, ceInfo.getColumns()).getPostData();
			Map<String, Object> returnMap = cdsprocessor.executeInsert(uriInfo, ceInfo, connection, timings);
			if (returnMap != null && returnMap.get(HANADMLExecutor.ROWS_AFFECTED) != null
					&& (Integer) returnMap.get(HANADMLExecutor.ROWS_AFFECTED) > 0) {
				// Create Read from DB by using reusable CreateEntityInfo and UpdateEntityInfo
			} else {
				CDSRuntimeException cdsRuntimeException = new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, INSERT_ERROR_MSG,
						LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR);			
				logger.error(INSERT_ERROR_MSG, cdsRuntimeException);
				throw cdsRuntimeException;
			}
		}catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.QUERY_FAILED, e.getMessage(),LocaleUtil.getLocaleforException(),
					HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		}
		return draftAdminUUID;
	}

	@Override
	public IDataProviderResponse createEntityForDraftsTable(PostUriInfo uriInfo, ODataEntry content,
			ODataContext context, UUID draftAdminUUID, DraftFlow flow, DataSourceParams dsParams) throws ODataApplicationException {
		IDataProviderResponse draftsEntity = null;
		try{
			LocaleUtil.setLocaleForRequest(context);		
			Connection conn = getConnection(dsParams);
			
			//Append DraftAdminUUID as foreign key
			Map<String, Object> payLoad = content.getProperties();
			payLoad.put(DraftUtilsV2.DRAFTS_FOREIGN_KEY, draftAdminUUID);
			//Generate UUID key fields
			//Current assumption is only one key field of type GUID
			EdmEntityType mainEntity = uriInfo.getTargetEntitySet().getEntityType();
			modifyKeysForCreateinDraft(mainEntity, payLoad, flow);
			//Modify Payload As required
			modifyMainPayloadOnFlow(payLoad, flow);
			
			//Call Create for Drafts Table
			Map<String, AdminDataAnnotation> adminDataMapForCreate = CSNUtil.getAdminDataMapForCreate(uriInfo.getTargetEntitySet().getEntityType().getNamespace(),uriInfo.getTargetEntitySet().getName(),(conn != null && conn.getClientInfo(APPLICATIONUSER) != null));
			CreateEntityInfo ceInfo = cdsprocessor.transormToCreateEntityInfo(uriInfo, content, adminDataMapForCreate);
			DraftProcessor.modifyCreateEntityInfoForDraft(ceInfo, content);
			Map<String,Object> returnMap = cdsprocessor.executeInsert(uriInfo, ceInfo, conn, timings);
			if(returnMap!=null && returnMap.get(HANADMLExecutor.ROWS_AFFECTED)!=null  && (Integer)returnMap.get(HANADMLExecutor.ROWS_AFFECTED)>0) {
				//Once Create is successful Extract keys from payload and  use it as filters in Read
				List<String> filters = this.getFilters(mainEntity, payLoad);
				//Create Read from DB by using reusable CreateEntityInfo and UpdateEntityInfo
				draftsEntity = this.readDataFromDraftsTable((UriInfo) uriInfo, context, filters, dsParams);
				
				//Do this only during the normal Create Flow
				if(flow != DraftFlow.DRAFT_EDIT) {
					//Static check to resolve more than 7 arguments cannot be passed.
					//Adding few in to dummy map.
					Map<String, Object> paramMap = new HashMap<>();
					paramMap.put("POST_URI_INFO", uriInfo);
					paramMap.put("CONTEXT", context);
					paramMap.put("DRAFT_FLOW", flow);
					this.handleOneToOneRelationship(paramMap, content, draftAdminUUID, dsParams, draftsEntity, mainEntity);
				}				
			}else {
				CDSRuntimeException cdsRuntimeException = new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, INSERT_ERROR_MSG, 
						LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR);
				logger.error(INSERT_ERROR_MSG, cdsRuntimeException);				
				throw cdsRuntimeException;
			}
		}catch(Exception genExp) {
			CDSRuntimeException cdsRuntimeException = new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, INSERT_ERROR_MSG, 
					LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR, genExp);
			logger.error(INSERT_ERROR_MSG, cdsRuntimeException);				
			throw cdsRuntimeException;
		}
		return draftsEntity;
	}

	private List<String> getFilters(EdmEntityType mainEntity, Map<String, Object> payLoad) throws ODataException {
		List<String> filters = new ArrayList<>();
		List<EdmProperty> keyProperties = mainEntity.getKeyProperties();
		for(EdmProperty edmKey : keyProperties) {
			if(!edmKey.getName().equals(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY) && payLoad.get(edmKey.getName()) != null) {
				filters.add(edmKey.getName().toUpperCase()+" = '"+payLoad.get(edmKey.getName()).toString()+"'");
			}
		}
		return filters;
		
	}
	
	/*
	 * Recursive back to handleOneToOneRelationship based on one to one entity.
	 */
	private void handleOneToOneRelationship(Map<String, Object> paramMap, ODataEntry content, UUID draftAdminUUID, DataSourceParams dsParams, 
			IDataProviderResponse draftsEntity, EdmEntityType mainEntity) throws ODataException {
		PostUriInfo uriInfo = (PostUriInfo)paramMap.get("POST_URI_INFO");
		ODataContext context = (ODataContext)paramMap.get("CONTEXT");
		DraftFlow flow = (DraftFlow)paramMap.get("DRAFT_FLOW");
		
		String serviceName = uriInfo.getEntityContainer().getNamespace();
		
		//now try to recursiuvely call 'createEntityForDraftsTable' method for 1:1 relationShps which are node
		if(mainEntity.getNavigationPropertyNames()!=null && !mainEntity.getNavigationPropertyNames().isEmpty()) {
			boolean isLocalizedForDraft= CSNUtil.isLocalizedForDraft(serviceName.replace("._.", "::"),mainEntity.getName());
			Map<String,List<String>> draftTree = CSNUtil.getDraftTree(serviceName);
			for(String navName:mainEntity.getNavigationPropertyNames()){
				EdmNavigationProperty navProp = (EdmNavigationProperty)mainEntity.getProperty(navName);
				if(!navProp.getMultiplicity().equals(EdmMultiplicity.MANY) && draftTree.get(mainEntity.getName()).contains(navProp.getType().getName()) && !(isLocalizedForDraft && navProp.getType().getName().contains("texts"))) { //there for it is a 1:1 entity and its a Draft Node
					//Create a UriInfoObject and payload using EdmEntityType
					Map<String, Object> navigatedContent = new HashMap<>();
					Object navigatedElement = content.getProperties().get(navProp.getName());
					if(navigatedElement instanceof ODataEntryImpl){
						navigatedContent = ((ODataEntryImpl) navigatedElement).getProperties();
					}
					UriInfoDraftHelperV2 uriInfoTransformer = UriInfoTransformerBuilder.getCreateUriInfoForDraftsOneToOneUsingParent(
							navProp, draftsEntity.getResultEntity(), EDMHelper.getEdmEntitySet((UriInfo)uriInfo, HeaderUtil.getlocale(context)).getEntityContainer(), navigatedContent);
					if (uriInfoTransformer.getUriInfo() != null && uriInfoTransformer.getContent() != null) {
						IDataProviderResponse response = this.createEntityForDraftsTable(
								uriInfoTransformer.getUriInfo(), uriInfoTransformer.getContent(), context,
								draftAdminUUID, flow, dsParams);
						draftsEntity.getResultEntity().put(navProp.getType().getName(), response.getResultEntity());
					}
				}
			}
		}
	}
	
	private void modifyMainPayloadOnFlow(Map<String, Object> payLoad, DraftFlow flow){
		switch(flow) {
			case DRAFT_CREATE_NEW:
				payLoad.put(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY, Boolean.FALSE);
				//Add other inferable boolean fields to DB
				payLoad.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.FALSE);
				payLoad.put(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY, Boolean.FALSE);
				return;
			case DRAFT_EDIT:
				//add extra fields to payload
				payLoad.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.TRUE);
				payLoad.put(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY, Boolean.FALSE);				
				//change isActiveEntity to false
				payLoad.put(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY, Boolean.FALSE);				
				return;
			default:
				return;
		}
	}
	
	private void modifyKeysForCreateinDraft(EdmEntityType mainEntity, Map<String, Object> payLoad, DraftFlow flow) throws ODataException {
		UUID draftsTableKey = UUID.randomUUID();
		List<EdmProperty> keyProperties;
		keyProperties = mainEntity.getKeyProperties();
		for(EdmProperty edmKey : keyProperties) {
			if(!edmKey.getName().equals(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY) && flow!=DraftFlow.DRAFT_EDIT) {
				if(payLoad.get(edmKey.getName())!=null ) {
					payLoad.put(edmKey.getName(), payLoad.get(edmKey.getName()));
				}else {
					payLoad.put(edmKey.getName(), draftsTableKey);
				}
			}
		}
	}
	
	@Override
	public IDataProviderResponse updateEntityForDraftsTable(PutMergePatchUriInfo uriInfo, ODataEntry content,
			String requestContentType, boolean merge, ODataContext context, DataSourceParams dsParams) throws ODataApplicationException {
		IDataProviderResponse response = null;
		Connection connection = getConnection(dsParams);
		try{
			//Recode to inject configurable name space
			UpdateEntityInfo ueInfo = DraftProcessor.transormToUpdateEntityInfoForAutoSaveDrafts(uriInfo, content, merge);
			cdsprocessor.executeUpdate(uriInfo, ueInfo, connection, timings);
			response = this.readDataFromDraftsTable((UriInfo) uriInfo, context, null, dsParams);
			Map<String, Object> draftsEntity = response.getResultEntity();
			//based on the foreign key of Admin Data Update Admin Data as well
			UUID draftAdminUUID = null;
			
			if(draftsEntity.get(DraftUtilsV2.DRAFTS_FOREIGN_KEY) instanceof String) {
				draftAdminUUID = UUID.fromString((String) draftsEntity.get(DraftUtilsV2.DRAFTS_FOREIGN_KEY));
			}else {
				draftAdminUUID = (UUID) draftsEntity.get(DraftUtilsV2.DRAFTS_FOREIGN_KEY);
			}
			
			updateEntityForDraftAdminData((UriInfo) uriInfo, context, draftAdminUUID, connection);
		}catch(Exception e){
			CDSRuntimeException cdsRuntimeException = new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, "UPDATE FAILED WITH INTERNAL ERROR", 
					LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
			logger.error("UPDATE FAILED WITH INTERNAL ERROR", cdsRuntimeException);				
			throw cdsRuntimeException;
		}
		
		return response;
	}

	@Override
	public IDataProviderResponse readDataFromDraftsTable(UriInfo uriInfo, ODataContext context, List<String> filters,
			DataSourceParams dsParams) throws ODataApplicationException {
		BaseDataProviderResponse response = new BaseDataProviderResponse();	
		try{
			Connection conn = getConnection(dsParams);
			List<Map<String, Object>> ec = DraftProcessor.handleAndExecuteFromDraftTable(uriInfo, context, 
			    filters, timings, conn, cdsprocessor);			
			if (ec != null && !ec.isEmpty()) {
				response.setResultEntity(ec.get(0));
			}else{
				throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND,
						NO_ENTITY_FOUND, HeaderUtil.getlocale(context), HttpStatusCodes.NOT_FOUND);
			}
		}catch(Exception e) {
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, e);
		}
		return response;
	}

	@Override
	public void deleteEntry(UriInfo uriInfo, ODataContext context, String entityName,Map<String,Object> keyValue, DataSourceParams dsParams) throws ODataApplicationException {
		final String delete_error = "Error during deletion of draft entity:";
		Connection conn = getConnection(dsParams);
		try{
			DeleteEntityInfo dEinfo = DraftProcessor.transformToDeleteEntityInfo(entityName, keyValue);
			cdsprocessor.executeDelete(uriInfo, dEinfo, conn, timings);
		}catch(Exception e){
			throw getException(delete_error, e);
		}
	}
	
	private final Connection getConnection(DataSourceParams dsParams) {
		return new CDSConnectionWrapper(ThreadSafeObjectStore.getConnection());
	}
	
	private void updateEntityForDraftAdminData(UriInfo uriInfo, ODataContext context, UUID draftAdminUUID, Connection conn) 
			throws ODataApplicationException, SQLException, IOException {
		// write logic for extracting username from header for testing purposes
		String userName= AuthorizationService.getUserName();
		if(userName == null || userName.isEmpty())
		userName = context.getRequestHeader("username");	
		UpdateEntityInfo uInfo = DraftProcessor.updateInfoForDraftAdminData(userName, draftAdminUUID);
		cdsprocessor.executeUpdate(uriInfo, uInfo, conn, timings);
	}
	
	@Override
	public List<Map<String, Object>> readAllDraftEntries(UriInfo uriInfo, ODataContext context, List<String> filters, DataSourceParams dsParams) throws ODataApplicationException {
		List<Map<String, Object>> draftEntities = new ArrayList<>(); 	
		try{
			Connection conn = getConnection(dsParams);
			List<Map<String, Object>> entityCollection = DraftProcessor.handleAndExecuteFromDraftTable(uriInfo, context, 
          filters, timings, conn, cdsprocessor);  
			this.inlineCount = entityCollection.size();
			draftEntities.addAll(this.applyPagination(uriInfo, entityCollection));
		}catch(Exception oe){
			throw getException(NO_ENTITY_MSG, oe);
		}
		return draftEntities;
	}
	
	@Override
	public List<Map<String, Object>> readOwnDrafts(UriInfo uriInfo, ODataContext context, List<Map<String, Object>> activeEntities, 
			DataSourceParams dsParams) throws ODataApplicationException {
		List<Map<String, Object>> entityCollection = new ArrayList<>();
		try{
			String userName = HeaderUtil.getUserName(context);
			List<Map<String, Object>> readAllDraftsWithAdminData = this.readAllDraftsWithAdminData(uriInfo, context, dsParams);
			List<Map<String, Object>> ownDrafts = DraftProcessor.ownDraftInstances(
					EDMHelper.getEdmEntitySet(uriInfo, null).getEntityType(), activeEntities, readAllDraftsWithAdminData, userName);
			this.inlineCount = ownDrafts.size();
			//Check for pagination is required Or not.
			entityCollection.addAll(this.applyPagination(uriInfo, ownDrafts));
		}catch(Exception oe) {
			throw getException(NO_ENTITY_MSG, oe);
		}
		return entityCollection;
	}
	
	@Override
	public List<Map<String, Object>> readUnSavedActive(UriInfo uriInfo, ODataContext context, List<Map<String, Object>> activeEntities, 
			DataSourceParams dsParams) throws ODataApplicationException {
		List<Map<String, Object>> entityCollection = new ArrayList<>();
		try{
			List<Map<String, Object>> readAllDraftsWithAdminData = this.readAllDraftsWithAdminData((UriInfo)uriInfo, context, dsParams);
			List<Map<String, Object>> unSavedActiveEntities = DraftProcessor.unsavedActiveInstances(
					EDMHelper.getEdmEntitySet(uriInfo, null).getEntityType(), activeEntities, readAllDraftsWithAdminData);
			this.inlineCount = unSavedActiveEntities.size();
			entityCollection.addAll(this.applyPagination(uriInfo, unSavedActiveEntities));
		}catch(Exception oe){
			throw getException(NO_ENTITY_MSG, oe);
		}
		return entityCollection;
	}
	
	@Override
	public List<Map<String, Object>> readLockedByOthersEntries(UriInfo uriInfo, ODataContext context, List<Map<String, Object>> activeEntities, 
			DataSourceParams dsParams) throws ODataApplicationException {
		List<Map<String, Object>> entityCollection = new ArrayList<>();
		try{
			String userName = HeaderUtil.getUserName(context);
			List<Map<String, Object>> readAllDraftsWithAdminData = this.readAllDraftsWithAdminData(uriInfo, context, dsParams);
			List<Map<String, Object>> lockedByOthersEntries=DraftProcessor.lockedByOtherActiveInstances(
					EDMHelper.getEdmEntitySet(uriInfo, null).getEntityType(), activeEntities, readAllDraftsWithAdminData,userName);
			this.inlineCount = lockedByOthersEntries.size();
			entityCollection.addAll(this.applyPagination(uriInfo, lockedByOthersEntries));
		}catch(Exception oe){
			throw getException(NO_ENTITY_MSG, oe);
		}
		return entityCollection;
	}
	
	@Override
	public List<Map<String, Object>> readUnchangedActive(UriInfo uriInfo, ODataContext context, List<Map<String, Object>> activeEntities, 
			DataSourceParams dsParams) throws ODataApplicationException {
		List<Map<String, Object>> entityCollection = new ArrayList<>();
		try{
			List<Map<String, Object>> readAllDraftsWithAdminData = this.readAllDraftsWithAdminData(uriInfo, context, dsParams);
			List<Map<String, Object>> unchangedEntities = DraftProcessor.unchangedActiveInstances(
					EDMHelper.getEdmEntitySet(uriInfo, null).getEntityType(), activeEntities, readAllDraftsWithAdminData);
			this.inlineCount = unchangedEntities.size();
			entityCollection.addAll(this.applyPagination(uriInfo, unchangedEntities));
		}catch(Exception oe){
			throw getException(NO_ENTITY_MSG, oe);
		}
		return entityCollection;
	}
	
	@Override
	public List<Map<String, Object>> readCombinedEntries(UriInfo uriInfo, ODataContext context, List<Map<String, Object>> activeEntities, 
			DataSourceParams dsParams) throws ODataApplicationException {
		List<Map<String, Object>> entityCollection = new ArrayList<>();
		try{
			List<Map<String, Object>> readAllDraftsWithAdminData = this.readAllDraftsWithAdminData(uriInfo, context, dsParams);
			List<Map<String, Object>> combinedEntries = DraftProcessor.combinedEntries(
					EDMHelper.getEdmEntitySet(uriInfo, null).getEntityType(), activeEntities, readAllDraftsWithAdminData);
			this.inlineCount = combinedEntries.size();
			entityCollection.addAll(this.applyPagination(uriInfo, combinedEntries));
		}catch(Exception oe){
			throw getException(NO_ENTITY_MSG, oe);
		}
		return entityCollection;
	}
	
	private final List<Map<String, Object>> readAllDraftsWithAdminData(UriInfo uriInfo, ODataContext context, DataSourceParams dsParams) throws ODataApplicationException {
		List<Map<String, Object>> ec = null;
		try{
			Connection conn=getConnection(dsParams);
			ec = DraftProcessor.handleAndExecuteFromDraftTable(uriInfo, context, 
          null, timings, conn, cdsprocessor);
		}catch(Exception oe){
			throw getException(NO_ENTITY_MSG, oe);
		}
		return ec;
	}
	
	/*
	 * Logic will be pushed down to DB layer on Query movement from CDSQL to HANAQL
	 */
	private List<Map<String, Object>> applyPagination(UriInfo uriInfo, List<Map<String, Object>> entityList) {
		return entityList
				.stream()
				.skip(uriInfo.getSkip() != null ? uriInfo.getSkip() : 0)
				.limit(uriInfo.getTop() != null ? uriInfo.getTop() : entityList.size())
				.collect(Collectors.toList());
	}
	
	private CDSRuntimeException getException(String message, Exception e){
		logger.error(message, e);
		return new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, NO_ENTITY_FOUND, LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND, e);
	}
	
	// $count for draft flow
	@Override
	public IDataProviderResponse countAllDraftEntries(UriInfo uriInfo, ODataContext context, List<String> filters,
			DataSourceParams dsParams) throws ODataApplicationException  {
		BaseDataProviderResponse response = new BaseDataProviderResponse();
		
		try{
			Connection conn = getConnection(dsParams);
			ResultSet result = DraftProcessor.fetchResultSetFromDraftTable(uriInfo, context, filters, timings, conn, cdsprocessor);
			response.setCount(String.valueOf(processCountResult(result)));
		}catch(Exception oe){
			throw getException(NO_ENTITY_MSG, oe);
		}
		return response;
	}
	
	// inline count for draft table
	@Override
	public int getInlineCount(UriInfo uriInfo, ODataContext context, DataSourceParams dsParams) throws ODataApplicationException {
		return inlineCount;
	}

	private int processCountResult(ResultSet result) throws SQLException {
		if (result!=null && result.next()) {
			return result.getInt(1);
		}
		return 0;
	}
}
