/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/

/**
* The CDSEXpressionVisitor class it contains algorithm is used to traverse a $filter or $orderby expression tree
* To visit a filter tree ,this class implemented the interface org.apache.olingo.odata2.api.uri.expression.ExpressionVisitor
* 
* @version 1.0
* @since   2016-11-22 
*/
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import static com.sap.cloud.sdk.service.prov.api.internal.SQLMapping.convertToUpperCaseIfRequired;

import java.util.List;

import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmLiteral;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeKind;
import org.apache.olingo.odata2.api.edm.EdmTyped;
import org.apache.olingo.odata2.api.uri.expression.BinaryExpression;
import org.apache.olingo.odata2.api.uri.expression.BinaryOperator;
import org.apache.olingo.odata2.api.uri.expression.ExpressionVisitor;
import org.apache.olingo.odata2.api.uri.expression.FilterExpression;
import org.apache.olingo.odata2.api.uri.expression.LiteralExpression;
import org.apache.olingo.odata2.api.uri.expression.MemberExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodOperator;
import org.apache.olingo.odata2.api.uri.expression.OrderByExpression;
import org.apache.olingo.odata2.api.uri.expression.OrderExpression;
import org.apache.olingo.odata2.api.uri.expression.PropertyExpression;
import org.apache.olingo.odata2.api.uri.expression.SortOrder;
import org.apache.olingo.odata2.api.uri.expression.UnaryExpression;
import org.apache.olingo.odata2.api.uri.expression.UnaryOperator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.hana.sql.DateUtils;
import com.sap.cloud.sdk.service.prov.rt.cds.CDSQueryGenerator;
import com.sap.cloud.sdk.service.prov.v2.rt.util.DraftUtilsV2;

public class CDSExpressionVisitor implements ExpressionVisitor {
	public static final String ERROR_MSG = "UNSUPPORTED_FILTER";
	// Following constant will be used to replace '/' in the URL which represents a Math division  
	public static final String ARITHMETIC_DIVISION = "ADIV123ZZ321DIVA";
	private EdmEntitySet targetEntitySet;
	private static final Logger logger = LoggerFactory.getLogger(CDSExpressionVisitor.class);
	private boolean isDraft = false;
	

	public CDSExpressionVisitor(EdmEntitySet targetEntitySet, boolean isDraft) {
		/*Target entity set is used in some cases when we need the namespace and entity name of a property in order to be 
		able to query the CSN for the persistence name in the hdbcds mode*/
		this.targetEntitySet = targetEntitySet;
		this.isDraft = isDraft;
	}
	
	@Override
	public Object visitFilterExpression(FilterExpression filterExpression, String expressionString, Object expression) {
		return expression;
	}

	@Override
	public Object visitBinary(BinaryExpression binaryExpression, BinaryOperator operator, Object leftSide,
			Object rightSide) {
		
		//Ensure not to replace " / " which is a result of the arithmetic operation division
		
		String leftString =leftSide.toString().replaceAll("/",ARITHMETIC_DIVISION);
		leftString=leftString.replaceAll("/",".");
		leftString=leftString.replaceAll(ARITHMETIC_DIVISION,"/");
		
		//Trigger only for draft.
		if(this.isDraft) {
			//Ignore construction of where clause for these draft table properties 
			if(leftString.trim().contains(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY.toUpperCase()) ||
					leftString.trim().contains(DraftUtilsV2.DRAFTS_SIBLING_ISACTIVEENTITY.toUpperCase())||
					leftString.trim().contains(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY.toUpperCase())||
					leftString.trim().contains(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY.toUpperCase())||
					leftString.trim().contains(DraftUtilsV2.DRAFTS_ADMIN_INPROCESSBYUSER.toUpperCase())) {
				return "";
			}
			
			//If right is blank and left side is an expression return it
			if("".equals(leftString.trim())) {
				if(rightSide.toString().startsWith("(") && rightSide.toString().endsWith(")") || rightSide.toString().startsWith("UPPER")) {
					return rightSide;
				}else{
					return "";
				}
			}
			
			//If left is blank and right side is an expression return it
			if("".equals(rightSide.toString().trim())){
				if(leftString.startsWith("(") && leftString.endsWith(")")) {
					return leftString;
				}else if (leftString.trim().endsWith(")")){
					return leftString;
				}else{
					return "";
				}
			}
		}
		
		StringBuffer buffer = new StringBuffer("( "+leftString);
		
		boolean caseMod = false;
		switch (operator) {
		//Arithmetic operation
		case ADD:
			buffer.append(' ').append("+").append(' ');
			break;
		case SUB:
			buffer.append(' ').append("-").append(' ');
			break;
		case MUL:
			buffer.append(' ').append("*").append(' ');
			break;
		case DIV:
			buffer.append(' ').append("/").append(' ');
			break;
		case MODULO:
			caseMod = true;
			break;
			//Logical operations
		case GT:
			buffer.append(' ').append('>').append(' ');
			break;
		case LT:
			buffer.append(' ').append('<').append(' ');
			break;
		case EQ:
			if (buffer.indexOf("LIKE") < 0) {
				if(rightSide.toString().trim().equalsIgnoreCase("null")) {
					buffer.append(' ').append("is").append(' ');
				}else {
					buffer.append(' ').append('=').append(' ');
				}
			}
			if(buffer.indexOf("LOCATE") > 0) {
				rightSide = Integer.parseInt(rightSide.toString()) + 1;
			}
			break;
		case NE:
			if(rightSide.toString().trim().equalsIgnoreCase("null")) {
				buffer.append(' ').append("is not").append(' ');
			}else {
				buffer.append(' ').append("<>").append(' ');
			}
			break;
		case LE:
			buffer.append(' ').append("<=").append(' ');
			break;
		case GE:
			buffer.append(' ').append(">=").append(' ');
			break;
		default:
			buffer.append(' ').append(operator.toUriLiteral()).append(' ');
		}
		String right = rightSide.toString();

		if ((right.contains("datetimeoffset'")) || (right.contains("datetime'")) || (right.contains("time'")) 
				|| DateUtils.checkIfDurationDataType(right)) {
				buffer.append(DateUtils.dateLiteralresolver(right));
		} else {
			if (buffer.indexOf("LIKE") < 0)
				buffer.append(right);
			else if (right.equals("false")) {
				String newBuffer = buffer.toString().replaceFirst(" LIKE ", " NOT LIKE ");
				buffer = new StringBuffer(newBuffer);
			}else if (right.equals("true")) {
				//DO NOTHING
			}else{
				buffer.append(right);
			}
		}
		if (caseMod) {
			buffer.delete(1, buffer.length());
			buffer.append("MOD (" + leftSide.toString() + " , " + rightSide + ")");
		}
		
		
		buffer.append(" )");

		return buffer.toString();
	}


	@Override
	public Object visitLiteral(LiteralExpression literal, EdmLiteral edmLiteral) {

		String returnValue;
		if (edmLiteral.getType().toString().equals("Edm.Decimal")) {
			return edmLiteral.getLiteral();
		}
		if (edmLiteral.getType().toString().equals("Edm.Int64")) {
			return edmLiteral.getLiteral();
		}
		// we have added the below piece of code to handle GUID in the filter expression. the filter expression for example looks like 
		// http://<host>:<port>/appname/odata/v2/<service name>/entity?$filter=<element> eq guid'acf1dcff-07ac-4c4b-a2cf-267e1735f5ff' so if return the uriliteral value we return something like guid'acf1dcff-07ac-4c4b-a2cf-267e1735f5ff'
		//and the sql query fails to execute. So we have to trim the guid part of it.
		if(edmLiteral.getType().toString().equals("Edm.Guid")){
			/* 
			 * We want to convert uuids in a case insensitive manner, to do that we convert to lower case
			 * 1) the actual value to which comparison has to be done
			 * 2) The value persisted in DB. (By calling LCASE function of HANA.This is done in the visitProperty method)
			 */
			return ("'" + edmLiteral.getLiteral() + "'").toLowerCase();	
		}
		if (edmLiteral.getType().toString().equals("Edm.Double")) {
      return edmLiteral.getLiteral();
    }
		returnValue = literal.getUriLiteral();
		return returnValue;
	}


	@Override
	public Object visitMethod(MethodExpression methodExpression, MethodOperator method, List<Object> parameters) {
		if ("ENDSWITH".equals(method.name())) {
			String value = parameters.get(1).toString().substring(1);
			value = value.substring(0, value.length() - 1);
			String fExpression = "UPPER("+parameters.get(0)+")" + " LIKE " + "UPPER(\'%" + value + "\')";
			return fExpression;
		}
		if ("STARTSWITH".equals(method.name())) {
			String value = parameters.get(1).toString().substring(1);
			value = value.substring(0, value.length() - 1);
			String fExpression = "UPPER("+parameters.get(0)+")" + " LIKE " + "UPPER(\'" + value + "%\')";
			return fExpression;
		}
		if ("SUBSTRINGOF".equals(method.name())) {
			/*In Case of Substring the order of parameters from odata is reversed. This is the standard
			 * http://www.odata.org/documentation/odata-version-2-0/uri-conventions/
			 */
			String value = parameters.get(0).toString().substring(1);
			value = value.substring(0, value.length() - 1);
			String fExpression = "UPPER("+parameters.get(1)+")" + " LIKE " + "UPPER(\'%" + value + "%\')";
			return fExpression;
		}
		if ("TOUPPER".equals(method.name())) {
			return "UCASE("+parameters.get(0)+")";
		}
		if ("TOLOWER".equals(method.name())) {
			return "LCASE("+parameters.get(0)+")";
		}
        if("LENGTH".equals(method.name())){
            String fExpression = "LENGTH("+parameters.get(0)+")";
            return fExpression;
        }
        if("INDEXOF".equals(method.name())){
            String value = parameters.get(1).toString().substring(1);
            value = value.substring(0, value.length() - 1);
            String fExpression = "LOCATE("+parameters.get(0)+" , '"+ value +"')";
            return fExpression;
        }
        if("SUBSTRING".equals(method.name())){
            if(parameters.size() == 2) {
                String value = parameters.get(1).toString();
                int position = Integer.parseInt(value.toString()) + 1;
                String fExpression = "SUBSTRING("+parameters.get(0)+" , "+ position +")";
                return fExpression;
            }else {
                String value = parameters.get(1).toString();
                int position = Integer.parseInt(value.toString()) + 1;
                String length = parameters.get(2).toString();
                String fExpression = "SUBSTRING("+parameters.get(0)+" , "+ position + " ," + length+")";
                return fExpression;
            }
        }
        if("TRIM".equals(method.name())) {
            String fExpression = "TRIM("+parameters.get(0)+")";
            return fExpression;
        }
		if("CONCAT".equals(method.name())) {
			String value = parameters.get(1).toString();
			String fExpression;
			if(value.contains("ROOT_ENTITY_SELECT_ALIAS.PATH_FROM_ROOT")){
				fExpression = "CONCAT("+parameters.get(0)+" , "+ value +")";
			}
			else{
				// This is done to remove the single quotes from the string passed in the URI
				value = value.replaceAll("^\'|\'$", "");
				fExpression = "CONCAT("+parameters.get(0)+" , '"+ value +"')";
			}
			return fExpression;
		}
        // Replace function not supported by Olingo
        /*
        if ("REPLACE".equals(method.name())) {
            String find = parameters.get(1).toString();
            String replace = parameters.get(2).toString();
            String value = parameters.get(3).toString();
            String fExpression = "REPLACE ( UPPER("+parameters.get(0)+", "+ find+", "+replace + ") LIKE " + "UPPER(\'%" + value + "\')";
            return fExpression;
        }*/
		return "";
	}

	/**
	 * This method executed for expression tree for any member operator ("/") which is used to reference a property of an complex type or entity type. 
	 */
	@Override
	public Object visitMember(MemberExpression memberExpression, Object path, Object property) {
		//String value = path.toString() + "/" + property.toString();
		String pathString = convertToUpperCaseIfRequired(path.toString());
		String propertyName = convertToUpperCaseIfRequired(property.toString());
		return "ROOT_ENTITY_SELECT_ALIAS" + "."+"PATH_FROM_ROOT."+ pathString.replaceAll("ROOT_ENTITY_SELECT_ALIAS.", "").replaceAll("PATH_FROM_ROOT.","") + "." + propertyName.replaceAll("ROOT_ENTITY_SELECT_ALIAS.", "").replaceAll("PATH_FROM_ROOT.","") ;
	}

	@Override
	public Object visitProperty(PropertyExpression propertyExpression, String uriLiteral, EdmTyped edmProperty) {
		// parseToWhereExpression difference 1 in SQLFilterExpressionVisitor
		// this.tableAlias + "." + fieldName; difference 2 in SQLFilterExpressionVisitor
		String persistenceName = uriLiteral;
		if(targetEntitySet != null) {
			try {
				String namespace = targetEntitySet.getEntityType().getNamespace();
				String fqName = namespace + "." + targetEntitySet.getName();
				persistenceName = CDSQueryGenerator.getPersistenceName(fqName, uriLiteral);
			} catch (EdmException e) {
				logger.error("Error while processing filter", e);
			}
		}
		String pathPrefixedName = "ROOT_ENTITY_SELECT_ALIAS" + "."+"PATH_FROM_ROOT."+convertToUpperCaseIfRequired(persistenceName);
		return pathPrefixedName;
	}

	@Override
	public Object visitUnary(UnaryExpression unaryExpression, UnaryOperator operator, Object operand) {
		switch (operator) {
		case MINUS:
			return "-" + operand;
		case NOT:
			return " NOT(" + operand + ") ";
		}
		return null;
	}

	@Override
	public Object visitOrderByExpression(OrderByExpression orderByExpression, String expressionString,List<Object> orders) {
		return null;
	}

	@Override
	public Object visitOrder(OrderExpression orderExpression, Object filterResult, SortOrder sortOrder) {
		return null;
	}



}
