/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.UUID;

import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.ep.feed.ODataDeltaFeed;
import org.apache.olingo.odata2.api.exception.ODataApplicationException;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.processor.ODataContext;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.info.PutMergePatchUriInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.api.internal.AdminDataAnnotation;
import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;
import com.sap.cloud.sdk.service.prov.api.statistics.SAPStatistics;
import com.sap.cloud.sdk.service.prov.api.util.RequestProcessingHelper;
import com.sap.cloud.sdk.service.prov.rt.cds.CDSQueryGenerator;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Column;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.CreateEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DBGeneratedData;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DeleteEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DraftAdminDataInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Filter;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.QueryHelper;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.UpdateEntityInfo;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException;
import com.sap.cloud.sdk.service.prov.v2.rt.util.DraftUtilsV2;
import com.sap.cloud.sdk.service.prov.v2.rt.util.HeaderUtil;

public class DraftProcessor {
	
	private DraftProcessor(){
	}
	
	private static final Logger logger = LoggerFactory.getLogger(DraftProcessor.class);
	
	public static CreateEntityInfo createInfoForDraftAdminData(String userName) {
		CreateEntityInfo ceInfo  = DraftAdminDataInfo.getCreateEntityInfo();
		//Setting Admin Data information in DraftAdminData for Create
		ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_CREATE_TIME, ceInfo.getColumns()).setPostData(DBGeneratedData.UTCTIMESTAMP);
		ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_CREATE_USER, ceInfo.getColumns()).setPostData(userName);
		ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_UPDATE_TIME, ceInfo.getColumns()).setPostData(DBGeneratedData.UTCTIMESTAMP);
		ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_UPDATE_USER, ceInfo.getColumns()).setPostData(userName);
		ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER, ceInfo.getColumns()).setPostData(userName);
		if(userName!=null)
		{
			ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_CREATED_BYME, ceInfo.getColumns()).setPostData(true);
			ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_BYME, ceInfo.getColumns()).setPostData(true);
		}
		

		//Generating and setting the UUID
		ceInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_TABLE_KEY, ceInfo.getColumns()).setPostData(UUID.randomUUID());
		return ceInfo;
	}
	
	public static UpdateEntityInfo updateInfoForDraftAdminData(String userName,UUID draftAdminUUID) {
		UpdateEntityInfo uInfo  = DraftAdminDataInfo.getUpdateEntityInfo();
		//Setting Admin Data information in DraftAdminData for Create
		uInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_CREATE_TIME, uInfo.getColumns()).setPostData(DBGeneratedData.NO_CHANGE);
		uInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_CREATE_USER, uInfo.getColumns()).setPostData(DBGeneratedData.NO_CHANGE);
		uInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_UPDATE_TIME, uInfo.getColumns()).setPostData(DBGeneratedData.UTCTIMESTAMP);
		uInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_UPDATE_USER, uInfo.getColumns()).setPostData(userName);
		uInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER, uInfo.getColumns()).setPostData(userName);
		
		if(userName!=null)
		{
			uInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_CREATED_BYME, uInfo.getColumns()).setPostData(true);
			uInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_BYME, uInfo.getColumns()).setPostData(true);
		}
		
		uInfo.getColumnByName(DraftAdminDataInfo.DRAFTS_ADMIN_TABLE_KEY, uInfo.getColumns()).setPostData(draftAdminUUID);
		return uInfo;
	}
	
	public static CreateEntityInfo transformToDraftsEntityInfo(UriInfo uriInfo, ODataEntry content) {
		//We can take assumption that the payload will be empty at this point except for the generated key
		//2. change table name to reflect underlying table instead of main table
		//.3.Take decision on what happens to additional data (wether they will be calculated or not)
		CreateEntityInfo cEinfo = new CreateEntityInfo();
		try{
			EdmEntitySet entitySet = uriInfo.getTargetEntitySet();
			cEinfo.setEntityName(entitySet.getName()+DraftUtilsV2.DRAFTS_TABLE_NAME_SUFFIX);// Append Name according to shadow tableNameing
			cEinfo.setParententityName(entitySet.getName()+DraftUtilsV2.DRAFTS_TABLE_NAME_SUFFIX);
			cEinfo.setColumns(resolvePropertiesToColumns(content.getProperties()));
		}catch(Exception e) {
			logger.debug(e.getMessage(), e);
		}
		return cEinfo;
	}

	public static void modifyCreateEntityInfoForDraft(CreateEntityInfo cEInfo, ODataEntry content) {
		try{
			cEInfo.setEntityName(cEInfo.getEntityName()+DraftUtilsV2.DRAFTS_TABLE_NAME_SUFFIX);// Append Name according to shadow tableNameing
			cEInfo.setParententityName(cEInfo.getEntityName());
		}catch(Exception e) {
			logger.debug(e.getMessage(), e);
		}
	}
	
	private static List<Column> resolvePropertiesToColumns(Map<String,Object> properties) {
		List<Column> columns = new ArrayList<>();
		for(Entry<String, Object> keys:properties.entrySet()){
			String columnName = keys.getKey();
			//Create fails by Olingo v2 for navigated items as invalid column name.
			//Eliminate during this particular entity creation time.
			if(keys.getValue() != null && !(keys.getValue() instanceof ODataEntry) && !(keys.getValue() instanceof ODataDeltaFeed)){
				Column col = formNewColumnObject(columnName,null,properties.get(columnName),false );
				columns.add(col);
			}
		}
		return columns;
	}
	
	private static Column formNewColumnObject(String columnName,String parentName, Object postValue ,boolean key ){
		Column col=new Column();
		col.setName(columnName);
		if(parentName!=null){
			col.setPathName(parentName+"."+columnName);
		}else{
			col.setPathName(columnName);
		}
		col.setPostData(postValue);
		col.setpKey(key);
		return col;
	}
	
	public static List<Map<String, Object>> handleAndExecuteFromDraftTable(UriInfo uriInfo, ODataContext context, 
	    List<String> filterExpressions, SAPStatistics timings, 
	    Connection conn, ODataToCDSProcessor cdsprocessor) throws ODataException, SQLException {
	  QueryHelper qH = null;
	  ResultSet result = null;
	  switch(RequestProcessingHelper.getDatasourceProvider()) {
		case HANA:
		  qH = new HanaQueryHelperV2ForDraftFlow(uriInfo, context);
		  getFiltersForDraftTable(filterExpressions, qH);
		  result = cdsprocessor.execute(conn, timings, qH);
		  break;
		case CDS:
		  qH = new QueryHelperV2Draft(uriInfo, context);
		  getFiltersForDraftTable(filterExpressions, qH);
		  CDSQueryGenerator cdsq=new CDSQueryGenerator(qH);
		  String sql = cdsq.generateSQl();
      result = cdsprocessor.execute(sql, timings, conn.createStatement());
	    break;
		}
    
		return processResultSetForDraftsTable(result, qH, context);
	}
	
	public static ResultSet fetchResultSetFromDraftTable(UriInfo uriInfo, ODataContext context, 
      List<String> filterExpressions, SAPStatistics timings, 
      Connection conn, ODataToCDSProcessor cdsprocessor) throws ODataException, SQLException {
    QueryHelper qH = null;
    ResultSet result = null;
    switch(RequestProcessingHelper.getDatasourceProvider()) {
    case HANA:
      qH = new HanaQueryHelperV2ForDraftFlow(uriInfo, context);
      getFiltersForDraftTable(filterExpressions, qH);
      result = cdsprocessor.execute(conn, timings, qH);
      break;
    case CDS:
      qH = new QueryHelperV2Draft(uriInfo, context);
      getFiltersForDraftTable(filterExpressions, qH);
      CDSQueryGenerator cdsq=new CDSQueryGenerator(qH);
      String sql = cdsq.generateSQl();
      result = cdsprocessor.execute(sql, timings, conn.createStatement());
      break;
    }
    
    return result;
  }

  /**
   * @param filterExpressions
   * @param qH
   */
  private static void getFiltersForDraftTable(List<String> filterExpressions, QueryHelper qH) {
    //Add filters as necessary
    if(filterExpressions!=null && !filterExpressions.isEmpty()) {
      List<Filter> filters=new ArrayList<>();
      for(String expression:filterExpressions) {
        filters.add(new Filter(expression));
      }
      qH.getEntityInfo().getFilters().clear();
      qH.getEntityInfo().getFilters().addAll(filters);
    }
  }
	
	public static String generateCDSStatementForReadingFromDraftsTable(QueryHelper qH) {
      CDSQueryGenerator cdsq=new CDSQueryGenerator(qH);
      return cdsq.generateSQl();
	}
	
	public static List<Map<String, Object>> processResultSetForDraftsTable(ResultSet result, QueryHelper qH, ODataContext context) throws ODataApplicationException {
		//After getting the resultSet Calculate and append the Calculated fields for Drafts Table 
		//As of now no Manipulation on Drafts table is required
		return processResultSet(result, qH, context);
	}
	
	private static List<Map<String, Object>> processResultSet(ResultSet result, QueryHelper qH, ODataContext context) throws ODataApplicationException {
		List<Map<String, Object>> ec = new ArrayList<>();
		try {
			ec = ResultSetProcessor.toEntityCollection(result, qH.getEntityInfo());	
		}  catch (Exception e) {
			logger.error(e.getMessage(),e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND,
					"Error Occured while converting Result Set to Entities ::"+ e.getMessage(), HeaderUtil.getlocale(context), HttpStatusCodes.NOT_FOUND, e);
		}
		return ec;
	}

	public static UpdateEntityInfo transormToUpdateEntityInfoForAutoSaveDrafts(PutMergePatchUriInfo uriInfo, ODataEntry content, boolean merge)throws ODataException {
		Map<String, AdminDataAnnotation> adminDataMapForUpdate = CSNUtil.getAdminDataMapForUpdate(uriInfo.getTargetEntitySet().getEntityType().getNamespace(),uriInfo.getTargetEntitySet().getName(),false);
		UpdateEntityInfo  ueInfo= HANADMLHelperV2.transformToUpdateEntityInfo(uriInfo, content, merge, adminDataMapForUpdate);
		//Change the table name to drafts table underlying 
		ueInfo.setEntityName(ueInfo.getEntityName()+DraftUtilsV2.DRAFTS_TABLE_NAME_SUFFIX);
		ueInfo.setParententityName(ueInfo.getEntityName()+DraftUtilsV2.DRAFTS_TABLE_NAME_SUFFIX);
		return ueInfo;
	}
	
	public static DeleteEntityInfo transformToDeleteEntityInfo(String entityName, Map<String,Object> keyValue){
		DeleteEntityInfo dEinfo = new DeleteEntityInfo();
		dEinfo.setEntityName(entityName);
		dEinfo.setParententityName(entityName);
		for (Map.Entry<String, Object> entry : keyValue.entrySet()){
			dEinfo.getColumns().add(new Column(entry.getKey()).setPostData(entry.getValue()).setpKey(true));
		}
		return dEinfo;
	}
	
	public static List<Map<String, Object>> ownDraftInstances(EdmEntityType entityType, List<Map<String, Object>> activeCollection, 
			List<Map<String, Object>> draftCollection, String userName) throws EdmException {
		String  keyPropertyName = getKeyPropertyName(entityType);
		//Currently dealing only in UUID
		Map<UUID, Map<String, Object>> activeInstances = convertEntityCollectiontoMap(keyPropertyName, activeCollection);
		
		filterDraftEntityCollectionOnUser(draftCollection, userName);
		
		//Iterate over draft instances and identify if there is an active instance
		for(Map<String, Object> draftEntity:draftCollection) {
			if(activeInstances.get(draftEntity.get(keyPropertyName)) != null) {
				//If -so correct the HAS_ACTIVE_ENTITY PROPERTY
				draftEntity.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.TRUE);
			}
		}
		//return the draftInstances
		return draftCollection;
	}
	
	public static List<Map<String, Object>> unsavedActiveInstances(EdmEntityType entityType, List<Map<String, Object>> activeCollection, List<Map<String, Object>> draftCollection) throws EdmException {
		String  keyPropertyName = getKeyPropertyName(entityType);
		//Currently dealing only in UUID
		Map<UUID, Map<String, Object>> activeInstances = convertEntityCollectiontoMap(keyPropertyName, activeCollection);
		//Filter out the Drafts with InProcessUser as empty
		filterDraftEntityCollectionOnUser(draftCollection, null);
		List<Map<String, Object>> unsavedEntities = new ArrayList<>();
		//Iterate over draft instances and add them to List only if the associated AdminData Inprocess User is empty
		for(Map<String, Object> draftEntity:draftCollection) {
			if(activeInstances.get(draftEntity.get(keyPropertyName)) != null) {
				Map<String, Object> activeEntityToAdd = activeInstances.get(draftEntity.get(keyPropertyName));
				//Associate the draftAdminData to this active instance
				activeEntityToAdd.put(DraftUtilsV2.DRAFTS_ADMIN_TABLE_NAME, draftEntity.get(DraftUtilsV2.DRAFTS_ADMIN_TABLE_NAME));
				unsavedEntities.add(activeEntityToAdd);
			}
		}
		//Add missing properties to these unsaved instances
		if(!unsavedEntities.isEmpty()) {
			unsavedEntities.stream().forEach(e->{
				e.put(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY, Boolean.TRUE);
				e.put(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY, Boolean.TRUE);
				e.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.FALSE);
			});
		}
		return unsavedEntities;
	}
	
	public static List<Map<String, Object>> unchangedActiveInstances(EdmEntityType entityType, List<Map<String, Object>> activeCollection, List<Map<String, Object>> draftCollection) throws EdmException {
		String keyPropertyName = getKeyPropertyName(entityType);
		//Currently dealing only in UUID
		Map<UUID, Map<String, Object>> activeInstances = convertEntityCollectiontoMap(keyPropertyName, activeCollection);
		//Iterate over draft instances and remove them from the active Map
		for(Map<String, Object> draftEntity:draftCollection) {
			activeInstances.remove(draftEntity.get(keyPropertyName));
		}
		//Create a List from the Map for unchanged Active Instances
		List<Map<String, Object>> unchangedEntities= new ArrayList<>(activeInstances.values());
		//Add missing properties to these Active instances
		if(!unchangedEntities.isEmpty()) {
			unchangedEntities.stream().forEach(e->{
				e.put(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY, Boolean.TRUE);
				e.put(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY, Boolean.FALSE);
				e.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.FALSE);
			});
		}
		return unchangedEntities;
	}
	
	public static List<Map<String, Object>> lockedByOtherActiveInstances(EdmEntityType entityType, List<Map<String, Object>> activeCollection, List<Map<String, Object>> draftCollection, 
			String userName) throws EdmException {
		String  keyPropertyName = getKeyPropertyName(entityType);
		//Currently dealing only in UUID
		Map<UUID, Map<String, Object>> activeInstances = convertEntityCollectiontoMap(keyPropertyName, activeCollection);
		//Filter out the Drafts with InProcessUser not as same as the user <H=Get drafts not in process by current user>
		filterDraftEntityCollectionOnUserNegation(draftCollection, userName);
		List<Map<String, Object>> lockedByotherEntities= new ArrayList<>();
		//Iterate over draft instances and add them to List only if the associated drafts are locked by another user
		for(Map<String, Object> draftEntity:draftCollection) {
			if(activeInstances.get(draftEntity.get(keyPropertyName)) != null) {
				Map<String, Object> activeEntityToAdd = activeInstances.get(draftEntity.get(keyPropertyName));
				//Associate the draftAdminData to this active instance
				activeEntityToAdd.put(DraftUtilsV2.DRAFTS_ADMIN_TABLE_NAME, draftEntity.get(DraftUtilsV2.DRAFTS_ADMIN_TABLE_NAME));
				lockedByotherEntities.add(activeEntityToAdd);
			}
		}
		//Add missing properties to these unsaved instances
		if(!lockedByotherEntities.isEmpty()) {
			lockedByotherEntities.stream().forEach(e->{
				e.put(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY, Boolean.TRUE);
				e.put(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY, Boolean.TRUE);
				e.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.FALSE);
			});
		}
		return lockedByotherEntities;
	}
	
	public static List<Map<String, Object>> combinedEntries(EdmEntityType entityType, List<Map<String, Object>> activeCollection, List<Map<String, Object>> draftCollection) throws EdmException {
		String keyPropertyName = getKeyPropertyName(entityType);
		//Currently dealing only in UUID
		Map<UUID, Map<String, Object>> activeInstances = convertEntityCollectiontoMap(keyPropertyName, activeCollection);
		List<Map<String, Object>> combinedEntries = new ArrayList<>();
		//Iterate over draft instances and add the HAS_ACTIVE_ENTITY true property which have a active entity
		for(Map<String, Object> draftEntity:draftCollection) {
			if(activeInstances.get(draftEntity.get(keyPropertyName))!=null) {
				draftEntity.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.TRUE);
			}
			//remove the activeInstance as this need not be in the final combined entry
			activeInstances.remove(draftEntity.get(keyPropertyName));
		}
		//Add all draft entries to the combined entries first
		combinedEntries.addAll(draftCollection);
		List<Map<String, Object>> revisedActiveEntities = new ArrayList<>(activeInstances.values());
		//Add missing properties to active instances
		if(!revisedActiveEntities.isEmpty()) {
			revisedActiveEntities.stream().forEach(e->{
				e.put(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY, Boolean.TRUE);
				e.put(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY, Boolean.FALSE);
				e.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.FALSE);
			});
		}
		
		//Now Add all active entities to the combined list
		combinedEntries.addAll(revisedActiveEntities);
		return combinedEntries;
	}
	
	private static String getKeyPropertyName(EdmEntityType entityType) throws EdmException {
		String keyPropertyName = null;
		for(String key:entityType.getKeyPropertyNames()) {
			if(!key.equalsIgnoreCase(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY)) {
				keyPropertyName = key;
			}
		}
		return keyPropertyName;
	}
	
	private static Map<UUID, Map<String, Object>> convertEntityCollectiontoMap(final String keyPropertyName, List<Map<String, Object>> entities){
		Map<UUID, Map<String, Object>> instancesMap = new LinkedHashMap<>();
		entities.stream().forEach(e->instancesMap.put((UUID) e.get(keyPropertyName), e));
		return instancesMap;
	}
	
	private static void filterDraftEntityCollectionOnUser(List<Map<String, Object>> ec, String userName) {
		for (Iterator<Map<String, Object>> iter = ec.iterator(); iter.hasNext();) {
		    filterConditionUser(iter, userName);
		}
	}
	
	private static void filterConditionUser(Iterator<Map<String, Object>> iter,String userName) {
		Map<String, Object> draftElement = iter.next();
	    Map<String, Object> adminElement = (Map<String, Object>)draftElement.get(DraftUtilsV2.DRAFTS_ADMIN_TABLE_NAME);
		if (adminElement != null) {
		    if(adminElement.get(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER) != null && userName != null) {
		    	if(!adminElement.get(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER).equals(userName)) {
			    	iter.remove();
			    }
		    }else {
		    	//If username null or empty remove all draft entries except the ones which has an empty or null Inprocess by user 
		    	if((userName == null || userName.isEmpty()) && 
		    			(adminElement.get(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER) != null && !adminElement.get(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER).toString().isEmpty())) {
		    		iter.remove();
		    	}
		    }
		}
	}
	
	private static void filterDraftEntityCollectionOnUserNegation(List<Map<String, Object>> ec, String userName) {
		for (Iterator<Map<String, Object>> iter = ec.iterator(); iter.hasNext(); ) {
		   filterConditionUserNegation(iter, userName);
		}
	}
	
	private static void filterConditionUserNegation(Iterator<Map<String, Object>> iter,String userName){
		Map<String, Object> draftElement = iter.next();
	    Map<String, Object> adminElement = (Map<String, Object>)draftElement.get(DraftUtilsV2.DRAFTS_ADMIN_TABLE_NAME);
	    if(adminElement.get(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER) != null && userName != null) {
	    	if(adminElement.get(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER).equals(userName)) {
		    	iter.remove();
		    }
	    }else {
	    	//If username  null or empty remove all draft entries except the ones which has an empty or null Inprocess by user 
	    	if((userName==null || userName.isEmpty()) && 
	    			(adminElement.get(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER) == null || adminElement.get(DraftAdminDataInfo.DRAFTS_ADMIN_INPROCESS_USER).toString().isEmpty())) {
	    		iter.remove();
	    	}
	    }
	}	
}
