/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.edm.EdmConcurrencyMode;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmFacets;
import org.apache.olingo.odata2.api.edm.EdmLiteralKind;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.edm.EdmSimpleType;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeException;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.edm.EdmTypeKind;
import org.apache.olingo.odata2.api.edm.EdmTyped;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.uri.KeyPredicate;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.info.DeleteUriInfo;
import org.apache.olingo.odata2.api.uri.info.PostUriInfo;
import org.apache.olingo.odata2.api.uri.info.PutMergePatchUriInfo;
import org.apache.olingo.odata2.core.edm.EdmBoolean;
import org.apache.olingo.odata2.core.edm.EdmGuid;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.api.internal.AdminDataAnnotation;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Column;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.CreateEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DBGeneratedData;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DeleteEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.EntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.UpdateEntityInfo;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException;
import com.sap.cloud.sdk.service.prov.v2.rt.util.LocaleUtil;

public class HANADMLHelperV2 {
	
	final static Logger logger = LoggerFactory.getLogger(HANADMLHelperV2.class);
	
	
	public static CreateEntityInfo transformToCreateEntityInfo(PostUriInfo uriInfo, ODataEntry content ,Map<String,AdminDataAnnotation> adminDataMapForCreate)throws ODataException{
			CreateEntityInfo cEinfo=new CreateEntityInfo();
		//Process ODataEntry For AdminData
			adminDataProcessing(content, adminDataMapForCreate);
			EdmEntitySet entitySet=uriInfo.getTargetEntitySet();
			entitySet.getEntityType().getKeyPropertyNames();
			cEinfo.setEntityName(entitySet.getName());
			cEinfo.setParententityName(entitySet.getName());
			cEinfo.setColumns(resolvePropertiesToColumns(entitySet,content.getProperties()));
		return cEinfo;
	}
	
	private static List<Column> resolvePropertiesToColumns(EdmEntitySet entitySet,Map<String,Object> properties) throws EdmException{
		List<Column> columns = new ArrayList<Column>();
		for(String columnName:properties.keySet()){
				Column col=formNewColumnObject(columnName,null,properties.get(columnName),false );
				columns.add(col);
		}
		return columns;
	}
	
	private static Column formNewColumnObject(String columnName,String parentName, Object postValue ,boolean key ){
		Column col=new Column();
		col.setName(columnName);
		if(parentName!=null){
			col.setPathName(parentName+"."+columnName);
		}else{
			col.setPathName(columnName);
		}
		col.setPostData(postValue);
		col.setpKey(key);
		return col;
	}
	
	public static String getNameSpace(EdmEntitySet entitySet) throws CDSRuntimeException{
		try {
			return entitySet.getEntityType().getNamespace();
		} catch (Exception e) {
			logger.error(e.getMessage(),e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(),LocaleUtil.getLocaleforException(),
					HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
		}
	}
	
	
	
	
	public static UpdateEntityInfo transformToUpdateEntityInfo(PutMergePatchUriInfo uriInfo, ODataEntry content,boolean merge,Map<String,AdminDataAnnotation> adminDataMapForUpdate)throws ODataException{
		//Process ODataEntry For AdminData
			adminDataProcessing(content, adminDataMapForUpdate);
			UpdateEntityInfo uEinfo=new UpdateEntityInfo();
			EdmEntitySet entitySet=uriInfo.getTargetEntitySet();
			uEinfo.setEntityName(entitySet.getName());
			uEinfo.setParententityName(entitySet.getName());
			uEinfo.setColumns(resolvePropertiesToColumnsUpdate(entitySet,content.getProperties(),adminDataMapForUpdate,merge));
			uEinfo.getColumns().addAll(resolveKeyPredicatesToColumnsUpdate(entitySet, uriInfo.getKeyPredicates()));
		return uEinfo;
	}
	
	/***
	 * Returns the UpdateEntityInfo object based on PutMergePatchUriInfo and
	 * properties relevant for media and administrative data
	 * 
	 * @param uriInfo
	 *            Contains the request URI information
	 * @param properties
	 *            Contains the media information
	 * @param adminDataMapForUpdate
	 *            Contains administrative data
	 * @return UpdateEntityInfo object with media and administrative data information
	 * @throws ODataException
	 */
	public static UpdateEntityInfo transformToUpdateEntityMediaInfo(PutMergePatchUriInfo uriInfo,
			Map<String, Object> properties, Map<String, AdminDataAnnotation> adminDataMapForUpdate)
			throws ODataException {
		// Process Media Entry For AdminData
		adminDataProcessingForMedia(properties, adminDataMapForUpdate);
		UpdateEntityInfo updateEntityInfo = new UpdateEntityInfo();
		EdmEntitySet entitySet = uriInfo.getTargetEntitySet();
		updateEntityInfo.setEntityName(entitySet.getName());
		updateEntityInfo.setParententityName(entitySet.getName());
		updateEntityInfo.setColumns(
				resolvePropertiesToColumnsUpdate(entitySet, properties));
		updateEntityInfo.getColumns().addAll(resolveKeyPredicatesToColumnsUpdate(entitySet, uriInfo.getKeyPredicates()));
		return updateEntityInfo;
	}
	
	public static CreateEntityInfo transformToCreateEntityMediaInfo(PostUriInfo uriInfo,
			Map<String, Object> properties, Map<String, AdminDataAnnotation> adminDataMapForUpdate)
			throws ODataException {
		// Process Media Entry For AdminData
		adminDataProcessingForMedia(properties, adminDataMapForUpdate);
		CreateEntityInfo createEntityInfo = new CreateEntityInfo();
		EdmEntitySet entitySet = uriInfo.getTargetEntitySet();
		createEntityInfo.setEntityName(entitySet.getName());
		createEntityInfo.setParententityName(entitySet.getName());
		createEntityInfo.setColumns(
				resolvePropertiesToColumnsUpdate(entitySet, properties));
		createEntityInfo.getColumns().addAll(resolveKeyPredicatesToColumnsUpdate(entitySet, uriInfo.getKeyPredicates()));
		return createEntityInfo;
	}
	
	private static List<Column> resolvePropertiesToColumnsUpdate(EdmEntitySet entitySet,Map<String,Object> properties,Map<String,AdminDataAnnotation> adminDataMapForUpdate,boolean merge) throws EdmException{
		List<Column> columns = new ArrayList<Column>();
		if(!merge){
			for(String columnName:entitySet.getEntityType().getPropertyNames()){
				if(!(adminDataMapForUpdate!=null &&
					adminDataMapForUpdate.get(columnName)!=null &&
					adminDataMapForUpdate.get(columnName)==AdminDataAnnotation.COMPUTED_FIELD)) {
					if(!(adminDataMapForUpdate!=null &&adminDataMapForUpdate.get(columnName)!=null && adminDataMapForUpdate.get(columnName)==AdminDataAnnotation.IMMUTABLE_FIELD)){							
						Column col=formNewColumnObject(columnName,null,properties.get(columnName),false  );
							columns.add(col);
						}
				}
			}
		}else{
			for(String columnName:properties.keySet()){
					if(!entitySet.getEntityType().getKeyPropertyNames().contains(columnName)){
						Column col=formNewColumnObject(columnName,null,properties.get(columnName),false );
						columns.add(col);
					}
					
			}
		}
		// All Key Columns are added in the End from Key Predicates in addKeyColumns method
		return columns;
	}
	
	// for update of media resource
	private static List<Column> resolvePropertiesToColumnsUpdate(EdmEntitySet entitySet, Map<String, Object> properties)
			throws EdmException {
		List<Column> columns = new ArrayList<>();
		// Updating media details in DB will always be a merge request
		for (Map.Entry<String,Object> property : properties.entrySet()) {
		    String columnName = property.getKey();
			if (!entitySet.getEntityType().getKeyPropertyNames().contains(columnName)) {
			    Object postValue = property.getValue();
				Column col = formNewColumnObject(columnName, null, postValue, false);
				columns.add(col);
			}
		}
		return columns;
	}
	
	private static List<Column> resolveKeyPredicatesToColumnsUpdate(EdmEntitySet entitySet,List<KeyPredicate> keyPredicates) throws EdmException{
		List<Column> columns = new ArrayList<Column>();
		for(KeyPredicate keyPred:keyPredicates){
			EdmSimpleType edmType = (EdmSimpleType) keyPred.getProperty().getType();
			Column col=formNewColumnObject(keyPred.getProperty().getName(),null,getValueOfProperty(keyPred,edmType ),true );
			col.setColumnDataType(edmType.getName());
			columns.add(col);
			
		}		
		return columns;
	}
	
	private static Object getValueOfProperty(KeyPredicate key, EdmSimpleType edmSimpleType) {
        Object valueOfKey;
        try {
                        valueOfKey = edmSimpleType.valueOfString(key.getLiteral(), EdmLiteralKind.DEFAULT, key.getProperty().getFacets(), edmSimpleType.getDefaultType());
                        if(edmSimpleType instanceof EdmGuid){
                                        valueOfKey= valueOfKey.toString();
                        }else if (edmSimpleType instanceof EdmBoolean) {
                                        valueOfKey=(boolean)valueOfKey? 1:0;
                        }
                        return valueOfKey;
        } catch (EdmSimpleTypeException e) {
                        logger.error("Error while converting keypredicate information",e);
        } catch (EdmException e) {
                        logger.error("Error while converting keypredicate information",e);
        }
        return null;
	}

	
	
	
	public static DeleteEntityInfo transformToDeleteEntityInfo(DeleteUriInfo uriInfo)throws ODataException{
		DeleteEntityInfo dEinfo=new DeleteEntityInfo();
		
		
			EdmEntitySet entitySet=uriInfo.getTargetEntitySet();
			dEinfo.setEntityName(entitySet.getName());
			dEinfo.setParententityName(entitySet.getName());
			dEinfo.getColumns().addAll(resolveKeyPredicatesToColumnsUpdate(entitySet, uriInfo.getKeyPredicates()));
		return dEinfo;
	}
	
	public static EntityInfo transformToUpdateLockEinfo(UriInfo uriInfo)throws ODataException{
		EntityInfo einfo=new EntityInfo();
		EdmEntitySet entitySet=uriInfo.getTargetEntitySet();
		einfo.setEntityName(entitySet.getName());
		einfo.setParententityName(entitySet.getName());
		einfo.getColumns().addAll(resolveKeyPredicatesToColumnsUpdate(entitySet, uriInfo.getKeyPredicates()));
		einfo.getColumns().add(formNewColumnObject(getEtagPropertyName(uriInfo.getTargetEntitySet().getEntityType()), null, null, false ));
		return einfo;
	}
	
	
	
	
	public static String getEtagPropertyName(EdmEntityType originEntityType) throws EdmException {
		String etagPropName = null;
		List<String> propertyNames = originEntityType.getPropertyNames();
		for (String name : propertyNames) {
			EdmTyped edmTyped = originEntityType.getProperty(name);
			EdmTypeKind typeKind = edmTyped.getType().getKind();
			if (EdmTypeKind.SIMPLE.equals(typeKind)) {
				EdmProperty edmProperty = (EdmProperty) edmTyped;
				EdmFacets facet = edmProperty.getFacets();
				if(null != facet && null != facet.getConcurrencyMode()  && facet.getConcurrencyMode() == EdmConcurrencyMode.Fixed ) {
					etagPropName = name;
					break;
				}
			}
		}
		return etagPropName;
	}
	
	
	
	public static EdmSimpleType getEtagPropertyType(UriInfo uriInfo,String propertyName) throws EdmException {
		EdmEntityType originEntityType = uriInfo.getTargetEntitySet().getEntityType();
		EdmSimpleType type = (EdmSimpleType)originEntityType.getProperty(propertyName).getType();
		return type;
	}
	
	public static Object processEtagValue(UriInfo uriInfo,String etagPropertyName,Object eTagPropertyValue) throws EdmException {
		if(eTagPropertyValue instanceof String){
			return eTagPropertyValue;
		}
		Object stringRepOfEtag = null;
		EdmEntityType originEntityType = uriInfo.getTargetEntitySet().getEntityType();
		EdmTyped edmTyped = originEntityType.getProperty(etagPropertyName);
		EdmType edmType=edmTyped.getType();
		 if (edmType instanceof EdmSimpleType) {
	          EdmSimpleType edmSimpleType = (EdmSimpleType) edmType;
	          EdmProperty edmProperty = (EdmProperty) edmTyped;
	          stringRepOfEtag= edmSimpleType.valueToString(eTagPropertyValue, EdmLiteralKind.DEFAULT,edmProperty.getFacets());
		 }
		return stringRepOfEtag;
	}
	
	
	private static void adminDataProcessing(ODataEntry content ,Map<String,AdminDataAnnotation> adminData)
	{
		if(adminData!=null && !adminData.keySet().isEmpty()){
			for(Map.Entry<String,AdminDataAnnotation> entry:adminData.entrySet()){
				if(entry.getValue()==AdminDataAnnotation.COMPUTED_FIELD) {
					content.getProperties().remove(entry.getKey());
				}
                          else if(entry.getValue()==AdminDataAnnotation.IMMUTABLE_FIELD) continue;
                          else {content.getProperties().put(entry.getKey(), resolveAdminAnnotationToDBGeneratedValue(entry.getValue()));
				}
			}
		}
	}
	
	private static void adminDataProcessingForMedia(Map<String,Object> properties, Map<String,AdminDataAnnotation> adminData)
	{
		if(adminData!=null && !adminData.keySet().isEmpty()){
			for(Map.Entry<String,AdminDataAnnotation> entry:adminData.entrySet()){
				if(entry.getValue()==AdminDataAnnotation.COMPUTED_FIELD) {
					properties.remove(entry.getKey());
				}else {
					properties.put(entry.getKey(), resolveAdminAnnotationToDBGeneratedValue(entry.getValue()));
				}
			}
		}
	}
	
	
	
	private static DBGeneratedData resolveAdminAnnotationToDBGeneratedValue(AdminDataAnnotation annotation){
		switch(annotation){
			case NOW: return DBGeneratedData.UTCTIMESTAMP;
			case DB_USER: return DBGeneratedData.DB_USER;
			case APPLICATION_USER: return DBGeneratedData.APPLICATION_USER;
			case NO_CHANGE: return DBGeneratedData.NO_CHANGE;
			default:return null;
		}
	}
	
}
