/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.fasterxml.jackson.databind.JsonNode;
import com.sap.gateway.core.api.provider.data.BaseDataProviderResponse;

public class MediaUtils {
	
	public static final String mediaColumn = "mediaColumn";
	public static final String mimeTypeColumn = "mimeTypeColumn";
	public static final String mimeType = "mimeType";	
	public static final String mediaTypeAnnotation = "@Core.MediaType";
	public static final String isMediaTypeAnnotation = "@Core.IsMediaType";
	
	/**
	 * Updates the response with the media and the associated MIME type.
	 * 
	 * @param resultEntity 
	 *            Represents the entity containing the media information
	 * @param response
	 *            Response that has to be updated with the media information
	 * @param mediaColumn
	 *            Database column containing the media data
	 * @param mimeTypeColumn
	 *            Database column containing the MIME type
	 * @param mimeType
	 *            MIME type defined directly in the model
	 */
	public static void populateMediaDetails(Map<String, Object> resultEntity, BaseDataProviderResponse response,
			String mediaColumn, String mimeTypeColumn, String mimeType) {
		response.setResultEntity(resultEntity);
		response.setMediaValue((byte[]) resultEntity.get(mediaColumn));

		// mediaTypeColumnName will be null if the mediaColumn is annotated
		// with a constant content type
		// the annotation would look like this
		// @Core.MediaType: 'image/png'

		if (mimeTypeColumn == null) {
			response.setResultEntityMediaType(mimeType);
		}

		// if mediaTypeColumnName is annotated with a column name containing
		// the content type
		// the annotation would look like this
		// @Core.MediaType: =(ImageContentType)
		// ImageContentType is name of the column.
		// and ImageContentType will have the following annotation
		// @Core.IsMediaType : true
		else {
			response.setResultEntityMediaType((String) resultEntity.get(mimeTypeColumn));
		}
	}

	/**
	 * Gets the media related details for the entity set from the model.
	 * @param serviceName Name of the service
	 * @param entitySetName Name of the entity set
	 * @param csn Model obtained from the CSN file
	 * @return Media column name and either the MIME type column name or the MIME type defined directly in the model
	 */
	public static Map<String, String> getMediaAnnotationFromCSN(String serviceName, String entitySetName, JsonNode csn) {
		Map<String, String> columnTypes = new HashMap<>();
		JsonNode node = csn.get("definitions").get(serviceName + "." + entitySetName).get("elements");
		Iterator<String> it = node.fieldNames();
		while (it.hasNext()) {
			String id = (String) it.next();
			if (node.get(id).get(mediaTypeAnnotation) != null) {
				columnTypes.put(mediaColumn, id);
				if (node.get(id).get(mediaTypeAnnotation).get("=") != null)
				{
					String mediaColumnType = node.get(id).get(mediaTypeAnnotation).get("=").textValue();
					if(node.get(mediaColumnType).get(isMediaTypeAnnotation) != null &&
							node.get(mediaColumnType).get(isMediaTypeAnnotation).asBoolean())
					columnTypes.put(mimeTypeColumn,
							node.get(id).get(mediaTypeAnnotation).get("=").textValue());
				}
				else
					columnTypes.put(mimeType, node.get(id).get(mediaTypeAnnotation).textValue());
			}
		}
		return columnTypes;
	}
	
}
