/**
* The ODataToCDSProcessor  class useful to execute sqlQuery & gives  java.sql.ResultSet
*
*
* @version 1.0
* @since   2016-10-27
*/
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Deque;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmSimpleType;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeKind;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.exception.ODataApplicationException;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.processor.ODataContext;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.info.DeleteUriInfo;
import org.apache.olingo.odata2.api.uri.info.PostUriInfo;
import org.apache.olingo.odata2.api.uri.info.PutMergePatchUriInfo;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.sap.cloud.sdk.hana.sql.DatasourceFactory;
import com.sap.cloud.sdk.hana.sql.HanaDatasource;
import com.sap.cloud.sdk.hana.sql.Query;
import com.sap.cloud.sdk.service.prov.api.internal.AdminDataAnnotation;
import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;
import com.sap.cloud.sdk.service.prov.api.statistics.SAPStatistics;
import com.sap.cloud.sdk.service.prov.api.util.RequestProcessingHelper;
import com.sap.cloud.sdk.service.prov.rt.cds.CDSQueryGenerator;
import com.sap.cloud.sdk.service.prov.rt.cds.HANADMLExecutor;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Column;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.ColumnType;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.CreateEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.DeleteEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.EntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Filter;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Parameter;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.QueryHelper;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.ReadEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.UpdateEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.datasource.factory.Datasource;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException;
import com.sap.cloud.sdk.service.prov.v2.rt.core.EntityDataSourceProvider;
import com.sap.cloud.sdk.service.prov.v2.rt.util.DraftUtilsV2;
import com.sap.cloud.sdk.service.prov.v2.rt.util.HeaderUtil;
import com.sap.cloud.sdk.service.prov.v2.rt.util.LocaleUtil;

public class ODataToCDSProcessor {

	final static Logger logger = LoggerFactory.getLogger(ODataToCDSProcessor.class);
	private Long pageTop;
	private Long pageSkip;
	private QueryHelper qHForDraftMainTable;
	private static final String limitOffset = "LIMIT ? OFFSET ?";



	/*
	 * For GetEntitySet,GetEntity,GetExpanEntitySet,GetExpandEntity
	 */
	public List<Map<String, Object>> handleAndExecute(UriInfo uriInfo,ODataContext context, JsonNode csn,Connection conn,SAPStatistics timings) throws ODataException, SQLException{

		Datasource datasource = null;
		String sql = null;
		List<Map<String, Object>> resultProcessor = null;
		switch(RequestProcessingHelper.getDatasourceProvider()) {
		case HANA:
			HanaQueryHelperV2 hanaQueryHelper = new HanaQueryHelperV2(uriInfo, context, csn);
			if (pageTop != null) {
				hanaQueryHelper.setTopValue(pageTop);
			}
			if (pageSkip != null) {
				hanaQueryHelper.setSkipValue(pageSkip);
			}
			datasource = DatasourceFactory.getDatasource(hanaQueryHelper);
			Query hanaQuery = ((HanaDatasource) datasource).getHanaQuery();
			sql = datasource.getSQL();
			logger.debug("Generated SQL>>>> {}." , sql);
			try (PreparedStatement stmt = conn.prepareStatement(sql);
					ResultSet result = execute(sql, conn, timings, stmt,hanaQueryHelper.getSchema(),
							(Deque<Parameter>) hanaQuery.getParameters());) {
				resultProcessor = ResultSetProcessor.toEntityCollection(result, hanaQueryHelper.getEntityInfo());
			}
			break;
		case CDS:
			QueryHelperV2 cdsQueryHelper = new QueryHelperV2(uriInfo, context, csn);
			if (pageTop != null) {
				cdsQueryHelper.setTopValue(pageTop);
			}
			if (pageSkip != null) {
				cdsQueryHelper.setSkipValue(pageSkip);
			}
			datasource = DatasourceFactory.getDatasource(cdsQueryHelper);
			sql = datasource.getSQL();
			logger.debug("Generated SQL>>>> " + sql);
			try(Statement stmt= conn.createStatement();
					ResultSet result = execute(sql, timings, stmt);){
				resultProcessor = ResultSetProcessor.toEntityCollection(result, cdsQueryHelper.getEntityInfo());
				}
		    break;
		}
		return resultProcessor;

	}


	/*
	 * For GetMediaResource
	 */
	/***
	 * Gets the media details for the entity.
	 * @param uriInfo Contains the request URI information
	 * @param context Contains the request context information
	 * @param csn Model obtained from the CSN file
	 * @param conn Current DB Connection
	 * @param timings SAP Statistics
	 * @param mediaColumn DB column name which has the media data
	 * @return the entity details with media information
	 * @throws ODataException
	 * @throws SQLException
	 */
	public List<Map<String, Object>> handleAndExecute(UriInfo uriInfo, ODataContext context, JsonNode csn,
			Connection conn, SAPStatistics timings, String mediaColumn) throws ODataException, SQLException {

		String sql = null;
		Datasource datasource = null;
		List<Map<String, Object>> resultProcessor = null;

		switch(RequestProcessingHelper.getDatasourceProvider()) {
		case HANA:
			HanaQueryHelperV2 hanaQueryHelper = new HanaQueryHelperV2(uriInfo, context, csn , mediaColumn);
			if (pageTop != null) {
				hanaQueryHelper.setTopValue(pageTop);
			}
			if (pageSkip != null) {
				hanaQueryHelper.setSkipValue(pageSkip);
			}
			// adding the media column manually to qhv2 EntityInfo
			if (mediaColumn != null) {
				Column media = new Column(mediaColumn);
				media.setAliasName(mediaColumn);
				media.setDbaliasName(mediaColumn);
				media.setPathName(mediaColumn);
				media.setColumnDataType("Edm.Binary");
				media.setPathName(mediaColumn);
				media.setType(ColumnType.Primitive);
				hanaQueryHelper.getEntityInfo().getColumns().add(media);
			}
			datasource = DatasourceFactory.getDatasource(hanaQueryHelper);
			Query hanaQuery = ((HanaDatasource) datasource).getHanaQuery();
			sql = datasource.getSQL();
			logger.debug("Generated SQL>>>> {}." , sql);
			try (PreparedStatement stmt = conn.prepareStatement(sql);
					ResultSet result = execute(sql, conn, timings, stmt,hanaQueryHelper.getSchema(),
							(Deque<Parameter>) hanaQuery.getParameters());) {
				resultProcessor = ResultSetProcessor.toEntityCollection(result, hanaQueryHelper.getEntityInfo());
			}
			break;
		case CDS:
			QueryHelperV2 cdsQueryHelper = new QueryHelperV2(uriInfo, context, csn);
			if (pageTop != null) {
				cdsQueryHelper.setTopValue(pageTop);
			}
			if (pageSkip != null) {
				cdsQueryHelper.setSkipValue(pageSkip);
			}
			// adding the media column manually to qhv2 EntityInfo
			if (mediaColumn != null) {
				Column media = new Column(mediaColumn);
				media.setAliasName(mediaColumn);
				media.setPathName(mediaColumn);
				media.setColumnDataType("Edm.Binary");
				media.setPathName(mediaColumn);
				media.setType(ColumnType.Primitive);
				cdsQueryHelper.getEntityInfo().getColumns().add(media);
			}
			datasource = DatasourceFactory.getDatasource(cdsQueryHelper);
			sql = datasource.getSQL();
			logger.debug("Generated SQL>>>> " + sql);
			try(Statement stmt= conn.createStatement();
					ResultSet result = execute(sql, timings, stmt);){
				resultProcessor = ResultSetProcessor.toEntityCollection(result, cdsQueryHelper.getEntityInfo());
				}
		    break;
		}
		return resultProcessor;

	}

	/*
	 * For For Read of Create and Update results
	 */
	public List<Map<String, Object>> handleAndExecute(UriInfo uriInfo,ODataContext context, ODataEntry content,JsonNode csn,Connection conn,SAPStatistics timings) throws ODataException, SQLException{
		
		String sql = null;
		List<Map<String, Object>> resultProcessor = null;
		switch(RequestProcessingHelper.getDatasourceProvider()) {
			case HANA:
				HanaQueryHelperV2 hqhv2 = new HanaQueryHelperV2(uriInfo, context, content, csn);
				Datasource datasource = DatasourceFactory.getDatasource(hqhv2);
				Query hanaQuery = ((HanaDatasource) datasource).getHanaQuery();

				sql = datasource.getSQL();
				logger.debug("Generated SQL>>>> {}." , sql);
				try(PreparedStatement stmt= conn.prepareStatement(sql);
					ResultSet result = execute(sql, conn, timings, stmt,hqhv2.getSchema(),
							(Deque<Parameter>) hanaQuery.getParameters());){
					resultProcessor = ResultSetProcessor.toEntityCollection(result, hqhv2.getEntityInfo());
				}
				break;
				
			case CDS:
				QueryHelperV2 qhv2 = new QueryHelperV2(uriInfo, context, content, csn);
				CDSQueryGenerator cdsql = new CDSQueryGenerator(qhv2);
				sql = cdsql.generateSQl();
				logger.debug("Generated SQL>>>> " + sql);
				try(Statement stmt= conn.createStatement();
					ResultSet result = execute(sql, timings, stmt)){
					resultProcessor = ResultSetProcessor.toEntityCollection(result, qhv2.getEntityInfo());
				}
				break;		
		}
		
		return resultProcessor;
	}


	/*
	 * For Inline Count,Get $ Count Flow
	 */

	public Integer handleAndExecute(UriInfo uriInfo,ODataContext context, JsonNode csn,Connection conn,SAPStatistics timings,boolean count,boolean mainTable) throws ODataException, SQLException{
		Datasource datasource = null;
		String sql = null;
		Integer items = null;
		switch(RequestProcessingHelper.getDatasourceProvider()) {
			case HANA:
				HanaQueryHelperV2 hanaQueryHelper = new HanaQueryHelperV2(uriInfo, context, csn,count,mainTable);
				datasource = DatasourceFactory.getDatasource(hanaQueryHelper);
				Query hanaQuery = ((HanaDatasource) datasource).getHanaQuery();
				Deque<Parameter> parameters = hanaQuery.getParameters();
				int size = parameters.size();
				while(size-- > 0) {
					Parameter parameter = parameters.removeFirst();
					if(parameter.getType() == "Edm.Int64")
						parameter.setValue(null);
					parameters.addLast(parameter);
				}
				sql = datasource.getSQL();
				int pos = sql.indexOf(limitOffset);
				if(pos != -1) {
					String temp = sql.substring(0, pos -1);
					sql = temp;
				}
				/*
				 * For aggregated queries we need to wrap the sql separately
				 * within a SQL that would retrieve the count from the original SQL.
				 */
				if(hanaQueryHelper.isSqlAggregated()) {
					sql = "SELECT COUNT(*) AS C FROM ( " + sql + " )";
				}
				logger.debug("Generated SQL>>>> {}." , sql);
				try (PreparedStatement stmt = conn.prepareStatement(sql);
						ResultSet result = execute(sql, conn, timings, stmt,hanaQueryHelper.getSchema(),
								parameters);) {
					if(result!=null) {
						if (result.next() != false) {
							 do {
						    	 items = result.getInt(1);
						    	 break;
						    } while (result.next());
						}
					}
				}
				break;
			case CDS:
				QueryHelperV2 qhv2 = new QueryHelperV2(uriInfo, context, csn,count, false);
				datasource = DatasourceFactory.getDatasource(qhv2);
				sql = datasource.getSQL();
				/*
				 * For aggregated queries we need to wrap the sql separately
				 * within a SQL that would retrieve the count from the original SQL.
				 */
				if(qhv2.isSqlAggregated()) {
					sql = new CDSQueryGenerator(null).generateCountAggregateSQl(sql);
				}
				logger.debug("Generated SQL>>>> " + sql);
				try(Statement stmt= conn.createStatement();
					ResultSet result = execute(sql, timings, stmt)){
					if(result!=null) {
						if (result.next() != false) {
							 do {
						    	 items = result.getInt(1);
						    	 break;
						    } while (result.next());
						}
					}
				}
				break;
		}
		return items;

	}



	public ResultSet execute(String sqlQuery, SAPStatistics timings,Statement stmt)throws SQLException {
		ResultSet resultSet = null;
		long jdbcTimeStart = System.currentTimeMillis();
		try{
		  resultSet = stmt.executeQuery(sqlQuery);
		}catch(SQLException e){
			logger.error(" CDS Query failed : " + sqlQuery);
			throw e;
		}
		timings.addTimings(SAPStatistics.GENERICDATAPROVIDER, System.currentTimeMillis() - jdbcTimeStart);
		return resultSet;
	}

	public ResultSet fetchResultSetFromMainTable(UriInfo uriInfo, ODataContext context, List<String> filterExpressions, 
	    Connection conn, SAPStatistics timings, boolean pagination) throws SQLException, ODataException {
	  List<Filter> filters = getFiltersFromFilterExpression(filterExpressions);
	  ResultSet rs = null;
	  switch(RequestProcessingHelper.getDatasourceProvider()) {
	  case HANA:
	    qHForDraftMainTable = new HanaQueryHelperV2MainForDraftFlow(uriInfo, context);
	    if (null != filters) {
	      qHForDraftMainTable.getEntityInfo().getFilters().addAll(filters);
	    }
	    if(pageTop != null) {
	    	  ((HanaQueryHelperV2MainForDraftFlow)qHForDraftMainTable).setTopValue(pageTop);  
	      }
	      if(pageSkip != null) {
	    	  ((HanaQueryHelperV2MainForDraftFlow)qHForDraftMainTable).setSkipValue(pageSkip);
	      }
	    Datasource datasource = DatasourceFactory.getDatasource(qHForDraftMainTable);
	    String sqlQuery = datasource.getSQL();
	    Query hanaQuery = ((HanaDatasource) datasource).getHanaQuery();
	    rs = execute(sqlQuery, conn, timings, conn.prepareStatement(sqlQuery), 
	        qHForDraftMainTable.getSchema(),
	        (Deque<Parameter>) hanaQuery.getParameters());
	    break;
	  case CDS:
	    qHForDraftMainTable = new QueryHelperv2MainForDraftFLow(uriInfo, context);
      //Now Mod the query Helper to align the columns and filters as required
      //No Change to Change table name
      //Remove calculated fields from ReadEntityInfo
      //Add filters as necessary
      if (null != filters) {
        qHForDraftMainTable.getEntityInfo().getFilters().addAll(filters);
      }
      if(pageTop != null && pagination) {
    	  ((QueryHelperv2MainForDraftFLow)qHForDraftMainTable).setTopValue(pageTop);  
      }
      if(pageSkip != null && pagination) {
    	  ((QueryHelperv2MainForDraftFLow)qHForDraftMainTable).setSkipValue(pageSkip);
      }
      else if((pageSkip != null || pageTop != null) && !pagination) {
    	  ((QueryHelperv2MainForDraftFLow)qHForDraftMainTable).setSkipValue((long)0);
      }
      CDSQueryGenerator cdsq=new CDSQueryGenerator(qHForDraftMainTable);
      String sql = cdsq.generateSQl();
      rs = execute(sql, timings, conn.createStatement());
	    break;
	  }
	  return rs;
  }
	
  public ResultSet execute(Connection conn, SAPStatistics timings, QueryHelper qH) throws SQLException {
    Datasource datasource = DatasourceFactory.getDatasource(qH);
    String sqlQuery = datasource.getSQL();
    Query hanaQuery = ((HanaDatasource) datasource).getHanaQuery();
    return execute(sqlQuery, conn, timings, conn.prepareStatement(sqlQuery), qH.getSchema(),
        (Deque<Parameter>) hanaQuery.getParameters());
  }
  
	
	private ResultSet execute(String sqlQuery, Connection conn,SAPStatistics timings,PreparedStatement stmt, String schema , Deque<Parameter> deque)throws SQLException {
		ResultSet resultSet = null;
		final String message = "Hana Query failed";
		long jdbcTimeStart = System.currentTimeMillis();
		try{
			int index = 1;
			while(deque != null && !deque.isEmpty()) {
				Parameter param = deque.remove();
				switch(param.getType()) {
				case "Edm.Int":
					stmt.setInt(index,Integer.parseInt(param.getValue()));
					break;
				case "Edm.Int64":
					if(param.getValue() == null);
					else
						stmt.setLong(index,Long.parseLong(param.getValue()));
					break;
				case "Edm.Double":
					stmt.setDouble(index,Double.parseDouble(param.getValue()));
					break;
				case "Edm.Single":
					stmt.setFloat(index,Float.parseFloat(param.getValue()));
					break;
				case "Edm.Decimal":
					double val = Double.parseDouble(param.getValue());
					stmt.setBigDecimal(index,BigDecimal.valueOf(val));
					break;
				case "Edm.Binary":
					stmt.setBytes(index, param.getValue().getBytes());
					break;
				case "Edm.Guid":
					stmt.setString(index,param.getValue().toLowerCase());
					break;
				case "Edm.String":
					stmt.setString(index,param.getValue());
					break;
				case "Edm.Date":
				case "Edm.DateTimeOffset":
					String CDSType = CSNUtil.getCDSDataType(param.getFqEntityName(), schema , param.getName());
					String parameter = param.getValue();
					if (parameter.contains("'")){
						parameter = parameter.replace("'","");
					}
					if(CDSType.equals("cds.Date")) {
						DateTime dt = new DateTime(parameter, DateTimeZone.UTC);
						stmt.setDate(index,new java.sql.Date( dt.getMillis()));
					}else if (CDSType.equals("cds.Timestamp")||CDSType.equals("cds.DateTime")) {
						DateTime dt = new DateTime(parameter, DateTimeZone.UTC);
						stmt.setTimestamp(index,new java.sql.Timestamp( dt.getMillis()));
					}else if (CDSType.equals("cds.Time")) {
						DateTime dt = new DateTime(parameter, DateTimeZone.UTC);
						stmt.setTime(index,new java.sql.Time( dt.getMillis()));
					}else {
						stmt.setString(index,param.getValue());
					}
					break;
				default:
					stmt.setString(index,param.getValue());
				}
				index++;
			}

			resultSet = stmt.executeQuery();

		}catch(SQLException e){
			logger.error(" {} : {}." , message, sqlQuery);
			throw e;
		}
		timings.addTimings(SAPStatistics.GENERICDATAPROVIDER, System.currentTimeMillis() - jdbcTimeStart);
		return resultSet;
	}



	public CreateEntityInfo transormToCreateEntityInfo(PostUriInfo uriInfo, ODataEntry content ,Map<String,AdminDataAnnotation> adminDataMapForCreate)throws ODataException {
		return HANADMLHelperV2.transformToCreateEntityInfo(uriInfo, content,adminDataMapForCreate);
	}


	public Map<String,Object> executeInsert(PostUriInfo uriInfo,CreateEntityInfo ceInfo, Connection conn, SAPStatistics timings)throws ODataApplicationException, SQLException, IOException {
		try {
			HANADMLExecutor hdmle=new HANADMLExecutor(HANADMLHelperV2.getNameSpace(uriInfo.getStartEntitySet()));
			return hdmle.processInsert(ceInfo, conn, timings);
		}catch (SQLIntegrityConstraintViolationException e) {
			logger.error(e.getMessage(),e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.CONFLICT, "Duplicate Entity Present in Datasource",LocaleUtil.getLocaleforException(),
					HttpStatusCodes.CONFLICT, e);
		}
	}


	public UpdateEntityInfo transormToUpdateEntityInfo(PutMergePatchUriInfo uriInfo, ODataEntry content,boolean merge,Map<String,AdminDataAnnotation> adminDataMapForUpdate)throws ODataException {
		return HANADMLHelperV2.transformToUpdateEntityInfo(uriInfo, content, merge,adminDataMapForUpdate);
	}

	/***
	 * Returns the UpdateEntityInfo object based on PutMergePatchUriInfo and
	 * properties relevant for media and administrative data
	 *
	 * @param uriInfo
	 *            Contains the request URI information
	 * @param properties
	 *            Contains the media information
	 * @param adminDataMapForUpdate
	 *            Contains administrative data
	 * @return UpdateEntityInfo object with media and administrative data information
	 * @throws ODataException
	 */
	public UpdateEntityInfo transormToUpdateEntityMediaInfo(PutMergePatchUriInfo uriInfo,
			Map<String, Object> properties, Map<String, AdminDataAnnotation> adminDataMapForUpdate)
			throws ODataException {
		return HANADMLHelperV2.transformToUpdateEntityMediaInfo(uriInfo, properties, adminDataMapForUpdate);
	}

	public CreateEntityInfo transormToCreateEntityMediaInfo(PostUriInfo uriInfo,
			Map<String, Object> properties, Map<String, AdminDataAnnotation> adminDataMapForUpdate)
			throws ODataException {
		return HANADMLHelperV2.transformToCreateEntityMediaInfo(uriInfo, properties, adminDataMapForUpdate);
	}

	public void executeUpdate(PutMergePatchUriInfo uriInfo,UpdateEntityInfo ueInfo, Connection conn, SAPStatistics timings)throws ODataApplicationException, SQLException, IOException {
		try {
			HANADMLExecutor hdmle=new HANADMLExecutor(HANADMLHelperV2.getNameSpace(uriInfo.getStartEntitySet()));
			int re=hdmle.processUpdate(ueInfo, conn, timings);
			if(re==0){
				CDSRuntimeException ce=new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, "Entity Not Found",LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND);
				logger.error(ce.getMessage(),ce);
				throw ce;
			}
		} catch(CDSRuntimeException ce) {
			throw ce;
		}
	}


	public DeleteEntityInfo transormToDeleteEntityInfo(DeleteUriInfo uriInfo)throws ODataException {
		return HANADMLHelperV2.transformToDeleteEntityInfo(uriInfo);
	}

	public void executeDelete(DeleteUriInfo uriInfo,DeleteEntityInfo deInfo, Connection conn, SAPStatistics timings)throws ODataApplicationException, SQLException, IOException {
		try {
			HANADMLExecutor hdmle=new HANADMLExecutor(HANADMLHelperV2.getNameSpace(uriInfo.getStartEntitySet()));
			int re=hdmle.processDelete(deInfo, conn, timings);
			if(re==0){
				CDSRuntimeException ce=new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, "Entity Not Found",LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND);
				logger.error(ce.getMessage(),ce);
				throw ce;
			}
		}catch(CDSRuntimeException ce) {
			throw ce;
		}
	}

	public EntityInfo transormToEntityInfoForLock(UriInfo uriInfo)throws ODataException {
		return HANADMLHelperV2.transformToUpdateLockEinfo(uriInfo);
	}

	public Object executeUpdateForLock(UriInfo uriInfo,EntityInfo eInfo, Connection conn)throws ODataApplicationException, SQLException, EdmException{
		HANADMLExecutor hdmle=new HANADMLExecutor(HANADMLHelperV2.getNameSpace(uriInfo.getStartEntitySet()));
		try(PreparedStatement ps=hdmle.executeRowLockForUpdate(eInfo, conn);
			ResultSet rs=ps.executeQuery();) {
			Map<String, Object> result = null;
			Object eTagValue = null;
			EdmSimpleType type = null;
			while (rs.next()) {
				result = ResultSetProcessor.extractResultSetRow(rs);
			}
			if (result != null) {
				//Extract the value of Etag propName populate Map and return
				String propName=HANADMLHelperV2.getEtagPropertyName(uriInfo.getTargetEntitySet().getEntityType());
				type=HANADMLHelperV2.getEtagPropertyType(uriInfo, propName);
				Optional<String> filterProperty = result.keySet().stream().filter(key -> propName.equalsIgnoreCase(key)).findFirst();
				eTagValue = (filterProperty.isPresent() ? result.get(filterProperty.get()) : null);
			} else {
				CDSRuntimeException ce=new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND, "Entity Not Found",LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_FOUND);
				logger.error(ce.getMessage(),ce);
				throw ce;
			}
			if(type == EdmSimpleTypeKind.Boolean.getEdmSimpleTypeInstance() && eTagValue!=null) {
				eTagValue = eTagValue.toString().equals("1");
			}
			return eTagValue;
		}
	}


	public void setPageTop(Long pageTop) {
		this.pageTop = pageTop;
	}


	public void setPageSkip(Long pageSkip) {
		this.pageSkip = pageSkip;
	}

  /**
   * @param filterExpressions
   * @return
   */
  private List<Filter> getFiltersFromFilterExpression(List<String> filterExpressions) {
    List<Filter> filters=null;
	  if(filterExpressions != null && !filterExpressions.isEmpty()) {
      filters=new ArrayList<>();
      for(String expression:filterExpressions) {
        filters.add(new Filter(expression));
      }
	  }
    return filters;
  }

	public List<Map<String, Object>> processResultSetForMainTable(ODataContext context, ResultSet result) throws ODataApplicationException {
		Map<String,List<String>> draftTree = CSNUtil.getDraftTree(qHForDraftMainTable.getSchema());
		List<Map<String, Object>> ec = this.processResultSet(context, result);
		//After getting the resultSet Calculate and append the Calculated fields for Drafts Table
		for(int i=0;i<ec.size();i++) {
			Map<String, Object> entityActive = ec.get(i);
			this.correctionInMainEntities(entityActive, determineLastNavigatedEntityName(qHForDraftMainTable.getSchema(), qHForDraftMainTable.getEntityInfo()), qHForDraftMainTable.getSchema(), draftTree);
		}
		return ec;
	}

	/*
	 * Parses the result set and creates Entity Collection
	 */
	public List<Map<String, Object>> processResultSet(ODataContext context, ResultSet result) throws ODataApplicationException {
		List<Map<String, Object>> ec;
		try {
			ec = ResultSetProcessor.toEntityCollection(result, qHForDraftMainTable.getEntityInfo());
		}  catch (Exception e) {
			logger.error(e.getMessage(),e);
			throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.NO_ENTITY_FOUND,
					"Error occured while converting result set to entities:"+ e.getMessage(), HeaderUtil.getlocale(context), HttpStatusCodes.NOT_FOUND, e);
		}
		return ec;
	}

	private void correctionInMainEntities(Map<String, Object> rootEntityExpanded, String entityName, String serviceName, Map<String,List<String>> draftTree){
		if(CSNUtil.isDraftEnabledEntity(entityName, serviceName)|| CSNUtil.isDraftNodeEntity(entityName, serviceName)) {
			addPropertiesToMainDraftEntity(rootEntityExpanded);
			for(Map.Entry<String, Object> entry : rootEntityExpanded.entrySet()){
				if(entry.getValue() instanceof List) { //1:N association
					List<Map<String, Object>> entityCollection = (List<Map<String, Object>>)entry.getValue();
					String associatedEntityName = EntityDataSourceProvider.getAssociatedEntityName(serviceName, entityName, entry.getKey());
					for(int i=0; i<entityCollection.size(); i++){
						this.correctionInMainEntities((Map<String, Object>)entityCollection.get(i), associatedEntityName, serviceName, draftTree);
					}
				}else if(entry.getValue() instanceof Map) { //1:1 association
					String associatedEntityName = EntityDataSourceProvider.getAssociatedEntityName(serviceName, entityName, entry.getKey());
					this.correctionInMainEntities((Map<String, Object>)entry.getValue(), associatedEntityName, serviceName, draftTree);
				}
			}
		}
	}

	private void addPropertiesToMainDraftEntity(Map<String, Object> entityActive){
		entityActive.put(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY, Boolean.TRUE);
		entityActive.put(DraftUtilsV2.DRAFTS_HASACTIVE_ENTITY, Boolean.FALSE);
		entityActive.put(DraftUtilsV2.DRAFTS_HASDRAFT_ENTITY, Boolean.FALSE);
	}

	private static String determineLastNavigatedEntityName(String serviceName, ReadEntityInfo eInfo){
		ReadEntityInfo lastNavEinfo=eInfo;
		String associatedEntityName = eInfo.getEntityName();
		while(lastNavEinfo.getEntityNavigated()!=null){
			associatedEntityName = EntityDataSourceProvider.getAssociatedEntityName(serviceName, associatedEntityName, lastNavEinfo.getEntityNavigated().getEntityName());
			lastNavEinfo=lastNavEinfo.getEntityNavigated();
		}

		return associatedEntityName;
	}


}
