/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import static com.sap.cloud.sdk.service.prov.api.internal.SQLMapping.convertToUpperCaseIfRequired;
import static com.sap.cloud.sdk.service.prov.rt.cds.AuthorizationFilter.addAuthorizationFilter;
import static com.sap.cloud.sdk.service.prov.v2.rt.cds.util.QueryHelperUtility.getEntityTypeMap;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.Stack;
import java.util.StringTokenizer;
import java.util.stream.Collectors;

import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.edm.EdmAnnotationAttribute;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmMultiplicity;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.edm.EdmSimpleType;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeKind;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.exception.MessageReference;
import org.apache.olingo.odata2.api.exception.ODataApplicationException;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.processor.ODataContext;
import org.apache.olingo.odata2.api.uri.KeyPredicate;
import org.apache.olingo.odata2.api.uri.NavigationPropertySegment;
import org.apache.olingo.odata2.api.uri.NavigationSegment;
import org.apache.olingo.odata2.api.uri.SelectItem;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.expression.CommonExpression;
import org.apache.olingo.odata2.api.uri.expression.ExpressionKind;
import org.apache.olingo.odata2.api.uri.expression.FilterExpression;
import org.apache.olingo.odata2.api.uri.expression.MemberExpression;
import org.apache.olingo.odata2.api.uri.expression.OrderByExpression;
import org.apache.olingo.odata2.api.uri.expression.OrderExpression;
import org.apache.olingo.odata2.api.uri.expression.PropertyExpression;
import org.apache.olingo.odata2.core.edm.provider.EdmEntityTypeImplProv;
import org.apache.olingo.odata2.core.edm.provider.EdmNavigationPropertyImplProv;
import org.apache.olingo.odata2.core.uri.KeyPredicateImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.sap.cloud.sdk.hana.sql.DateUtils;
import com.sap.cloud.sdk.service.prov.api.filter.BinaryExpressionNode;
import com.sap.cloud.sdk.service.prov.api.filter.ExpressionNode;
import com.sap.cloud.sdk.service.prov.api.filter.LiteralNode;
import com.sap.cloud.sdk.service.prov.api.filter.PropertyNode;
import com.sap.cloud.sdk.service.prov.api.filter.impl.FilterNodeInternal;
import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;
import com.sap.cloud.sdk.service.prov.api.internal.SQLMapping;
import com.sap.cloud.sdk.service.prov.api.security.AuthorizationService;
import com.sap.cloud.sdk.service.prov.api.security.UnauthorizedException;
import com.sap.cloud.sdk.service.prov.rt.cds.CDSQueryGenerator;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Column;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.ColumnType;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Filter;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.OrderBy;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.QueryHelper;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.ReadEntityInfo;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException.MessageKeys;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.util.QueryHelperUtility;
import com.sap.cloud.sdk.service.prov.v2.rt.util.DraftUtilsV2;
import com.sap.cloud.sdk.service.prov.v2.rt.util.LocaleUtil;

public class QueryHelperV2 implements QueryHelper {

    final static Logger logger = LoggerFactory.getLogger(QueryHelperV2.class);
    private static final String READ = "READ";
    private static final String ACCESS_DENIED = "The user does not have access to perform this operation.";
    private static final String DRAFT_ROOT = "isDraftRoot";
    private static String ALIASHLP = "ZZ";
    private static String sapNamespace = "http://www.sap.com/Protocols/SAPData";
    private static String PARAMETER_ENTITY_SUFFIX = "Parameters";
    private static String PARAMETER_NAVIGATION_PROPERTY = "Set";
    private String schema;
    private ReadEntityInfo eInfo;
    private Long topValue;
    private Long skipValue;
    private String targetWhereClause = "";
    private List<String> globalSelect = new ArrayList<String>();
    private List<String> selectExtension = new ArrayList<String>();
    private HashMap<String, ReadEntityInfo> orderByFromNavigation = new HashMap<>();
    private String globalOrderByStringFromURI = null;
    private List<Map<String, String>> orderByPropsFromCSON = new ArrayList<Map<String, String>>();
    private JsonNode csn;
    private String serviceName = "";
    private UriInfo globalUriInfo;
    private boolean isAggregateEntity = false;
    private boolean isSqlAggregated = false;
    private boolean isDraftRoot = false;
    private boolean isDraft = false;
    private Object andFilters = null;
    private String finalStr = null;
    private EdmEntitySet globaltargetEntitySet;
    private String finalCustomExpressionString = null;
    private String finalOperator = " and ";
    private String customFilterOperator = "and";
    /*
     * Constructor To Obtain QueryHelper For Obtaining the Created Entity
     */
    public QueryHelperV2(UriInfo uriInfo, ODataContext context, ODataEntry content, JsonNode csn)
            throws ODataException {

        this.serviceName = uriInfo.getTargetEntitySet().getEntityType().getNamespace();
        this.csn = csn;
        getQueryHelperEntityUriInfo(uriInfo, context, content);
    }

    /*
     * ReadEntitySet Constructor and ReadEntity Constructor and Media Resource
     * Constructor
     */
    public QueryHelperV2(UriInfo uriInfo, ODataContext context, boolean isDraftRoot, boolean isDraft, JsonNode csn)
            throws ODataException {
        this.serviceName = uriInfo.getTargetEntitySet().getEntityType().getNamespace();
        this.csn = csn;
        this.isDraftRoot = isDraftRoot;
        this.isDraft = isDraft;
        this.selectExtension = (List<String>) context.getParameter("selectOption");
        processUriInfo((UriInfo) uriInfo, context);
    }

    /*
     * ReadEntitySet Constructor and ReadEntity Constructor and Media Resource
     * Constructor
     */
    public QueryHelperV2(UriInfo uriInfo, ODataContext context, JsonNode csn) throws ODataException {
        this(uriInfo, context, false, false, csn);
    }

    /*
     * Constructor For $count and Inline Count Extra parameter count helps us
     * redirect to count Flow
     *
     */
    public QueryHelperV2(UriInfo uriInfo, ODataContext context, JsonNode csn, boolean count, boolean isDraft)
            throws ODataException {
        this.serviceName = uriInfo.getTargetEntitySet().getEntityType().getNamespace();
        this.csn = csn;
        this.isDraft = isDraft;
        processUriInfo((UriInfo) uriInfo, context);

        /*
         * Correction To EntityInfo For Count
         */
        // Traverse to last entity set and change columns to be fetched
        ReadEntityInfo lastnavEinfo = eInfo;
        while (lastnavEinfo.getEntityNavigated() != null) {
            // Remove Orderby
            lastnavEinfo.getOrderby().clear();
            lastnavEinfo.getOrderBySeq().clear();
            lastnavEinfo = lastnavEinfo.getEntityNavigated();
        }
        // Remove Orderby last time
        lastnavEinfo.getOrderby().clear();
        lastnavEinfo.getOrderBySeq().clear();

        // Do not remove group by if annotated with sap semantics aggregate
        if (!this.isAggregateEntity || uriInfo.isCount()) {
            lastnavEinfo.getColumns().clear();
        }

        // Remove any Expand Entries
        lastnavEinfo.getEntitiesExpanded().clear();

        // Remove Top Skip
        this.topValue = null;
        this.skipValue = null;
        // check if count or inlinecount on entity annotated as aggregated
        if (this.isAggregateEntity) {
            if (uriInfo.isCount() || uriInfo.getSelect().isEmpty()) {
                lastnavEinfo.setCountStar(true);
            } else {
                this.isSqlAggregated = true;
            }
        } else {
            // Only one keyPropName is required even in case of composite keys .// Count is
            // an aggregation against parent entity set logic should hold
            String keypropName = uriInfo.getTargetEntitySet().getEntityType().getKeyPropertyNames().get(0);
            Column c = new Column("COUNT(" + keypropName + ") ");
            c.setDbaliasName("C").setType(ColumnType.Primitive);
            lastnavEinfo.getColumns().add(c);
        }
    }

    private void getQueryHelperEntityUriInfo(UriInfo uriInfo, ODataContext context, ODataEntry content)
            throws ODataException {
        EdmEntitySet entitySet = uriInfo.getTargetEntitySet();
        this.schema = entitySet.getEntityType().getNamespace();
        this.selectExtension = (List<String>) context.getParameter("selectOption");
        List<KeyPredicate> keypredicates = new ArrayList<KeyPredicate>();
        List<EdmProperty> kepProps = entitySet.getEntityType().getKeyProperties();
        if (uriInfo.getTargetKeyPredicates() != null && !uriInfo.getTargetKeyPredicates().isEmpty()) {
            keypredicates = uriInfo.getTargetKeyPredicates();
        } else {
            for (EdmProperty key : kepProps) {
                KeyPredicate kp = null;
                if ((EdmSimpleType) key.getType() == EdmSimpleTypeKind.DateTimeOffset.getEdmSimpleTypeInstance()) {
                    kp = new KeyPredicateImpl(
                            DateUtils.convertCalendartoLiteral((Timestamp) content.getProperties().get(key.getName())),
                            key);
                } else {
                    kp = new KeyPredicateImpl(HANADMLHelperV2
                            .processEtagValue(uriInfo, key.getName(), content.getProperties().get(key.getName()))
                            .toString(), key);
                }
                keypredicates.add(kp);
            }
        }
        this.globalUriInfo = uriInfo;
        this.eInfo = prepareEntityInfo(entitySet, "", null, keypredicates, "");
    }

    private void processUriInfo(UriInfo uriInfo, ODataContext context) throws ODataException {
        EdmEntitySet entitySet = uriInfo.getTargetEntitySet();
        this.schema = entitySet.getEntityType().getNamespace();
        this.globalUriInfo = uriInfo;
        this.andFilters = context.getParameter("andFilters");
        if (this.andFilters != null)
            this.findRootCustomNode(andFilters);
        Object draftRoot = context.getParameter(DRAFT_ROOT);
        if (draftRoot != null) {
            this.isDraftRoot = Boolean.valueOf((String) draftRoot);
            context.setParameter(DRAFT_ROOT, null);
        }
        // Start Counstruction of ReadEntityInfo from the First urlpart (Normal
        // Scenario)

        if (isParameterEntity(uriInfo.getStartEntitySet())) {// Aggregation and Parameter Special Case
            prepareEntityInfoForParameterViews(uriInfo, false);
        } else {
            this.eInfo = prepareEntityInfo(uriInfo.getStartEntitySet(), "", uriInfo.getNavigationSegments(), "");
        }

        if (uriInfo.isCount() && this.isDraft && uriInfo.getSelect().isEmpty()) {
            this.eInfo.setCountStar(true);
        }
    }

    private void findRootCustomNode(Object andFilters) {

        finalOperator = ((FilterNodeInternal) andFilters).getOperator();
        finalOperator = getOperandValue(finalOperator);
        Queue<FilterNodeInternal> queue1 = new LinkedList<FilterNodeInternal>();
        queue1.add((FilterNodeInternal) andFilters);

        // ExpressionNode customRootNode = process(queue1);

        boolean allCustomNode = true;
        List<FilterNodeInternal> childrenNode = ((FilterNodeInternal) andFilters).getChildren();

        for (FilterNodeInternal child : childrenNode) {
            if (!child.isCustomNode()) {
                allCustomNode = false;
                break;
            }
        }

        if (allCustomNode) {
            customFilterOperator = finalOperator;
            finalCustomExpressionString = computeCustomExpressionString((FilterNodeInternal) andFilters);
        } else {
            Optional<FilterNodeInternal> customRootNode = ((FilterNodeInternal) andFilters).getChildren().stream()
                    .filter(node -> node.isCustomNode()).findAny();

            if (customRootNode.isPresent()) {
                customFilterOperator = customRootNode.get().getOperator();
                customFilterOperator = getOperandValue(customFilterOperator);
                finalCustomExpressionString = computeCustomExpressionString(customRootNode.get());
            }
        }

    }

    private String getOperandValue(String finalOperator2) {
        String operator;
        switch (finalOperator2) {
            // Arithmetic operation
            case "AND":
                operator = " and ";
                break;
            case "OR":
                operator = " or ";
                break;
            default:
                operator = " and ";
        }
        return operator;
    }

    private String computeCustomExpressionString(ExpressionNode expressionNode) {
        StringBuilder str = new StringBuilder();

        Queue<String> q = new LinkedList<String>();
        Queue<ExpressionNode> qNode = new LinkedList<ExpressionNode>();
        // Stack<ExpressionNode> st = new Stack();

        List<FilterNodeInternal> children = ((FilterNodeInternal) expressionNode).getChildren();
        children.forEach(exprNode -> qNode.add(exprNode));

        if (expressionNode instanceof BinaryExpressionNode) {
            ExpressionNode node1 = qNode.remove();
            ExpressionNode node2 = qNode.remove();

            if (!(node1 instanceof BinaryExpressionNode || node2 instanceof BinaryExpressionNode)) {
                str.append(getValueFromNode(node2));
                str.append(QueryHelperUtility.getBinaryOperandValue((BinaryExpressionNode) expressionNode));
                str.append("'" + getValueFromNode(node1) + "'");
                q.add(str.toString());
            } else {
                computeCustomExpressionString(node1);
                computeCustomExpressionString(node2);
            }
        }
        while (!q.isEmpty()) {
            if (null != finalStr) {
                //  finalStr = "(" + finalStr + ")and(" + q.remove() + ")";
                finalStr = "( " + finalStr + " )"+ customFilterOperator +"( "+ q.remove() + " )";
            } else {
                finalStr = q.remove();
            }
        }
        return finalStr;

    }

    public Object visitCustomProperty(String uriLiteral) {
        // parseToWhereExpression difference 1 in SQLFilterExpressionVisitor
        // this.tableAlias + "." + fieldName; difference 2 in
        // SQLFilterExpressionVisitor
        String persistenceName = uriLiteral;
        if (globaltargetEntitySet != null) {
            try {
                String namespace = globaltargetEntitySet.getEntityType().getNamespace();
                String fqName = namespace + "." + globaltargetEntitySet.getName();
                persistenceName = CDSQueryGenerator.getPersistenceName(fqName, uriLiteral);
            } catch (EdmException e) {
                logger.error("Error while processing filter", e);
            }
        }
        String pathPrefixedName = "ROOT_ENTITY_SELECT_ALIAS" + "." + "PATH_FROM_ROOT."
                + convertToUpperCaseIfRequired(persistenceName);
        /*
         * try { if(edmProperty.getType().equals(EdmSimpleTypeKind.Guid.
         * getEdmSimpleTypeInstance())) { return "LCASE(" + pathPrefixedName + ")";
         * UUID/Guid values need case insensitive comparison, See comments in
         * visitLiteral method. } } catch (EdmException e) { return pathPrefixedName; }
         */
        return pathPrefixedName;
    }

    private String getValueFromNode(ExpressionNode node) {

        String right = null;
        if (node instanceof PropertyNode) {
            String str = ((PropertyNode) node).getPath();
            return (String) visitCustomProperty(str);
        }

        if (node instanceof LiteralNode) {
            right = ((LiteralNode) node).getValue().toString();
            if ((right.contains("datetimeoffset'")) || (right.contains("datetime'")) || (right.contains("time'"))
                    || DateUtils.checkIfDurationDataType(right)) {
                right = DateUtils.dateLiteralresolver(right);
                right = right.replace("'", "");
            }
        }
        return right;
    }

    private ReadEntityInfo prepareEntityInfo(EdmEntitySet entitySet, String parentAssociation,
                                             List<NavigationSegment> navigationSegments, Object... modifiers) throws ODataException {
        String navigationTree = "";
        if (modifiers[0] instanceof String) {
            navigationTree = (String) modifiers[0];
        } else if (modifiers[1] instanceof String) {
            navigationTree = (String) modifiers[1];
        }
        ReadEntityInfo re = new ReadEntityInfo();
        String entityName;
        // boolean marks this entity for processing All filters,OrderBy,Expands
        // Originate from this segment
        boolean targetArtifact = false;// Default Assume this is a navigation Segment
        boolean startArtifact = false;

        // Prepare Global Select
        prepareGlobalSelectList();
        //Prepare Global OrderBy List
        globalOrderByStringFromURI = QueryHelperUtility.prepareGlobalOrderBy(globalUriInfo);

        this.orderByPropsFromCSON = CSNUtil.getOrderByProperties(serviceName, entitySet.getName());

        if (globalSelect != null && !globalSelect.isEmpty()) {
            re.setSelectOption(globalSelect);
        }

        if (parentAssociation.isEmpty()) {
            startArtifact = true;
        }

        // If it is the starting artifact :: then entity Name should be entity Set Name
        // otherwise entityName should be navigation Property Name
        entityName = startArtifact ? entitySet.getName() : navigationSegments.get(0).getNavigationProperty().getName();

        if ((navigationSegments == null || navigationSegments.isEmpty()) // This means the call is on startEntitySet
                // Itself. // No navigation
                || (!startArtifact && navigationSegments != null && navigationSegments.size() == 1)) {// This means we
            // have
            // recursively
            // arrived at
            // the last
            // Navigation
            // segment for
            // processing
            targetArtifact = true;
        }
        // Initial Code default Settings
        // Suffix with draft table name
        if (this.isDraft && startArtifact) {
            re.setEntityName(entityName + DraftUtilsV2.DRAFTS_TABLE_NAME_SUFFIX);
        } else {
            re.setEntityName(entityName);
        }
        re.setServiceName(serviceName);
        re.setParententityName(parentAssociation);

        // Determine If the CurrentEntitySet is the Target Entity Set (Last Set in the
        // Entity)

        if (targetArtifact) { // Logically all the magic happens on this artifact
            re.setFetchColumns(true);
            // set the columns

            EdmEntityType type = entitySet.getEntityType();
            EdmAnnotationAttribute annotationAttribute = type.getAnnotations().getAnnotationAttribute("semantics",
                    sapNamespace);

            if (annotationAttribute != null && annotationAttribute.getText().equals("aggregate")) {
                this.isAggregateEntity = true;
                modifyEntityInfoForAggregation(re, globalUriInfo.getSelect(), type, type.getName(),
                        globalUriInfo.getFilter());
            } else {
                re.getColumns().addAll(resolvePropertiesToColumns(re.getEntityName(), entitySet.getEntityType(),
                        parentAssociation != null ? parentAssociation : null, ""));
            }
            // set key predicates as necessary
            if (!startArtifact) {

                // Add Predicates
                re.getFilters()
                        .addAll(navigationSegments.get(0).getKeyPredicates() != null
                                ? getKeyPredicates(navigationSegments.get(0).getKeyPredicates())
                                : null);
                // handle custom query options like 'search'
                handleSearchQueryOption(globalUriInfo.getCustomQueryOptions(), re, entitySet.getName(), true,
                        navigationTree);
            } else {
                // Check if Modifiers have additional Predicates in Case of getting Created and
                // Updated entity Flow
                if (modifiers.length > 0 && modifiers[0] != null && modifiers[0] instanceof List<?>) {
                    re.getFilters().addAll(getKeyPredicates((List<KeyPredicate>) modifiers[0]));
                } // Add Predicates For Read Flow
                else {
                    re.getFilters()
                            .addAll(globalUriInfo.getKeyPredicates() != null
                                    ? getKeyPredicates(globalUriInfo.getKeyPredicates())
                                    : null);
                    re.setKeyPredicates(globalUriInfo.getKeyPredicates() != null
                            ? QueryHelperUtility.getKeyPredicatesMap(globalUriInfo.getKeyPredicates(), this.isDraftRoot)
                            : null);
                }
                // handle custom query options like 'search'
                handleSearchQueryOption(globalUriInfo.getCustomQueryOptions(), re, entitySet.getName(), false,
                        navigationTree);
            }

            // Add authorization filter
            initAuth(entitySet, re);

            // Handle Filters for last segment
            if (globalUriInfo.getFilter() != null) {
                String filterExp = null;
                filterExp = prepeareFilter(globalUriInfo);
                if (filterExp != null && !"".equals(filterExp.trim())) {
                    if (finalCustomExpressionString != null && !"".equals(finalCustomExpressionString.trim()))
                        filterExp = "( " + filterExp + finalOperator + "( " + finalCustomExpressionString + " ) )";
                    re.getFilters().add(new Filter(filterExp));
                }
                // setting filter expression for Statble id
                re.setFilterExpForStableId(globalUriInfo.getFilter().getExpressionString());

            } else if (finalCustomExpressionString != null && !"".equals(finalCustomExpressionString.trim())) {
                String filterExp = "( " + finalCustomExpressionString + " )";
                re.getFilters().add(new Filter(filterExp));
            }

            // Add OrderBy for navigation prop
            if (globalUriInfo.getOrderBy() != null)
                updateOrderBy(globalUriInfo.getOrderBy(), re, entitySet);

            // Add OrderBy
            if (globalUriInfo.getOrderBy() != null)
                setOrderByInReadEntityInfo(globalUriInfo.getOrderBy(), re);
            // Add OrderBy if it is set in the CSON
            QueryHelperUtility.addOrderBypropertiesFromCSON(orderByPropsFromCSON, re, entitySet.getEntityType());
            // Add OrderBy for Key predicates after actual OrderBy (Cross Check they are not
            // over-riding actual order by in UriInfo)
            if (!isAggregateEntity)
                addOrderByForKeyPredicates(re, entitySet.getEntityType());

            // Recursive Call for Expand

            if (globalUriInfo.getExpand() != null && !globalUriInfo.getExpand().isEmpty()) {
                re.getEntitiesExpanded().addAll(
                        prepExpandInfoList(globalUriInfo.getExpand(), parentAssociation + ALIASHLP + entityName));
            }

        } else { // This is a part of the navigation Segment , Do not Fetch Columns, Do not Set
            // Columns , Take into account key predicate filter and parent OrderBy
            re.setFetchColumns(false);
            if (!startArtifact) {
                re.setFilters(getKeyPredicates(navigationSegments.get(0).getKeyPredicates()));
            } else {
                re.setFilters(getKeyPredicates(globalUriInfo.getKeyPredicates()));
            }

            // Add Authorization filter
            try {
                addAuthorizationFilterNavigatedSegment(entitySet, re);
            } catch (UnauthorizedException e) {
                throw new ODataApplicationException(e.getMessage(), LocaleUtil.getLocaleforException(),
                        HttpStatusCodes.FORBIDDEN);
            }
            if (navigationSegments != null) {
                List<NavigationSegment> subNavigationSegments = startArtifact ? navigationSegments
                        : navigationSegments.subList(1, navigationSegments.size());
                if (subNavigationSegments != null && !subNavigationSegments.isEmpty()) {
                    re.setEntityNavigated(this.prepareEntityInfo(subNavigationSegments.get(0).getEntitySet(),
                            parentAssociation + ALIASHLP + entityName, subNavigationSegments, navigationTree.trim()
                                    + " " + subNavigationSegments.get(0).getNavigationProperty().getName()));
                }
            }
        }

        return re;
    }


    private void initAuth(EdmEntitySet entitySet, ReadEntityInfo re) throws ODataException {
        if (!targetWhereClause.equals("")) {
            addAuthorizationFilter(re, getEntityTypeMap(entitySet), targetWhereClause,false);
        } else if (globalUriInfo.getNavigationSegments().isEmpty()) {
            addAuthorizationFilter(re, getEntityTypeMap(entitySet), AuthorizationService.getWhereCondition(),false);
        }
    }

    private String initAuthNavigationSegment(EdmEntitySet entitySet) throws ODataException {
        String entityFQN = entitySet.getEntityType().toString();
        if (AuthorizationService.getWhereCondition() != null && !AuthorizationService.getWhereCondition().equals("")
                && targetWhereClause.equals(""))
            targetWhereClause = AuthorizationService.getWhereCondition();
        return entityFQN;
    }

    private void addAuthorizationFilterNavigatedSegment(EdmEntitySet entitySet, ReadEntityInfo re)
            throws ODataException, UnauthorizedException {
        String entityFQN = initAuthNavigationSegment(entitySet);
        if (!AuthorizationService.hasEntityAccess(entityFQN, READ)) {
            logger.error("User does not have access to navigate via this segment");
            throw new UnauthorizedException(ACCESS_DENIED);
        } else {
            addAuthorizationFilter(re, getEntityTypeMap(entitySet), AuthorizationService.getWhereCondition(),false);
        }
    }


    private List<ReadEntityInfo> prepExpandInfoList(List<ArrayList<NavigationPropertySegment>> expandsList,
                                                    String parentAssociation) throws EdmException {
        List<ReadEntityInfo> eInfoList = new ArrayList<ReadEntityInfo>();
        // Prepare a List of readEntitinfo for Multiple Expands on Same Level eg.
        // Person?$expand=apps,notes
        for (ArrayList<NavigationPropertySegment> expands : expandsList) {
            ReadEntityInfo eInfo = prepareExpand(expands, parentAssociation, "", eInfoList);
            if (eInfo != null) {
                eInfoList.add(eInfo);
            }
        }
        return eInfoList;
    }

    private ReadEntityInfo prepareExpand(List<NavigationPropertySegment> expands, String parentAssociation,
                                         String navigationSegmentConcatenated, List<ReadEntityInfo> siblingExpandList) throws EdmException {
        NavigationPropertySegment expand = expands.get(0);
        try {
            checkExpandAuthorization(expand);
        } catch (UnauthorizedException e) { 
        	logger.error(e.getMessage());
        	MessageReference msgref = MessageReference.create(EdmException.class, e.getMessage());
        	throw new EdmException(msgref, new ODataApplicationException(e.getMessage(),
        			LocaleUtil.getLocaleforException(), HttpStatusCodes.FORBIDDEN));
        	}
        // Ignore draft admin expand for draft root entity
        if (this.isDraftRoot && DraftUtilsV2.DRAFTS_ADMIN_TABLE_NAME.equals(expand.getNavigationProperty().getName())) {
            return null;
        }

        // Determine if ReadEntityInfo Already Exists for this Expand . Use Case ::
        // Customer?$expand=orders,orders/items,orders/items/inneritems
        ReadEntityInfo eInfo = QueryHelperUtility.determineReadEntityInfoForExpand(expand.getNavigationProperty().getName(),
                siblingExpandList);

        eInfo.setEntityName(expand.getNavigationProperty().getName());

        eInfo.setParententityName(parentAssociation);
        eInfo.setFetchColumns(true);
        if (expand.getNavigationProperty().getMultiplicity() == EdmMultiplicity.MANY) {
            eInfo.setIscollection(true);
        } else {
            eInfo.setIscollection(false);
        }

        if (orderByFromNavigation.size() > 0) {
            ReadEntityInfo orderByEntity = orderByFromNavigation.get(expand.getNavigationProperty().getName());
            if (orderByEntity != null)
                QueryHelperUtility.addOrderByProperties(orderByEntity, eInfo);
        }
        this.orderByPropsFromCSON = CSNUtil.getOrderByProperties(serviceName, expand.getTargetEntitySet().getName());
        eInfo.setColumns(resolvePropertiesToColumns(eInfo.getEntityName(), expand.getTargetEntitySet().getEntityType(),
                parentAssociation, navigationSegmentConcatenated + expand.getNavigationProperty().getName() + "/"));
        // Add OrderBy if it is set in the CSON
        QueryHelperUtility.addOrderBypropertiesFromCSON(orderByPropsFromCSON, eInfo, expand.getTargetEntitySet().getEntityType());
        // Add Default OrderBy on keys so the cartesian product comes in reserved order
        addOrderByForKeyPredicates(eInfo, expand.getTargetEntitySet().getEntityType());

        // Keep Recursing incase of MultiLevel Expand eg.
        // Person?$expand=apps/departments
        List<NavigationPropertySegment> subExpandsList = expands.subList(1, expands.size());
        if (subExpandsList != null && !subExpandsList.isEmpty()) {
            ReadEntityInfo innerRE = prepareExpand(subExpandsList,
                    parentAssociation + ALIASHLP + expand.getNavigationProperty().getName(),
                    navigationSegmentConcatenated + expand.getNavigationProperty().getName() + "/",
                    eInfo.getEntitiesExpanded());
            if (innerRE != null) {
                eInfo.getEntitiesExpanded().add(innerRE);
            }
        }
        return eInfo;
    }

    private void checkExpandAuthorization(NavigationPropertySegment expand) throws EdmException, UnauthorizedException {
        String entityFQN = expand.getTargetEntitySet().getEntityType().toString();
        if (!AuthorizationService.hasExpandedEntityAccess(entityFQN, READ)) {
            StringBuilder errorMsg = new StringBuilder("Expanding via ");
            errorMsg.append(entityFQN);
            errorMsg.append(" not allowed for this user");
            logger.debug(errorMsg.toString());
            throw new UnauthorizedException(ACCESS_DENIED);
        }
    }

    private void handleSearchQueryOption(Map<String, String> customQueryOptions, ReadEntityInfo readEntityInfo,
                                         String searchedEntity, boolean hasNavigation, String navigationTree) throws ODataException {

        String searchTerm = null;
        if (customQueryOptions != null && !customQueryOptions.isEmpty()) {
            readEntityInfo.setCustomQueryOptions(customQueryOptions);
            for (String key : customQueryOptions.keySet()) {
                if (key.toLowerCase().equals("search")) {
                    searchTerm = customQueryOptions.get(key);
                    break;
                }
            }
        }
        if (searchTerm != null) {
            searchTerm = searchTerm.replaceAll("^\"|\"$", "");

            if (searchTerm.contains("*") || searchTerm.contains("?")) {
                throw new CDSRuntimeException(MessageKeys.FUZZY_SEARCH_NOT_SUPPORTED, "Fuzzy Search not supported yet.",
                        Locale.ENGLISH, HttpStatusCodes.NOT_IMPLEMENTED);
            }

            String searchText = " LIKE UPPER('%" + searchTerm + "%')";
            StringBuffer searchQuery = new StringBuffer();
            List<String> searchableProperties = CSNUtil.getSearchableProperties(serviceName, searchedEntity);
            String queryStartStr = "";
            String queryEndStr = "";

            if (searchableProperties != null) {
                searchableProperties = searchableProperties.stream().map(prop -> convertToUpperCaseIfRequired(prop))
                        .collect(Collectors.toList());
                if (hasNavigation) {
                    // Re-look SQL during quoted mode
                    queryStartStr = "UPPER(ROOT_ENTITY_SELECT_ALIAS."
                            + convertToUpperCaseIfRequired(this.getSearchableNavigatedEntities(navigationTree) + ".");
                    queryEndStr = ")";
                } else {
                    if (SQLMapping.isPlainSqlMapping()) {
                        queryStartStr = "UPPER(";
                        queryEndStr = ")";
                    } else {
                        queryStartStr = "UPPER(\"";
                        queryEndStr = "\")";
                    }
                }

                for (Iterator<String> iterator = searchableProperties.iterator(); iterator.hasNext(); ) {

                    searchQuery.append(queryStartStr + iterator.next() + queryEndStr + searchText);
                    if (iterator.hasNext()) {
                        searchQuery.append(" OR ");
                    }

                }
                // Internal Incident:1970092900 CXS OData v2 AutoExp: $filter is Ignored if
                // Custom Search is Present
                if (searchableProperties.size() > 1) {
                    searchQuery.insert(0, "(");
                    searchQuery.append(")");
                }
                readEntityInfo.getFilters().add(new Filter(searchQuery.toString()));
            } else {
                throw new CDSRuntimeException(MessageKeys.SEARCH_NOT_IMPLEMENTED,
                        "Search not implemented for \"" + searchedEntity + "\" EntitySet.",
                        LocaleUtil.getLocaleforException(), HttpStatusCodes.NOT_IMPLEMENTED);
            }

        }

    }

    private String getSearchableNavigatedEntities(String navigationTree) {
        String deepNavEntityNames = "";
        StringTokenizer tokenizer = new StringTokenizer(navigationTree, " ");
        while (tokenizer.hasMoreTokens()) {
            if ("".equals(deepNavEntityNames.trim())) {
                deepNavEntityNames = tokenizer.nextToken();
                continue;
            }
            deepNavEntityNames = deepNavEntityNames.trim() + "." + tokenizer.nextToken();
        }
        return deepNavEntityNames;
    }

    private String getAggregationFunctionFromCSN(String element, String entity) throws CDSRuntimeException {

        if (csn == null)
            throw new CDSRuntimeException(MessageKeys.CSN_FILE_NOT_FOUND, "CSN file required but not found.",
                    LocaleUtil.getLocaleforException(), HttpStatusCodes.INTERNAL_SERVER_ERROR);

        JsonNode aggregation = csn.get("definitions").get(serviceName + "." + entity).get("elements").get(element)
                .get("@Aggregation.default");
        if (aggregation != null) {
            return aggregation.get("#").asText();
        }

        return null;
    }

    private List<Column> resolvePropertiesToColumns(String entPrefixName, EdmEntityType entity, String parent,
                                                    String navigationName) throws EdmException {
        List<Column> columns = new ArrayList<Column>();
        if (entity != null && entity.getPropertyNames() != null && entity.getPropertyNames().size() > 0) {
            for (String prop : entity.getPropertyNames()) {
                // Eliminate few columns for draft root entity
                if (this.isDraftRoot && !DraftUtilsV2.isValidPropertyForMainTable(prop)) {
                    continue;
                } else if(this.isDraft == false && DraftUtilsV2.isValidPropertyForMainTable(prop) == false) {
                	continue;
                }
                String etagPropertyName = HANADMLHelperV2.getEtagPropertyName(entity);
                boolean key = false;
                boolean selected = false;
                Column col = QueryHelperUtility.formNewColumnObject(prop, parent + ALIASHLP + entPrefixName);
                if (entity.getKeyPropertyNames().contains(prop)) {
                    key = true;
                    col.setpKey(true);
                    selected = true;
                }
                if (!key) { // Key properties should alaways be selected
                    if (globalSelect.isEmpty() // This means no $select in query
                            || prop.equalsIgnoreCase(etagPropertyName) // This means that this is a concurrency property
                            || globalSelect.contains(navigationName + prop) // This means $select is there and property
                            // has been selected
                            || globalSelect.contains(navigationName)  // This means $select is there and all
                            // associated properties has been selected
                            || isOrderByFromCSONContains(prop) // If there a default orderby present in CSON, that property is also added
                            //  in the entityInfo so that while adding the orderby clause, proper alias name can be added
                            || QueryHelperUtility.checkOrderByClause(prop, navigationName, globalOrderByStringFromURI)) {
                        selected = true;
                    }
                }
                col.setType(ColumnType.Primitive);
                col.setColumnDataType(entity.getProperty(prop).getType().getName());
                if (selected) {
                    columns.add(col);
                }
            }
        }
        return columns;
    }

    private boolean isOrderByFromCSONContains(String property) {
        for (Map<String, String> orderByprop : orderByPropsFromCSON) {
            if (orderByprop.containsKey(property)) {
                return true;
            }
        }
        return false;
    }


    private void addOrderByForKeyPredicates(ReadEntityInfo eInfo, EdmEntityType entity) throws EdmException {

        for (String keyP : entity.getKeyPropertyNames()) {
            // Ignore isActiveEntity key for draft root entity
            if (this.isDraftRoot && DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY.equals(keyP)) {
                continue;
            } else if(this.isDraft == false && DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY.equals(keyP)) {
            	continue;
            }
            if (!eInfo.getOrderBySeq().contains(keyP)) {
                eInfo.getOrderBySeq().add(keyP);
                OrderBy orderBy = new OrderBy();
                orderBy.setDescending(false);
                orderBy.setPropertyName(keyP);
                orderBy.setAliasName(getAliasName(eInfo.getColumns(), keyP));
                eInfo.getOrderby().put(keyP, orderBy);
            }
        }

    }

    private List<Filter> getKeyPredicates(List<KeyPredicate> keyPredicate) throws EdmException {
        List<Filter> filters = new ArrayList<Filter>();
        Filter filter = null;
        for (KeyPredicate key : keyPredicate) {
            // Ignore IsActiveEntity for draft root entity
            if (this.isDraftRoot && key.getProperty().getName().equals(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY)) {
                continue;
            } else if(this.isDraft == false && key.getProperty().getName().equals(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY) == true) {
            	continue;
            }
            filter = new Filter();
            String keyName = convertToUpperCaseIfRequired(key.getProperty().getName());
            String pathPrefixedKeyName = "ROOT_ENTITY_SELECT_ALIAS" + "." + "PATH_FROM_ROOT."
                    + keyName.replace("/", ".");
            EdmSimpleType keyType = (EdmSimpleType) key.getProperty().getType();
            String value = QueryHelperUtility.getValueOfProperty(key, keyType);
            String expression = pathPrefixedKeyName + " = " + value;
            filter.setExpression(expression);
            filters.add(filter);
        }
        return filters;

    }

    public String prepeareFilter(UriInfo uriInfo) throws ODataException {
        // Second parameter is draft root or draft entries either one gets triggered
        return uriInfo.getFilter()
                .accept(new CDSExpressionVisitor(uriInfo.getTargetEntitySet(), (this.isDraftRoot || this.isDraft)))
                .toString();
    }

    private void modifyEntityInfoForAggregation(ReadEntityInfo eInfo, List<SelectItem> selectItems,
                                                EdmEntityType entityType, String viewName, FilterExpression filterExpression)
            throws CDSRuntimeException, EdmException {

        String AGGREGATE_ENTITY_KEY = "ID__";
        List<String> measures = new ArrayList<String>();
        List<String> groupingProperties = new ArrayList<String>();

        List<EdmProperty> properties = selectItems == null || selectItems.isEmpty()
                ? entityType.getPropertyNames().stream().filter(p -> !p.equals(AGGREGATE_ENTITY_KEY)).map(p -> {
            try {
                return (EdmProperty) entityType.getProperty(p);
            } catch (EdmException e) {
                return null;
            }
        }).collect(Collectors.toList())
                : selectItems.stream()
                .filter(s -> s.getNavigationPropertySegments() == null || s.getNavigationPropertySegments().isEmpty()).map(s -> s.getProperty()).collect(Collectors.toList());
        
        if(eInfo.getSelectOption() != null && !eInfo.getSelectOption().isEmpty()) {
        	for(String selectProperty:eInfo.getSelectOption()) {
        		EdmProperty property = (EdmProperty) entityType.getProperty(selectProperty);
        		if(!properties.contains(property))
        			properties.add(property);
        	}
        }
        
        for (EdmProperty prop : properties) {
            EdmAnnotationAttribute aggregationRole = prop.getAnnotations().getAnnotationAttribute("aggregation-role",
                    sapNamespace);
            if (aggregationRole != null && aggregationRole.getText().equals("measure")) {
                measures.add(prop.getName());
            } else if (!prop.getName().equals(AGGREGATE_ENTITY_KEY)) {
                groupingProperties.add(prop.getName());
            }
        }

        eInfo.setGroupBy(groupingProperties);
        eInfo.setColumns(
                groupingProperties.stream().map(p -> new Column(p).setDbaliasName(p)).collect(Collectors.toList()));
        if (filterExpression != null) {
            String filterExpressionString = filterExpression.getExpressionString();
            for (String s : measures) {
                String measureRegex = ".*\\b" + s + "\\b.*";
                if (filterExpressionString.matches(measureRegex)) {
                    throw new CDSRuntimeException(MessageKeys.FILTER_ON_MEASURE, "$filter cannot be applied on measure",
                            Locale.ENGLISH, HttpStatusCodes.BAD_REQUEST);
                }
            }
        }

        for (String m : measures) {
        	String entityTypeName = entityType.getName();
        	if(eInfo.isParameterisedView() && entityTypeName.endsWith("Type")) {
        		entityTypeName = entityTypeName.substring(0, entityTypeName.lastIndexOf("Type"));
        	}
        	String columnName;

        	String aggregationFunction = getAggregationFunctionFromCSN(m, entityTypeName);
        	if("COUNT_DISTINCT".equals(aggregationFunction)) {
        		String refElement = convertToUpperCaseIfRequired(CSNUtil.getAggregationRefElement(serviceName, m, entityTypeName));
        		if(refElement != null)
        			columnName = "COUNT" + "( DISTINCT " + refElement + ")";
        		else
        			columnName = "COUNT" + "( DISTINCT " + m + ")";
        	}else {
        		columnName = aggregationFunction + "(" + m + ")";
        	}
        	eInfo.getColumns().add(new Column(columnName).setDbaliasName(m));
        }

    }

    private void setOrderByInReadEntityInfo(OrderByExpression orderByExpression, ReadEntityInfo eInfo)
            throws CDSRuntimeException {
        List<OrderExpression> orderExpressions = orderByExpression.getOrders();
        for (OrderExpression orderExpression : orderExpressions) {
            // Get the OrderBy Expression Only For the concerned Level which is only the
            // parent Level
            String orderByon = orderExpression.getExpression().getUriLiteral();
            // Eliminate order by if it contains HasActiveEntity,HasDraftEntity,IsActiveEntity for draft root entity
            if (this.isDraftRoot && !DraftUtilsV2.isValidPropertyForMainTable(orderByon)) {
                continue;
            } else if(this.isDraft == false && DraftUtilsV2.isValidPropertyForMainTable(orderByon) == false) {
            	continue;
            }
            if (orderByon.contains("/")) {
                continue;
            }

            eInfo.getOrderBySeq().add(orderByon);
            OrderBy oBy = new OrderBy();
            oBy.setPropertyName(orderByon);
            oBy.setDescending(orderExpression.getSortOrder().name() != "desc" ? false : true);
            oBy.setAliasName(getAliasName(eInfo.getColumns(), orderByon));
            eInfo.getOrderby().put(orderByon, oBy);
        }
    }

    private String getAliasName(List<Column> columns, String orderByName) {
        for (Column column : columns) {
            if (column.getName().equals(orderByName))
                return column.getDbaliasName();
        }
        return orderByName;
    }

    @Override
    public String getSchema() {
        return schema;
    }

    @Override
    public ReadEntityInfo getEntityInfo() {
        return eInfo;
    }

    @Override
    public Long limitValue() {
        return topValue;
    }

    @Override
    public Long offsetValue() {
        return skipValue;
    }

    public void setTopValue(Long topValue) {
        this.topValue = topValue;
    }

    public void setSkipValue(Long skipValue) {
        this.skipValue = skipValue;
    }
    @Override
    public boolean isAggregateEntity() {
        return isAggregateEntity;
    }

    public void setAggregateEntity(boolean isAggregateEntity) {
        this.isAggregateEntity = isAggregateEntity;
    }

    public boolean isSqlAggregated() {
        return isSqlAggregated;
    }

    public void setSqlAggregated(boolean isSqlAggregated) {
        this.isSqlAggregated = isSqlAggregated;
    }

    private void prepareGlobalSelectList() throws EdmException {
        // Incase $select parameter is not passed from URI, Do not modify the
        // globalSelect
        if (globalUriInfo.getSelect().size() != 0) {
            if (selectExtension != null && selectExtension.size() > 0) {
                for (String selectedItem : selectExtension) {
                    globalSelect.add(selectedItem);
                }
            } else {
                for (SelectItem sel : globalUriInfo.getSelect()) {
                    StringBuffer selectedItem = new StringBuffer("");
                    for (NavigationPropertySegment npg : sel.getNavigationPropertySegments()) {
                        selectedItem.append(npg.getNavigationProperty().getName() + "/");
                    }
                    if (sel.getProperty() != null) {
                        selectedItem.append(sel.getProperty().getName());
                    }
                    globalSelect.add(selectedItem.toString());
                }
            }
        }

    }


    /**
     * Prepare EntityInfo for Views with Parameters
     *
     * @param uriInfo
     * @param countStar
     * @throws ODataException
     */
    private void prepareEntityInfoForParameterViews(UriInfo uriInfo, boolean countStar) throws ODataException {

        logger.debug("Entering QueryHelperV2 >> {prepareEntityInfoForParameterViews}");
        try {
            EdmEntitySet targetEntitySet = uriInfo.getTargetEntitySet();
            List<KeyPredicate> keyPredicate = uriInfo.getKeyPredicates();
            FilterExpression filterExpression = uriInfo.getFilter();
            List<SelectItem> selectItems = uriInfo.getSelect();
            String viewName = uriInfo.getStartEntitySet().getName();
            ReadEntityInfo eInfo = new ReadEntityInfo();

            eInfo.setEntityName(viewName);
            eInfo.setParententityName(viewName);
            eInfo.setIscollection(true);
            eInfo.setFetchColumns(true);
            eInfo.setServiceName(serviceName);
            for (int i = 0; i < keyPredicate.size(); i++) {
                String paramValue = keyPredicate.get(i).getLiteral();
                EdmType edmType = keyPredicate.get(i).getProperty().getType();
                eInfo.getParameters().put(keyPredicate.get(i).getProperty().getName(), QueryHelperUtility.convertParam(eInfo, edmType, keyPredicate.get(i).getProperty().getName(), paramValue));
            }
            eInfo.setIsParameterisedView(true);
            EdmEntityType targetType = targetEntitySet.getEntityType();
            EdmAnnotationAttribute annotationAttribute = targetType.getAnnotations().getAnnotationAttribute("semantics", sapNamespace);
            if (annotationAttribute != null && annotationAttribute.getText().equals("aggregate")) {
            	this.isAggregateEntity = true;
                modifyEntityInfoForAggregation(eInfo, selectItems, targetType, viewName, filterExpression);
            }
            else {
                eInfo.setColumns(resolvePropertiesToColumns(viewName, targetEntitySet.getEntityType(), viewName, ""));
                addOrderByForKeyPredicates(eInfo, targetEntitySet.getEntityType());
            }
            //Handle Filter

            if (filterExpression != null) {
                String filterExp = prepeareFilter(uriInfo);
                if (filterExp != null) {
                    eInfo.getFilters().add(new Filter(filterExp));
                }
            }
            eInfo.setCountStar(countStar);
            // for order by
            if (uriInfo.getOrderBy() != null) {
                setOrderByInReadEntityInfo(uriInfo.getOrderBy(), eInfo);
            }
            this.eInfo = eInfo;
        } catch (EdmException e ) {
            logger.error(e.getMessage(), e);
            throw new CDSRuntimeException(CDSRuntimeException.MessageKeys.INTERNAL_ERROR, e.getMessage(), Locale.ENGLISH,
                    HttpStatusCodes.INTERNAL_SERVER_ERROR, e);
        }
    }

    private boolean isParameterEntity(EdmEntitySet startEntitySet) throws EdmException {
        return startEntitySet.getEntityType().getName().endsWith(PARAMETER_ENTITY_SUFFIX) && startEntitySet.getEntityType().getProperty(PARAMETER_NAVIGATION_PROPERTY) != null;
    }


    private void updateOrderBy(OrderByExpression orderByExpression, ReadEntityInfo re, EdmEntitySet entitySet) throws EdmException, CDSRuntimeException {
        List<OrderExpression> orderExpressions = orderByExpression.getOrders();
        boolean inExpandPresent = false;
        List<ReadEntityInfo> orderbyEInfoList = new ArrayList<ReadEntityInfo>();
        for (OrderExpression orderExpression : orderExpressions) {
            ReadEntityInfo re1 = null;
            String lastnavigatedPropName = null;
            String firstnavigatedPropName = null;
            // Get the OrderBy Expression Only For the concerned Level which is only the parent Level
            String orderByon = orderExpression.getExpression().getUriLiteral();
            // Eliminate order by if it contains HasActiveEntity,HasDraftEntity,IsActiveEntity for draft root entity
            if (this.isDraftRoot && !DraftUtilsV2.isValidPropertyForMainTable(orderByon)) {
                continue;
            } else if(this.isDraft == false && DraftUtilsV2.isValidPropertyForMainTable(orderByon) == false) {
            	continue;
            }
            if (!orderByon.contains("/")) // This flow is only for when orderby is having navigation/propetyName
                return;
            
            if (ExpressionKind.MEMBER.equals(orderExpression.getExpression().getKind())) {
                EdmProperty edmProperty = null;
                MemberExpression memberExpression = (MemberExpression) orderExpression.getExpression();
                CommonExpression currentExpression = memberExpression;
                if (currentExpression != null) {
                    if (currentExpression != null
                            && currentExpression.getKind() == ExpressionKind.MEMBER) {
                        MemberExpression currentMember = (MemberExpression) currentExpression;
                        PropertyExpression p1 = (PropertyExpression) currentMember.getProperty();
                        edmProperty = (EdmProperty) p1.getEdmProperty();
                        orderByon = edmProperty.getName();
                    }
                }
                
                //Stack is used to fetch the all the entity names in orderby query except the last one 
                currentExpression = memberExpression.getPath();
                Stack<String> navigatedPropNames_stack = new Stack<String>();
                while (currentExpression != null) {
                	if(currentExpression.getKind() == ExpressionKind.MEMBER) 
                		currentExpression = ((MemberExpression) currentExpression).getPath();
                	else 
                		break;
                    if (currentExpression != null) {
                    	if(currentExpression.getKind() == ExpressionKind.MEMBER) {
                    		MemberExpression currentMember = (MemberExpression) currentExpression;
                    		navigatedPropNames_stack.push(currentMember.getProperty().getUriLiteral());
                    	}else if(currentExpression.getKind() == ExpressionKind.PROPERTY) {
                    		navigatedPropNames_stack.push(currentExpression.getUriLiteral());
                    		break;
                    	}
                    }
                }
                
                //To fetch the last entity name in orderby query
                if (memberExpression != null && memberExpression.getPath() != null) {
                	CommonExpression currentexpression = memberExpression.getPath();
                	if(currentexpression.getKind() == ExpressionKind.MEMBER) {
                		MemberExpression currentMember = (MemberExpression) currentexpression;
                    	lastnavigatedPropName = currentMember.getProperty().getUriLiteral();
                	}
                	else if(currentexpression.getKind() == ExpressionKind.PROPERTY) {
                		lastnavigatedPropName = currentexpression.getUriLiteral();
                	}
                }
                
                HashMap<String, EdmEntitySet> entitysets  = new HashMap<>();
                for(EdmEntitySet edm : entitySet.getEntityContainer().getEntitySets()) {
                	entitysets.put(edm.getName(), edm);
                }
                
                String parentAssociation = null;
                if("".equals(re.getParententityName())) {
                	parentAssociation = ALIASHLP + re.getEntityName();
                }
                else {
                	parentAssociation = re.getParententityName()+ALIASHLP + re.getEntityName();	
                }
                // Ignore draft admin expand for draft root entity
                if (this.isDraftRoot && DraftUtilsV2.DRAFTS_ADMIN_TABLE_NAME.equals(lastnavigatedPropName)) {
                    return;
                }
                String navigatedPropName = entitySet.getName();
                if(!navigatedPropNames_stack.empty()) {
                	firstnavigatedPropName = navigatedPropNames_stack.peek();
                }
                else {
                	firstnavigatedPropName = lastnavigatedPropName;
                }
                
                //to check the multiplicity and to form the parent association
            	while (!navigatedPropNames_stack.isEmpty()) {
            		EdmEntitySet entityset = entitysets.get(navigatedPropName);
            		EdmEntityTypeImplProv entityType = (EdmEntityTypeImplProv) entitySet.getEntityType();
                    EdmNavigationPropertyImplProv edmNavigationProp = (EdmNavigationPropertyImplProv) entityType.getProperty(navigatedPropNames_stack.peek());
                    QueryHelperUtility.checkEdmMultiplicity(edmNavigationProp);
                    navigatedPropName = edmNavigationProp.getToRole();
            		parentAssociation = parentAssociation + ALIASHLP + navigatedPropNames_stack.pop();
            	}
            	EdmEntitySet entityset = entitysets.get(navigatedPropName);
            	EdmEntityTypeImplProv entityType = (EdmEntityTypeImplProv) entityset.getEntityType();
                EdmNavigationPropertyImplProv edmNavigationProp = (EdmNavigationPropertyImplProv) entityType.getProperty(lastnavigatedPropName);
                QueryHelperUtility.checkEdmMultiplicity(edmNavigationProp);
                
                currentExpression = memberExpression.getPath();
                if (currentExpression != null)
                    re1 = QueryHelperUtility.prepareOrderBy(memberExpression.getPath(), orderByon, orderExpression.getSortOrder().name(), parentAssociation,
                            orderbyEInfoList, re, globalSelect, globalOrderByStringFromURI, null);
            }
            if (null != re1) {
                orderbyEInfoList.add(re1);
            }
            if (globalUriInfo.getExpand() != null && !globalUriInfo.getExpand().isEmpty()) {
                // Check this navigation property in $expand
                inExpandPresent = QueryHelperUtility.checkNavPropinExpand(globalUriInfo.getExpand(), firstnavigatedPropName);
                if (null != re1 && inExpandPresent) {
                    orderByFromNavigation.put(firstnavigatedPropName, re1);
                }
            }
        }
        if (orderbyEInfoList.size() > 0 && orderbyEInfoList.size() != orderByFromNavigation.size()) {
            re.getEntitiesExpanded().addAll(orderbyEInfoList);
        }
    }
    
	@Override
	public List<String> getMeasuresPropertiesList() {
		return null;
	}
}