/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.rt.cds;

import static com.sap.cloud.sdk.service.prov.api.internal.SQLMapping.isPlainSqlMapping;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import java.util.UUID;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;
import com.sap.cloud.sdk.service.prov.api.util.RequestProcessingHelper;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Column;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.ReadEntityInfo;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.StableId;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException;

public class ResultSetProcessor {	
	final static Logger logger = LoggerFactory.getLogger(ResultSetProcessor.class);
	
	private static final String UTC = "UTC";
	private final static String AGGREGATE_ENTITY_KEY = "ID__";
	
	public static List<Map<String, Object>> toEntityCollection(ResultSet rs, ReadEntityInfo re)
			throws SQLException, CDSRuntimeException {
		List<Map<String, Object>> outerMap=new ArrayList<Map<String,Object>>();
		Map<String, Object> singleResultSetRow = null;
		Map<String, Object> prevSingleResultSetRow = null;
		boolean isAggregateEntity = isAggregateEntity(re) || CSNUtil.isAggregatedEntity(re.getServiceName(), re.getEntityName());
		while(rs.next()) {
			singleResultSetRow=extractResultSetRow(rs);
			// Start Processing from the lastNavigatedEntityInfo, No previous columns will be fetched
			//If It is Agrregate Entity No need to Process for possible Expand
			Map<String, Object> entry;
			if(!isAggregateEntity) {
				entry=prepEntity(determineLastNavigatedEntityInfo(re), outerMap, singleResultSetRow, prevSingleResultSetRow);
			} else {
				entry=singleResultSetRow;
			} 
			if(entry!=null) {
				outerMap.add(entry);
			} else {
				continue;
			}
			prevSingleResultSetRow=singleResultSetRow;
		}
		if(!outerMap.isEmpty() && isAggregateEntity) {
			if (isPlainSqlMapping() || RequestProcessingHelper.getDatasourceProvider() == com.sap.cloud.sdk.service.prov.rt.datasource.factory.DatasourceType.HANA) {
				List<String> dbAliases = re.getColumns().stream().map(c -> c.getDbaliasName())
						.collect(Collectors.toList());
				for (Map<String, Object> aggregatedData : outerMap) {
					for (String dbAlias : dbAliases) {
						//Case rearrangement done because Olingo expects cases according to edmx. We here make use of aliases saved earlier while creating entityInfo.
						aggregatedData.put(dbAlias, aggregatedData.remove(dbAlias.toUpperCase()));
					}
				}
			}
			if(!outerMap.get(0).containsKey(AGGREGATE_ENTITY_KEY)) {
				for(Map<String, Object> m : outerMap) {
					m.put(AGGREGATE_ENTITY_KEY, (new StableId(m, re)).toString());
				}
			}
		}
		return outerMap;
	}
	
	private static Map<String,Object> prepEntity(ReadEntityInfo eInfo,List<Map<String, Object>> collection,Map<String, Object> singleResultSetRow,Map<String, Object> prevSingleResultSetRow) throws SQLException {
		// Extract already Created Entry or new Entry From Existing Collection
		HashMap<String,Object> entity=(HashMap<String, Object>) extractEntityFromCollection(collection, eInfo ,singleResultSetRow,prevSingleResultSetRow);
		//Check if Primary Types Has Values
		/*
		 * This will return null during recursive expand processing (Case where parent entity has no children to expand to)
		 */
		List<Column> keyColumns=eInfo.getKeyColumns(eInfo.getColumns());
		if(keyColumns!=null&&!keyColumns.isEmpty()){
			for(Column col:keyColumns){
				if(singleResultSetRow.get(col.getDbaliasName())==null){
						return null;
				}
			}
		}
		for(int i=0;i<eInfo.getColumns().size();i++){
			String propertyName=eInfo.getColumns().get(i).getName();
			if(entity.get(propertyName)==null){
				entity.put(propertyName, getProperty(eInfo.getColumns().get(i), singleResultSetRow));
			}
		}
		//For Expand
		if (!eInfo.getEntitiesExpanded().isEmpty()) {
			for (int i = 0; i < eInfo.getEntitiesExpanded().size(); i++) {
				// For each entry a Map is created against navigation propertyName
				//First Check if it was already created in previous iterations
				Map<String,Object> entityLink;	       // This is for holding to-one expand
				List<Map<String,Object>> entityLinks;  //This is for holding to-many expand
				//First case ::  Expand is of type to-one
				
				if (!eInfo.getEntitiesExpanded().get(i).isIscollection()) {
					List<Map<String,Object>> dummyListHolder=new ArrayList<Map<String,Object>>();
					if(entity.get(eInfo.getEntitiesExpanded().get(i).getEntityName())!=null) {
						entityLink=(Map<String, Object>) entity.get(eInfo.getEntitiesExpanded().get(i).getEntityName());
						dummyListHolder.add(entityLink);
					}else {
						entityLink=new HashMap<String,Object>();
					}
					// Hold Expanded Entity on current Resultset
					Map<String,Object> expandedEntity=prepEntity(eInfo.getEntitiesExpanded().get(i), dummyListHolder, singleResultSetRow, prevSingleResultSetRow);
					if(expandedEntity!=null) {
						entity.put(eInfo.getEntitiesExpanded().get(i).getEntityName(), expandedEntity);
					}
				}
				//Second case ::  Expand is of type to-many
				else {
					if(entity.get(eInfo.getEntitiesExpanded().get(i).getEntityName())!=null) {
						entityLinks=(List<Map<String,Object>>) entity.get(eInfo.getEntitiesExpanded().get(i).getEntityName());
					}else {
						entityLinks=new ArrayList<Map<String,Object>>();
					}
					
					//	Recursively Prepare one entry of underlying entity List
					// Hold Expanded Entity on current Resultset
					HashMap<String,Object> expandedEntity=(HashMap<String, Object>) prepEntity(eInfo.getEntitiesExpanded().get(i), entityLinks, singleResultSetRow, prevSingleResultSetRow);
					if(expandedEntity!=null) {
						// Ensure you add an uniquely formed entity to avoid repititions which can occur due to the cartesian product
						if(!entityLinks.contains(expandedEntity)) {
							entityLinks.add(expandedEntity);
						}
					}
					entity.put(eInfo.getEntitiesExpanded().get(i).getEntityName(), entityLinks);
				}
			}
		}
		return entity;
	}
	
	
	
	
	private static Map<String,Object> extractEntityFromCollection(List<Map<String, Object>> ec, ReadEntityInfo eInfo ,Map<String, Object> singleResultSetRow,Map<String, Object> prevSingleResultSetRow) throws SQLException{
		if (ec != null && ec.size() > 0) {
		    
			Map<String,Object> prevEntry = ec.get(ec.size() - 1);// Check only the previous processed result. Returned ResultSet are alaways ordered by design of the query formed
			List<Column> keyColumns=eInfo.getKeyColumns(eInfo.getColumns());
			boolean entityFound = keyColumns != null && !keyColumns.isEmpty();
			if(entityFound){
				for(Column col:keyColumns){
					if(!singleResultSetRow.get(col.getDbaliasName()).equals(prevSingleResultSetRow.get(col.getDbaliasName()))){
						entityFound=false;
					}
				}
			}
			if (entityFound) {
				//Previous Entry is not necessarily from Previous ResultSet.As repeated results from Cartesian product are not added. Now we cross check if current entry and previous entry from Collection actually have similar keys
				boolean crosscheck=true;
				for(Column col:keyColumns){
					if(!prevEntry.get(col.getName()).equals(   getProperty(col, singleResultSetRow)        )) {
						crosscheck=false;
					}
				}
				//only if cross check passes we remove previous entry
				if(crosscheck) {
					ec.remove(prevEntry);
					return prevEntry;
				}
				// Else let it form a new object which never gets added
			}
		}
		return new HashMap<String,Object>();
	}
	
	
	
	private static Object getProperty(Column column,Map<String, Object> singleResultSetRow)throws SQLException {
		String propertyName = column.getAliasName() != null ? column.getAliasName() : column.getName();
		String aliasName = null;
		if (column.getDbaliasName() != null) {
			aliasName = column.getDbaliasName();
		} else {
			aliasName = isPlainSqlMapping() ? propertyName.toUpperCase() : propertyName;
		}
		Object val=singleResultSetRow.get(aliasName);
		if(val!=null&&column.getColumnDataType()!=null) {
			if(column.getColumnDataType().equalsIgnoreCase("Guid")) {
				val=UUID.fromString((String) val);
			}else if(column.getColumnDataType().equalsIgnoreCase("Double")&& val instanceof String) {
				val=Double.parseDouble((String) val);
			}else if (column.getColumnDataType().equalsIgnoreCase("Binary")&& val instanceof String) {
				val=val.toString().getBytes();
			}
		}
		return val;
	}
	
	static Map<String, Object> extractResultSetRow(ResultSet rs)throws SQLException {
		HashMap<String, Object> row = new LinkedHashMap<String, Object>();
		ResultSetMetaData meta=rs.getMetaData();
		int count=meta.getColumnCount();
		for (int i = 0; i < count; i++) {
			int columnNumber = i + 1;
			int columnType = meta.getColumnType(columnNumber);
			String columnName = meta.getColumnLabel(columnNumber);
			if (columnType == Types.BLOB) {
				ByteArrayOutputStream byteArray = new ByteArrayOutputStream();
				int reads = 0;
				// adding a null check for handling empty media scenarios
				// rs.getBinaryStream(columnNumber) will be null for empty media
				try (InputStream inputStream = rs.getBinaryStream(columnNumber);) {
					if(inputStream!=null) {
						reads = inputStream.read();
						while (reads != -1) {
							byteArray.write(reads);
							reads = inputStream.read();
						}
						row.put(columnName, byteArray.toByteArray());
					} else {
						row.put(columnName, null);
					}
				} catch (IOException e) {
					logger.error(e.getMessage(), e);
				}
			} else if (columnType == Types.CLOB || columnType == Types.NCLOB) { //Support for LargeString datatypes.
				row.put(columnName, rs.getString(columnNumber));
			} else if ((columnType == Types.BOOLEAN) || (columnType == Types.BIT) || (columnType == Types.TINYINT)) {
				// CDS Boolean data type mapped to TinyINT
				
				row.put(columnName, rs.getBoolean(columnNumber));
				if(rs.wasNull()) {
					row.put(columnName, null);
				}
				
			} else if (columnType == Types.DATE) {
				Calendar cal = Calendar.getInstance(); 
				cal.setTimeZone(TimeZone.getTimeZone(UTC)); 
				TimeZone.setDefault(TimeZone.getTimeZone("GMT"));
				row.put(columnName, rs.getDate(columnNumber,cal));
			} else if (columnType == Types.TIME) {
				Calendar cal = Calendar.getInstance(); 
				cal.setTimeZone(TimeZone.getTimeZone(UTC)); 
				TimeZone.setDefault(TimeZone.getTimeZone("GMT"));
				row.put(columnName, rs.getTimestamp(columnNumber,cal));
			}
			// TODO : TIME_WITH_TIMEZONE we are not supporting (TBD)
			else if (columnType == Types.TIMESTAMP) {
				Calendar cal = Calendar.getInstance(); 
				cal.setTimeZone(TimeZone.getTimeZone(UTC)); 
				TimeZone.setDefault(TimeZone.getTimeZone("GMT"));
				Timestamp ts = rs.getTimestamp(columnNumber,cal);
				row.put(columnName, ts);
			}
			else {
				row.put(columnName, rs.getObject(columnNumber));
			}
		}
		return row;
	}
	
	private static boolean isAggregateEntity(ReadEntityInfo entityInfo) {
		for (Column c : entityInfo.getColumns()) {
			String name = c.getName();
			if (name.startsWith("SUM(") || name.startsWith("AVG(") || name.startsWith("MIN(") || name.startsWith("MAX(")
					|| name.startsWith("COUNT( DISTINCT")) {
				return true;
			}
		}
		return !entityInfo.getGroupBy().isEmpty(); //If there are no aggregated properties, if there is group by, it still means that its an aggregatedResult.
	}
	
	private static ReadEntityInfo determineLastNavigatedEntityInfo(ReadEntityInfo eInfo){
		ReadEntityInfo lastNavEinfo=eInfo;
		while(lastNavEinfo.getEntityNavigated()!=null){
			lastNavEinfo=lastNavEinfo.getEntityNavigated();
		}
		return lastNavEinfo;
	}
		
		
}
