package com.sap.cloud.sdk.service.prov.v2.rt.cds.util;

import static com.sap.cloud.sdk.service.prov.api.internal.SQLMapping.convertToUpperCaseIfRequired;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmEntityType;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmMultiplicity;
import org.apache.olingo.odata2.api.edm.EdmSimpleType;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeKind;
import org.apache.olingo.odata2.api.edm.EdmType;
import org.apache.olingo.odata2.api.uri.KeyPredicate;
import org.apache.olingo.odata2.api.uri.NavigationPropertySegment;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.expression.CommonExpression;
import org.apache.olingo.odata2.api.uri.expression.ExpressionKind;
import org.apache.olingo.odata2.api.uri.expression.MemberExpression;
import org.apache.olingo.odata2.core.edm.provider.EdmNavigationPropertyImplProv;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.hana.sql.DateUtils;
import com.sap.cloud.sdk.service.prov.api.filter.BinaryExpressionNode;
import com.sap.cloud.sdk.service.prov.api.internal.CSNUtil;
import com.sap.cloud.sdk.service.prov.api.security.CaseInsensitiveString;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Column;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.ColumnType;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.OrderBy;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.Parameter;
import com.sap.cloud.sdk.service.prov.rt.cds.domain.ReadEntityInfo;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.HANADMLHelperV2;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException;
import com.sap.cloud.sdk.service.prov.v2.rt.cds.exceptions.CDSRuntimeException.MessageKeys;
import com.sap.cloud.sdk.service.prov.v2.rt.util.DraftUtilsV2;
import com.sap.cloud.sdk.service.prov.v2.rt.util.LocaleUtil;

public class QueryHelperUtility {

	private static final String DESC = "desc";
	private static String ALIASHLP = "ZZ";
	final static Logger logger = LoggerFactory.getLogger(QueryHelperUtility.class);
	
	public static void addOrderBypropertiesFromCSON(List<Map<String, String>> orderByProperties, ReadEntityInfo eInfo,
			EdmEntityType entityType) {
		for(Map<String, String> orderByprop:orderByProperties)
			for(String prop:orderByprop.keySet()) {
			if (!eInfo.getOrderBySeq().contains(prop)) {
				eInfo.getOrderBySeq().add(prop);
				String order = orderByprop.get(prop);
				OrderBy orderBy = new OrderBy();
				if(DESC.equals(order)) {
				orderBy.setDescending(true);
				}else {
				orderBy.setDescending(false);
				}
				orderBy.setPropertyName(prop);
				orderBy.setAliasName(getAliasName(eInfo.getColumns(),prop));
				eInfo.getOrderby().put(prop,orderBy);
		}
		}
	}
	
	public static String getAliasName(List<Column> columns, String orderByName) {
		for (Column column : columns) {
			if (column.getName().equals(orderByName)) {
				orderByName = column.getDbaliasName();
				break;
			}
		}
		return orderByName;
	}
	
	public static String getBinaryOperandValue(BinaryExpressionNode node) {
		String operator = null;
		switch (node.getOperator()) {
		// Arithmetic operation
		case "ADD":
			operator = " + ";
			break;
		case "SUB":
			operator = " - ";
			break;
		case "MUL":
			operator = " * ";
			break;
		case "DIV":
			operator = " / ";
			break;
		case "MODULO":
			operator = " MODULO ";
			break;
		// Logical operations
		case "GT":
			operator = " > ";
			break;
		case "LT":
			operator = " < ";
			break;
		case "EQ":
			operator = " = ";
			break;
		case "NE":
			operator = " <> ";
			break;
		case "LE":
			operator = " <= ";
			break;
		case "GE":
			operator = " >= ";
			break;
		case "AND":
			operator = " and ";
			break;
		default:
			operator = " = ";
		}
		return operator;
	}
	
	public static void addOrderByProperties(ReadEntityInfo source, ReadEntityInfo destination) throws EdmException {
		destination.getOrderBySeq().addAll(source.getOrderBySeq());
		destination.getOrderby().putAll(source.getOrderby());
	}
	
	public static boolean checkNavPropinExpand(List<ArrayList<NavigationPropertySegment>> expandsList,String navPropName) 
			throws EdmException{
		boolean isPropPresent = false;
		for(ArrayList<NavigationPropertySegment> navpropSegment:expandsList) {
			if(navPropName.equals(navpropSegment.get(0).getNavigationProperty().getName())) {
				isPropPresent = true;
				break;
			}
		}
		return isPropPresent;
	}
	
	public static ReadEntityInfo prepareExpandForOrderBy(EdmEntityType edmType, String navPropName, String orderByon, String order, String parentAssociation,
			 List<ReadEntityInfo> siblingExpandList, ReadEntityInfo parentEntityInfo, List<String> globalSelect, String globalOrderBy) throws EdmException {
		
				
		// Determine if ReadEntityInfo Already Exists for this Expand . Use Case ::
		// Customer?$expand=orders,orders/items,orders/items/inneritems
		ReadEntityInfo navEntityInfo = determineReadEntityInfoForExpand(navPropName,
				siblingExpandList);
		navEntityInfo.setEntityName(navPropName);
		navEntityInfo.setParententityName(parentAssociation);
		navEntityInfo.setFetchColumns(true);
		navEntityInfo.setIscollection(false);
		navEntityInfo.setColumns(resolvePropertiesToColumnsForNavigatedEntity(navEntityInfo.getEntityName(), edmType,
				parentAssociation, navPropName + "/",orderByon,globalSelect,globalOrderBy));
		if(orderByon != null) {
			navEntityInfo.getOrderBySeq().add(orderByon);
			OrderBy oBy = new OrderBy();
			oBy.setPropertyName(orderByon);
			oBy.setDescending(order != "desc" ? false : true);
			oBy.setAliasName(getAliasName(navEntityInfo.getColumns(), orderByon));
			navEntityInfo.getOrderby().put(orderByon, oBy);
			parentEntityInfo.getOrderByForExpanded().put(navPropName+"/"+orderByon, oBy); // This is added so that while forming the query, orderby with navigation property
			parentEntityInfo.getOrderBySeqForExpanded().add(orderByon);                   // can be considered as the first property 
		}
		
		return navEntityInfo;
	}
	
	public static ReadEntityInfo determineReadEntityInfoForExpand(String entityNavName,
			List<ReadEntityInfo> siblingExpandList) {
		ReadEntityInfo reInfo = null;
		for (ReadEntityInfo re : siblingExpandList) {
			if (re.getEntityName().equals(entityNavName)) {
				reInfo = re;
				siblingExpandList.remove(reInfo);
				return reInfo;
			}
		}
		return new ReadEntityInfo();
	}
	
	public static List<Column> resolvePropertiesToColumnsForNavigatedEntity(String entPrefixName, EdmEntityType entity, String parent,
			String navigationName,String orderByon, List<String> globalSelect, String globalOrderBy) throws EdmException {
		List<Column> columns = new ArrayList<Column>();
		if (entity != null && entity.getPropertyNames() != null && entity.getPropertyNames().size() > 0) {
			for (String prop : entity.getPropertyNames()) {
				
				String etagPropertyName = HANADMLHelperV2.getEtagPropertyName(entity);
				boolean key = false;
				boolean selected = false;
				Column col = formNewColumnObject(prop, parent + ALIASHLP + entPrefixName);
				if (entity.getKeyPropertyNames().contains(prop)) {
					key = true;
					col.setpKey(true);
					selected = true;
				}
				if (!key) { // Key properties should alaways be selected
					if (globalSelect.isEmpty() // This means no $select in query
							|| prop.equalsIgnoreCase(etagPropertyName) // This means that this is a concurrency property
							|| globalSelect.contains(navigationName + prop) // This means $select is there and property
																			// has been selected
							|| globalSelect.contains(navigationName)  // This means $select is there and all
																		// associated properties has been selected
							||  checkOrderByClause(prop,navigationName,globalOrderBy)) {
						selected = true;
					}
				}
				col.setType(ColumnType.Primitive);
				col.setColumnDataType(entity.getProperty(prop).getType().getName());
				if (selected) {
					columns.add(col);
				}
			}
		}
		return columns;
	}
	
	public static Column formNewColumnObject(String propertyName, String aliasPrefix) {
		Column col = new Column();
		col.setName(propertyName);
		col.setDbaliasName(convertToUpperCaseIfRequired(aliasPrefix + ALIASHLP + propertyName));
		return col;
	}
	
	public static boolean isOrderByNavigation(String property,String navigationName,String orderByon, HashMap<String, ReadEntityInfo> orderByFromNavigation) {
		if(navigationName!=""&&orderByFromNavigation.size()>0) {
		navigationName = navigationName.replace("/", "");
		ReadEntityInfo orderByEntity = orderByFromNavigation.get(navigationName);
		if(orderByEntity!=null) {
		OrderBy orderByprop=orderByEntity.getOrderby().get(property);
		if(orderByprop==null)
			return false;
		else
			return true;
		}
		}else if(orderByon!=null&&orderByon.equals(property)) {
			return true;
		}
		return false;
	}

	public static boolean checkOrderByClause(String prop, String navigationName, String globalOrderByStringFromURI) {
		boolean isOrderBypresent = false;
		if(globalOrderByStringFromURI!=null) {
		if(navigationName!=null)
			isOrderBypresent = globalOrderByStringFromURI.contains(prop);
		else
		    isOrderBypresent = globalOrderByStringFromURI.contains(navigationName+"/"+prop);
		}
		return isOrderBypresent;
	}
	
	public static String prepareGlobalOrderBy(UriInfo globalUriInfo){
		String globalOrderByStringFromURI = null;
		if (globalUriInfo.getOrderBy()!=null) {
			globalOrderByStringFromURI = globalUriInfo.getOrderBy().getExpressionString();
		}
		return globalOrderByStringFromURI;
	}
	
	public static String convertParam(ReadEntityInfo eInfo, EdmType edmType, String paramName, String paramValue) {

		if (("Edm.DateTimeOffset".equals(edmType.toString())) || ("Edm.DateTime".equals(edmType.toString()))) {
			String cdsDataType = CSNUtil.getCDSParamDataType(eInfo.getEntityName(), eInfo.getServiceName(), paramName);
			if("cds.DateTime".equals(cdsDataType)) {
				paramValue = "timestamp'"+paramValue.replace("Z", "").replace("T", " ")+ "'";
			}else if("cds.Date".equals(cdsDataType)) {
				paramValue = "date'"+paramValue.substring(0, 10)+ "'";
			}else {
				paramValue = "'" + paramValue + "'";
			}
		} else if ("Edm.Time".equals(edmType.toString())) {
			paramValue = parseTime(paramValue);
		}else if ("Edm.String".equals(edmType.toString())) {
			paramValue = "'" + paramValue + "'";
		}

		return paramValue;
	}
	
	public static String parseTime(String value) {
		String hourValue = String.format("%02d", Integer.parseInt(value.substring(2, value.indexOf('H'))));
		String minValue = String.format("%02d",
				Integer.parseInt(value.substring(value.indexOf('H') + 1, value.indexOf('M'))));
		String secValue = String.format("%02d",
				Integer.parseInt(value.substring(value.indexOf('M') + 1, value.indexOf('S'))));

		value = "\'" + hourValue + ":" + minValue + ":" + secValue + "\'";
		return value;
	}

	public static Map<CaseInsensitiveString, String> getEntityTypeMap(EdmEntitySet entitySet) {
		Map<CaseInsensitiveString, String> propertyTypeMap = new HashMap<>();
		if (entitySet == null) {
			return null;
		}
		try {
			EdmEntityType edmType = entitySet.getEntityType();
			edmType.getPropertyNames().stream().forEach(p -> {
				try {
					propertyTypeMap.put(new CaseInsensitiveString(p), edmType.getProperty(p).getType().toString());
				} catch (EdmException e) {
					logger.error("Not able to get properties from edm Type", e);
				}
			});
		} catch (EdmException e) {
			logger.error("Not able to get property names from edm Type", e);
		}
		return propertyTypeMap;
	}

	// Todo: Handle exception
	public static Parameter getParameterizedParamater(KeyPredicate key, EdmSimpleType edmSimpleType) throws EdmException {

		switch (edmSimpleType.toString()) {
			case "Edm.Decimal":
				return new Parameter(key.getLiteral(), "Edm.Decimal", key.getProperty().getName());
			case "Edm.Double":
				return new Parameter(key.getLiteral(), "Edm.Double", key.getProperty().getName());
			case "Edm.Single":
				return new Parameter(key.getLiteral(), "Edm.Single", key.getProperty().getName());
			case "Edm.Int64":
				return new Parameter(key.getLiteral(), "Edm.Int64", key.getProperty().getName());
			case "Edm.Int32":
			case "Edm.Int16":
			case "System.Uint7":
			case "Edm.Byte":
			case "Edm.SByte":
				return new Parameter(key.getLiteral(), "Edm.Int", key.getProperty().getName());
			case "Edm.Binary":
				return new Parameter(key.getLiteral(), "Edm.Binary", key.getProperty().getName());
			case "Edm.Guid":
				return new Parameter(key.getLiteral(), "Edm.Guid", key.getProperty().getName());
			case "Edm.String":
				return new Parameter(key.getLiteral(), "Edm.String", key.getProperty().getName());
			case "Edm.DateTime":
			case "Edm.Time":
				return new Parameter(DateUtils.dateLiteralresolver("datetime\'" + key.getLiteral() + "\'"), "Edm.Date",
						key.getProperty().getName());
			case "Edm.DateTimeOffset":
				return new Parameter(DateUtils.dateLiteralresolver("datetimeoffset\'" + key.getLiteral() + "\'"),
						"Edm.DateTimeOffset", key.getProperty().getName());
			default:
				return new Parameter(key.getLiteral(), "Edm.String", key.getProperty().getName());
		}

	}
	
	public static ReadEntityInfo prepareOrderBy(CommonExpression currentExpression, String orderByon, String order, String parentAssociation,
			 List<ReadEntityInfo> siblingExpandList, ReadEntityInfo parentEntityInfo, List<String> globalSelect, String globalOrderByStringFromURI, ReadEntityInfo childEntityInfo) throws EdmException {
   	if(currentExpression.getKind() == ExpressionKind.PROPERTY) {
   		String navigatedPropName = currentExpression.getUriLiteral();
   		ReadEntityInfo re = QueryHelperUtility.prepareExpandForOrderBy((EdmEntityType) currentExpression.getEdmType(), navigatedPropName, orderByon, order, parentAssociation,
   				siblingExpandList, parentEntityInfo, globalSelect, globalOrderByStringFromURI);
   		if(childEntityInfo != null) 
   			re.getEntitiesExpanded().add(childEntityInfo);
   		return re;
   	}
   	
   	MemberExpression currentMember = (MemberExpression) currentExpression;
   	String navigatedPropName = currentMember.getProperty().getUriLiteral(); 
   	ReadEntityInfo re = QueryHelperUtility.prepareExpandForOrderBy((EdmEntityType) currentExpression.getEdmType(), navigatedPropName, orderByon, order, parentAssociation,
				siblingExpandList, parentEntityInfo, globalSelect, globalOrderByStringFromURI);
   	if(childEntityInfo != null) 
   		re.getEntitiesExpanded().add(childEntityInfo);
   	int index = parentAssociation.lastIndexOf("ZZ");
   	parentAssociation = parentAssociation.substring(0, index + 2);
   	return prepareOrderBy(((MemberExpression) currentExpression).getPath(), null, order, parentAssociation, siblingExpandList, parentEntityInfo, globalSelect, globalOrderByStringFromURI, re);
   }

   public static void checkEdmMultiplicity(EdmNavigationPropertyImplProv edmNavigationProp) throws EdmException, CDSRuntimeException {
   	if (edmNavigationProp != null) {
           EdmMultiplicity em = edmNavigationProp.getMultiplicity();
           if (em.compareTo(EdmMultiplicity.MANY) == 0)
               throw new CDSRuntimeException(MessageKeys.ORDERBY_ON_NAVIGATION,
                       "$OrderBy cannot be applied on navigated Property", LocaleUtil.getLocaleforException(),
                       HttpStatusCodes.BAD_REQUEST);
       }
   }
   
   public static Map<String, String> getKeyPredicatesMap(List<KeyPredicate> keyPredicates, boolean isDraftRoot) throws EdmException {

       Map<String, String> keyPredicateMap = new HashMap<String, String>();
       for (KeyPredicate key : keyPredicates) {
           // Ignore IsActiveEntity for draft root entity
           if (isDraftRoot && key.getProperty().getName().equals(DraftUtilsV2.DRAFTS_ISACTIVE_ENTITY)) {
               continue;
           }
           EdmSimpleType keyType = (EdmSimpleType) key.getProperty().getType();
           keyPredicateMap.put(key.getProperty().getName(), getValueOfProperty(key, keyType));
       }
       return keyPredicateMap;
   }
   
   public static String getValueOfProperty(KeyPredicate key, EdmSimpleType edmSimpleType) {
       String valueOfKey = "";
       if (edmSimpleType == EdmSimpleTypeKind.DateTimeOffset.getEdmSimpleTypeInstance()) {
           valueOfKey = DateUtils.dateLiteralresolver("datetimeoffset\'" + key.getLiteral() + "\'");
       } else if (edmSimpleType == EdmSimpleTypeKind.DateTime.getEdmSimpleTypeInstance()) {
           valueOfKey = DateUtils.dateLiteralresolver("datetime\'" + key.getLiteral() + "\'");
       } else if (!(edmSimpleType == EdmSimpleTypeKind.Int16.getEdmSimpleTypeInstance()
               || edmSimpleType == EdmSimpleTypeKind.Int32.getEdmSimpleTypeInstance()
               || edmSimpleType == EdmSimpleTypeKind.Int64.getEdmSimpleTypeInstance()
               || edmSimpleType == EdmSimpleTypeKind.Decimal.getEdmSimpleTypeInstance()
               || edmSimpleType == EdmSimpleTypeKind.Double.getEdmSimpleTypeInstance()
               || edmSimpleType == EdmSimpleTypeKind.Boolean.getEdmSimpleTypeInstance())) {
           valueOfKey = "\'" + key.getLiteral() + "\'";
       } else {
           valueOfKey = key.getLiteral();
       }
       if (edmSimpleType == EdmSimpleTypeKind.Guid.getEdmSimpleTypeInstance()) {
           valueOfKey = valueOfKey.toLowerCase();// Even if UUID Value is passed as uppercase
       }
       return valueOfKey;
   }
}
