/**
 * This class extends BaseDataProvider .Execute OData V2 Query & Returns  IDataProviderResponse
 *
 * 
 * @version 1.0
 * @since   2016-10-27 
 */
package com.sap.cloud.sdk.service.prov.v2.data.provider;

import static com.sap.cloud.sdk.service.prov.v2.rt.core.util.AuthorizationUtilV2.checkAuthorization;
import static com.sap.cloud.sdk.service.prov.v2.rt.core.util.AuthorizationUtilV2.handleAuthorizationException;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TimeZone;
import java.util.UUID;

import org.apache.olingo.odata2.api.commons.HttpStatusCodes;
import org.apache.olingo.odata2.api.commons.InlineCount;
import org.apache.olingo.odata2.api.edm.EdmEntitySet;
import org.apache.olingo.odata2.api.edm.EdmException;
import org.apache.olingo.odata2.api.edm.EdmFunctionImport;
import org.apache.olingo.odata2.api.edm.EdmLiteral;
import org.apache.olingo.odata2.api.edm.EdmLiteralKind;
import org.apache.olingo.odata2.api.edm.EdmMultiplicity;
import org.apache.olingo.odata2.api.edm.EdmParameter;
import org.apache.olingo.odata2.api.edm.EdmProperty;
import org.apache.olingo.odata2.api.edm.EdmSimpleType;
import org.apache.olingo.odata2.api.edm.EdmSimpleTypeException;
import org.apache.olingo.odata2.api.edm.EdmStructuralType;
import org.apache.olingo.odata2.api.edm.EdmTypeKind;
import org.apache.olingo.odata2.api.edm.EdmTyped;
import org.apache.olingo.odata2.api.ep.EntityProviderWriteProperties.ODataEntityProviderPropertiesBuilder;
import org.apache.olingo.odata2.api.ep.entry.ODataEntry;
import org.apache.olingo.odata2.api.exception.ODataApplicationException;
import org.apache.olingo.odata2.api.exception.ODataException;
import org.apache.olingo.odata2.api.exception.ODataNotImplementedException;
import org.apache.olingo.odata2.api.processor.ODataContext;
import org.apache.olingo.odata2.api.uri.KeyPredicate;
import org.apache.olingo.odata2.api.uri.SelectItem;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.expression.OrderExpression;
import org.apache.olingo.odata2.api.uri.expression.SortOrder;
import org.apache.olingo.odata2.api.uri.info.DeleteUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntitySetCountUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntitySetUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetEntityUriInfo;
import org.apache.olingo.odata2.api.uri.info.GetFunctionImportUriInfo;
import org.apache.olingo.odata2.api.uri.info.PostUriInfo;
import org.apache.olingo.odata2.api.uri.info.PutMergePatchUriInfo;
import org.apache.olingo.odata2.core.uri.UriInfoImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.api.EntityMetadata;
import com.sap.cloud.sdk.service.prov.api.Message;
import com.sap.cloud.sdk.service.prov.api.exception.ServiceSDKException;
import com.sap.cloud.sdk.service.prov.api.filter.Expression;
import com.sap.cloud.sdk.service.prov.api.request.MessageContainerImpl;
import com.sap.cloud.sdk.service.prov.api.request.OperationRequest;
import com.sap.cloud.sdk.service.prov.api.request.QueryRequest;
import com.sap.cloud.sdk.service.prov.api.request.impl.CreateRequestImpl;
import com.sap.cloud.sdk.service.prov.api.request.impl.DeleteRequestImpl;
import com.sap.cloud.sdk.service.prov.api.request.impl.OperationRequestImpl;
import com.sap.cloud.sdk.service.prov.api.request.impl.OrderByExpressionImpl;
import com.sap.cloud.sdk.service.prov.api.request.impl.QueryRequestImpl;
import com.sap.cloud.sdk.service.prov.api.request.impl.ReadRequestImpl;
import com.sap.cloud.sdk.service.prov.api.request.impl.UpdateRequestImpl;
import com.sap.cloud.sdk.service.prov.api.response.impl.CreateResponseImpl;
import com.sap.cloud.sdk.service.prov.api.response.impl.DeleteResponseImpl;
import com.sap.cloud.sdk.service.prov.api.response.impl.ErrorResponseImpl;
import com.sap.cloud.sdk.service.prov.api.response.impl.OperationResponseImpl;
import com.sap.cloud.sdk.service.prov.api.response.impl.QueryResponseImpl;
import com.sap.cloud.sdk.service.prov.api.response.impl.ReadResponseImpl;
import com.sap.cloud.sdk.service.prov.api.response.impl.UpdateResponseImpl;
import com.sap.cloud.sdk.service.prov.api.security.UnauthorizedException;
import com.sap.cloud.sdk.service.prov.api.statistics.SAPStatistics;
import com.sap.cloud.sdk.service.prov.api.util.DataConversionUtility;
import com.sap.cloud.sdk.service.prov.api.util.ProcessorHelper;
import com.sap.cloud.sdk.service.prov.v2.request.util.FunctionImportUtil;
import com.sap.cloud.sdk.service.prov.v2.response.util.EntityDataResponseHandler;
import com.sap.cloud.sdk.service.prov.v2.response.util.PojoResponseHandler;
import com.sap.cloud.sdk.service.prov.v2.rt.core.EntityDataV2;
import com.sap.cloud.sdk.service.prov.v2.rt.core.extensions.EntityMetadataV2;
import com.sap.cloud.sdk.service.prov.v2.rt.core.filter.ExpressionBuilderImplV2;
import com.sap.cloud.sdk.service.prov.v2.rt.data.provider.DataProvider;
import com.sap.gateway.core.api.batch.IBatchResult;
import com.sap.gateway.core.api.batch.IBatchTask;
import com.sap.gateway.core.api.batch.ICountEntitySetParameter;
import com.sap.gateway.core.api.batch.IParameter;
import com.sap.gateway.core.api.batch.IReadEntityParameter;
import com.sap.gateway.core.api.batch.IReadEntitySetParameter;
import com.sap.gateway.core.api.batch.IReadExpandedEntityParameter;
import com.sap.gateway.core.api.batch.IReadExpandedEntitySetParameter;
import com.sap.gateway.core.api.batch.IRequest;
import com.sap.gateway.core.api.batch.impl.BatchResult;
import com.sap.gateway.core.api.exception.ODataApplicationExceptionWithMessageContainer;
import com.sap.gateway.core.api.message.MessageContainer;
import com.sap.gateway.core.api.provider.data.BaseDataProviderResponse;
import com.sap.gateway.core.api.provider.data.IDataProviderResponse;
import com.sap.gateway.core.api.sap.statistics.SapStatisticsConstants;
import com.sap.gateway.core.api.srvrepo.IServiceInfo;
import com.sap.gateway.core.internal.GWUriInfo;


public class CXSDataProvider extends DataProvider {

	private final static Logger logger = LoggerFactory.getLogger(CXSDataProvider.class);
	private static Logger log = LoggerFactory.getLogger(CXSDataProvider.class);
	private static final String Function = "Function";
	private static final String Action = "Action";
    private static final String READ = "READ";
    private static final String WRITE = "WRITE";
    private static final String FUNCTION = "*";
	boolean isBatchRequest = false;
	boolean expanding = false;
	private String serviceName;
	private FunctionImportUtil util = new FunctionImportUtil();
	private SAPStatistics timings = new SAPStatistics();

	public CXSDataProvider(IServiceInfo service) {
		super(service);
		this.serviceName = service.getServiceName();
	}

	/**
	 * read Entity
	 */
	public IDataProviderResponse readEntity(GetEntityUriInfo uriInfo, ODataContext context) throws ODataException {

		logger.debug("Entering CXSDataProvider >> {readEntity}");
		timings = new SAPStatistics();
        try {
            checkAuthorization(uriInfo, READ);
        } catch (UnauthorizedException ue) {
            logger.error("User does not have authorization to perform the read operation on this entity.");
            return handleAuthorizationException(ue);
        }

		BaseDataProviderResponse response = new BaseDataProviderResponse();
		EdmEntitySet entitySet = uriInfo.getStartEntitySet();
		if (expanding) {
			entitySet = uriInfo.getTargetEntitySet();
		}
		if (uriInfo.getNavigationSegments() != null && uriInfo.getNavigationSegments().size() > 0) {
			entitySet = uriInfo.getTargetEntitySet();
		}
		EntityMetadata entityMetadata = new EntityMetadataV2(entitySet);
		Map<String, Object> keys = getKeys(uriInfo.getKeyPredicates());
		ReadResponseImpl readResponse = null;
		ReadRequestImpl readRequest = new ReadRequestImpl(null, entitySet.getName(), entityMetadata,
				context.getRequestHeaders(), keys, context.getHttpMethod());
		if (expanding && (!uriInfo.getStartEntitySet().getName().equals(uriInfo.getTargetEntitySet().getName()))) {
			((ReadRequestImpl) readRequest).setSourceEntityName(uriInfo.getStartEntitySet().getName());
			try {
				((ReadRequestImpl) readRequest).setSourceKeys(getKeys(uriInfo.getKeyPredicates()));
			} catch (Exception e1) {
				log.error("Error in fetching keys");
			}
		}
		if (uriInfo.getNavigationSegments() != null && uriInfo.getNavigationSegments().size() > 0 && !expanding) {
			((ReadRequestImpl) readRequest).setSourceEntityName(uriInfo.getStartEntitySet().getName());
			try {
				((ReadRequestImpl) readRequest).setSourceKeys(getKeys(uriInfo.getKeyPredicates()));
				// since we support only 1 navigation,so hard coding get(0) of array
				// NavigationSegments
				((ReadRequestImpl) readRequest)
						.setKeys(getKeys(uriInfo.getNavigationSegments().get(0).getKeyPredicates()));
			} catch (Exception e1) {
				log.error("Error in fetching keys");
			}
		}
		try {
			readResponse = (ReadResponseImpl) ProcessorHelper.invokeOperation(readRequest, serviceName,
					ProcessorHelper.READ, timings);
		} catch (ServiceSDKException e) {
			log.error("Exception in read Entity", e);
			if (e.getStatusCode() == 501) {
				ErrorResponseImpl errorResp = new ErrorResponseImpl(501, "", "Method Not Implemented", null, null);
				response.setException(new ODataApplicationException(errorResp.getMessage(), Locale.getDefault(),
						HttpStatusCodes.NOT_IMPLEMENTED, ""));
			} else
				throw new ODataException(e.getMessage(), e);
		}
		if (readResponse != null) {
			if (readResponse.getData() != null) {
				Map<String, Object> resultMap = DataConversionUtility.convertToMap(readResponse.getData());
				List<Map<String, Object>> resultList = new ArrayList<>();
				resultList.add(resultMap);

				/*
				 * Correction for UUID only to be applied in case of Extension Auto -Exposure
				 * has logic in ResultSetProcessor
				 */
				castStringToGuidWithCompelxProperty(getUUIDColumns((UriInfo) uriInfo), resultList,
						uriInfo.getTargetEntitySet().getEntityType());

				response.setResultEntity(resultList.get(0));
			}
		}
		if (readResponse != null) {
			if ((ErrorResponseImpl) readResponse.getErrorResponse() != null) {
				response.setException(convertToODataApplicationException(
						(ErrorResponseImpl) readResponse.getErrorResponse(), ProcessorHelper.READ));
			}
		}
		if (readResponse != null && readResponse.getHeaders() != null && !readResponse.getHeaders().isEmpty()) {
			setResponseHeaders(response, readResponse.getHeaders());
		}
		setStatisticsInContext(context, response, timings);
		logger.debug("Exiting CXSDataProvider >> {readEntity}");
		return response;
	}

	/**
	 * To-DO : This method needs refactoring to accomodate any edmtype as key
	 * 
	 * @param key
	 * @return
	 * @throws EdmException
	 * @throws EdmSimpleTypeException
	 * @throws ODataException
	 */
	// To-DO this method should be enhanced to support all Edm types as key
	private Map<String, Object> getKeys(List<KeyPredicate> keyPredicates) throws EdmSimpleTypeException, EdmException {
		Map<String, Object> keys = new HashMap<>();
		for (KeyPredicate keyPredicate : keyPredicates) {
			EdmProperty property = keyPredicate.getProperty();
			EdmSimpleType type = (EdmSimpleType) property.getType();
			if (type.toString().equals("Edm.String")) {
				String value = type.valueOfString(keyPredicate.getLiteral(), EdmLiteralKind.DEFAULT,
						property.getFacets(), String.class);
				keys.put(property.getName(), value);
			}
			if (type.toString().equals("Edm.Int32")) {
				int value = type.valueOfString(keyPredicate.getLiteral(), EdmLiteralKind.DEFAULT, property.getFacets(),
						Integer.class);
				keys.put(property.getName(), value);
			}

			if (type.toString().equals("Edm.Int64")) {
				long value = type.valueOfString(keyPredicate.getLiteral(), EdmLiteralKind.DEFAULT, property.getFacets(),
						Long.class);
				keys.put(property.getName(), value);
			}

			if (type.toString().equals("Edm.Double")) {
				double value = type.valueOfString(keyPredicate.getLiteral(), EdmLiteralKind.DEFAULT,
						property.getFacets(), Double.class);
				keys.put(property.getName(), value);
			}

			if (type.toString().startsWith("Edm.Date") || type.toString().equals("Edm.Time")) {
				TimeZone.setDefault(TimeZone.getTimeZone("GMT"));
				Date value = type.valueOfString(keyPredicate.getLiteral(), EdmLiteralKind.DEFAULT, property.getFacets(),
						Date.class);
				keys.put(property.getName(), value);
			}
			if (type.toString().equals("Edm.Guid")) {
				UUID value = type.valueOfString(keyPredicate.getLiteral(), EdmLiteralKind.DEFAULT, property.getFacets(),
						UUID.class);
				keys.put(property.getName(), value);
			}
		}
		return keys;
	}

	
	@Override
	public IDataProviderResponse countEntitySet(GetEntitySetCountUriInfo uriInfo, ODataContext context) throws ODataException {
			GWUriInfo entitySetUriInfo = new GWUriInfo((UriInfo) uriInfo, uriInfo.getStartEntitySet(), uriInfo.getKeyPredicates(), uriInfo.getNavigationSegments(), uriInfo.getFilter());
			BaseDataProviderResponse response = (BaseDataProviderResponse) readEntitySetWithCountBoolean(entitySetUriInfo, context, true);
			if (response.getCount() == null) {
				response.setCount(Integer.toString(response.getResultEntities().size()));
			}
			return response;
	}
	
	
	/*
	 * private QueryRequest toggleFilterOptionInQueryRequest(RequestContext
	 * requestContext, String entityName, EntityMetadata entityMetadata, Map<String,
	 * List<String>> requestHeaders, String httpMethod, FilterExpression
	 * filterExpression) { // Based on the existence of odatav2-filter-X.XX.XXX.jar
	 * filter is enabled or // disabled. This jar is not a direct pom dependency //
	 * Rather it is made available by the dependency of the OData Provisioning //
	 * Application // Enable this feature is the service is found (via available
	 * jars) ServiceLoader<QueryRequestV2WithFilterFactory> serviceLoader =
	 * ServiceLoader .load(QueryRequestV2WithFilterFactory.class);
	 * Iterator<QueryRequestV2WithFilterFactory> iter = serviceLoader.iterator(); if
	 * (!iter.hasNext()) { // when jar is not present return new
	 * QueryRequestImpl(requestContext, entityName, entityMetadata, requestHeaders,
	 * httpMethod); } else { // when jar is present QueryRequestV2WithFilterFactory
	 * queryRequestV2WithFilterFactory = iter.next(); if
	 * (queryRequestV2WithFilterFactory != null) { QueryRequestV2WtFilter
	 * reqWithFilter = queryRequestV2WithFilterFactory.getInstance(requestContext,
	 * entityName, entityMetadata, requestHeaders, httpMethod, filterExpression);
	 * reqWithFilter.setServiceName(serviceName); return reqWithFilter ; } }
	 * 
	 * throw new ODataRuntimeException("Invalid Filter Expression"); }
	 */

	/**
	 * read entity collection
	 */
	@SuppressWarnings("unchecked")
	public IDataProviderResponse readEntitySet(GetEntitySetUriInfo uriInfo, ODataContext context)
			throws ODataException {
		return readEntitySetWithCountBoolean(uriInfo, context, false);
	}
	
	private IDataProviderResponse readEntitySetWithCountBoolean(GetEntitySetUriInfo uriInfo, ODataContext context, boolean count)
			throws ODataException {
		logger.debug("Entering CXSDataProvider >> {readEntitySet}");
		timings = new SAPStatistics();
        try {
            checkAuthorization(uriInfo, READ);
        } catch (UnauthorizedException ue) {
            logger.error("Read operation on this entity set not allowed for this user");
            return handleAuthorizationException(ue);
        }

		BaseDataProviderResponse response = new BaseDataProviderResponse();
		List<Map<String, Object>> resultEntities = new ArrayList<Map<String, Object>>();
		EdmEntitySet entitySet = uriInfo.getStartEntitySet();
		if (expanding) {
			entitySet = uriInfo.getTargetEntitySet();
		}
		if (uriInfo.getNavigationSegments() != null && uriInfo.getNavigationSegments().size() > 0) {
			entitySet = uriInfo.getTargetEntitySet();
		}
		EntityMetadata entityMetadata = new EntityMetadataV2(entitySet);
		QueryResponseImpl queryResponse = null;
		QueryRequestImpl queryRequest = new QueryRequestImpl(null, entitySet.getName(), entityMetadata,
				context.getRequestHeaders(), context.getHttpMethod());
		Expression filterTree = new ExpressionBuilderImplV2().buildForV2(uriInfo.getFilter());
		queryRequest.setQueryExpression(filterTree);
		queryRequest.setServiceName(serviceName);
		setOrderBy(queryRequest, uriInfo);
		setTopOption(queryRequest, uriInfo);
		setInlineCountOption(queryRequest,uriInfo);
		setSkipOption(queryRequest, uriInfo);
		setSelect(queryRequest, uriInfo);
		setCustomQueryOptions(queryRequest, uriInfo);
		queryRequest.setCountCall(count);
		
		if (expanding && (!uriInfo.getStartEntitySet().getName().equals(uriInfo.getTargetEntitySet().getName()))) {
			((QueryRequestImpl) queryRequest).setSourceEntityName(uriInfo.getStartEntitySet().getName());
			try {
				((QueryRequestImpl) queryRequest).setSourceKeys(getKeys(uriInfo.getKeyPredicates()));
			} catch (Exception e1) {
				log.error("Error in fetching keys");
			}
		}

		if (uriInfo.getNavigationSegments() != null && uriInfo.getNavigationSegments().size() > 0 && !expanding) {
			((QueryRequestImpl) queryRequest).setSourceEntityName(uriInfo.getStartEntitySet().getName());
			try {
				((QueryRequestImpl) queryRequest).setSourceKeys(getKeys(uriInfo.getKeyPredicates()));
			} catch (Exception e1) {
				log.error("Error in fetching keys");
			}
		}

		try {
			queryResponse = (QueryResponseImpl) ProcessorHelper.invokeOperation(queryRequest, serviceName,
					ProcessorHelper.QUERY, timings);

			if (queryResponse.getData() != null) {
				resultEntities = (List<Map<String, Object>>) queryResponse.getData();
			}
			if (queryResponse.getEntityData() != null) {
				resultEntities = (List<Map<String, Object>>) EntityDataResponseHandler
						.getResultEntityFromEntityData(queryResponse.getEntityData());
			}
			if (queryResponse.getPojoData() != null) {
				resultEntities = (List<Map<String, Object>>) PojoResponseHandler
						.getHashMapFromListOfPojo(queryResponse.getPojoData());
			}

			/*
			 * Correction for UUID only to be applied in case of Extension Auto -Exposure
			 * has logic in ResultSetProcessor
			 */
			if (response != null && resultEntities != null && !resultEntities.isEmpty()) {
				castStringToGuidWithCompelxProperty((getUUIDColumns((UriInfo) uriInfo)), resultEntities,
						uriInfo.getTargetEntitySet().getEntityType());
			}
			
			// handle inlinecount
			if (queryRequest.isInlineCountCall()) { //?$inlinecount=allpages
				if (queryResponse.getInlineCount() > -1) {
					response.setInlineCount(queryResponse.getInlineCount());
				} else {
					response.setInlineCount((resultEntities == null ? 0 : resultEntities.size()));
				}
			} //if inline count is false do not set inline count ?$inlinecount=none
			
			
			if(queryResponse.getCount() > -1) {
				response.setCount(queryResponse.getCount()+"");
			}
			
			// Fix for CSN:1880253389 Starts
			boolean isSkipHappened = false;
			// If custom code has not done $skip already then care for $skip at framework
			// level.
			if (uriInfo.getSkip() != null && !queryResponse.isSkipDone()) {
				// $skip
				handleSkipOption(resultEntities, uriInfo.getSkip(), response);
				isSkipHappened = true;
			}
			if (isSkipHappened) {
				// $skip and $top
				handleTopOption(response.getResultEntities(), uriInfo.getTop(), response, queryResponse);
			} else {
				// $top or All result entities
				handleTopOption(resultEntities, uriInfo.getTop(), response, queryResponse);
			}
			
			// Fix for CSN:1880253389 Ends
			if ((ErrorResponseImpl) queryResponse.getErrorResponse() != null) {
				response.setException(convertToODataApplicationException(
						(ErrorResponseImpl) queryResponse.getErrorResponse(), ProcessorHelper.QUERY));
			}
			setResponseHeaders(response, queryResponse.getHeaders());
			setStatisticsInContext(context, response, timings);
		} catch (ServiceSDKException e) {
			log.error("Exception in read Entity", e);
			if (e.getStatusCode() == 501) {
				ErrorResponseImpl errorResp = new ErrorResponseImpl(501, "", "Method Not Implemented", null, null);
				response.setException(new ODataApplicationException(errorResp.getMessage(), Locale.getDefault(),
						HttpStatusCodes.NOT_IMPLEMENTED, ""));
			} else
				throw new ODataException(e.getMessage(), e);
		}

		logger.debug("Exiting CXSDataProvider >> {readEntity}");
		return response;

	}

	private void setTopOption(QueryRequest queryRequest, GetEntitySetUriInfo uriInfo) {
		if (uriInfo.getTop() != null) {
			((QueryRequestImpl) queryRequest).setTopOption(uriInfo.getTop());
		}
	}

	private void setInlineCountOption(QueryRequest queryRequest, GetEntitySetUriInfo uriInfo) {
		if (uriInfo.getInlineCount() != null) {
			if (InlineCount.ALLPAGES.toString().equalsIgnoreCase(uriInfo.getInlineCount().name())) {
				((QueryRequestImpl) queryRequest).setIsInlineCount(true);
			} else {
				((QueryRequestImpl) queryRequest).setIsInlineCount(false);
			}
		}
	}

	
	private void setSkipOption(QueryRequest queryRequest, GetEntitySetUriInfo uriInfo) {
		if (uriInfo.getSkip() != null) {
			((QueryRequestImpl) queryRequest).setSkipOption(uriInfo.getSkip());
		}
	}

	private void setCustomQueryOptions(QueryRequest queryRequest, GetEntitySetUriInfo uriInfo) {
		if (uriInfo.getCustomQueryOptions() != null) {
			((QueryRequestImpl) queryRequest).setCustomQueryOptions(uriInfo.getCustomQueryOptions());
		}
	}

	
	/*
	 * private void handleInlineCountOption(List<Map<String, Object>>
	 * resultEntities, String inlineCountOption, BaseDataProviderResponse response,
	 * QueryResponseImpl queryResponse) {
	 * 
	 * // If custom code has not done $top already then care for $top at framework
	 * // level. if (queryResponse.getInlineCount() > -1) {
	 * response.setInlineCount(queryResponse.getInlineCount()); } else {
	 * response.setInlineCount(resultEntities.size()); } }
	 */
	 

	private void handleTopOption(List<Map<String, Object>> resultEntities, Integer topOption,
			BaseDataProviderResponse response, QueryResponseImpl queryResponse) {
		List<Map<String, Object>> topResult = new ArrayList<Map<String, Object>>();

		// If custom code has not done $top already then care for $top at framework
		// level.
		if (topOption != null && !queryResponse.isTopDone()) {
			int topNumber = topOption.intValue();
			if (topNumber >= 0) {
				if (topNumber <= resultEntities.size()) {
					topResult.addAll(resultEntities.subList(0, topNumber));
				} else {
					topResult.addAll(resultEntities);
				}
			}
			response.setResultEntities(topResult);
		} else {
			response.setResultEntities(resultEntities);
		}
	}

	private void handleSkipOption(List<Map<String, Object>> resultEntities, Integer skipOption,
			BaseDataProviderResponse response) {
		List<Map<String, Object>> skipResult = new ArrayList<Map<String, Object>>();

		if (skipOption != null) {
			int skipNumber = skipOption.intValue();
			if (skipNumber >= 0) {
				// If skip number is > than result entities size then it should be empty
				if (skipNumber <= resultEntities.size()) {
					skipResult.addAll(resultEntities.subList(skipOption, resultEntities.size()));
				}
			}
			response.setResultEntities(skipResult);
		}
	}

	private void setSelect(QueryRequest queryRequest, GetEntitySetUriInfo uriInfo) {
		if (uriInfo.getSelect() != null) {
			for (SelectItem select : uriInfo.getSelect()) {
				if (select != null) {
					if (select.getProperty() != null) {
						String str = null;
						try {
							str = select.getProperty().getName();
						} catch (EdmException e) {
							logger.error("Error while setting select option");
						}
						((QueryRequestImpl) queryRequest).setSelectItems(str);
					}

				}
			}
		}
	}

	private void setOrderBy(QueryRequest queryRequest, GetEntitySetUriInfo uriInfo) {
		OrderByExpressionImpl orderItem = null;
		if (uriInfo.getOrderBy() != null) {
			for (OrderExpression orderExpr : uriInfo.getOrderBy().getOrders()) {
				SortOrder sortOrder = orderExpr.getSortOrder();
				String order = sortOrder.name();
				boolean isDescending = order.equals("desc");
				String expression = orderExpr.getExpression().getUriLiteral();
				orderItem = new OrderByExpressionImpl(isDescending, expression);
				((QueryRequestImpl) queryRequest).setOrderByOption(orderItem);
			}
		}
	}

	public IDataProviderResponse createEntity(PostUriInfo uriInfo, ODataEntry content, String requestContentType,
			ODataEntityProviderPropertiesBuilder providerPropertiesBuilder, ODataContext context)
			throws ODataException {
		timings = new SAPStatistics();
        try {
            checkAuthorization(uriInfo, WRITE);
        } catch (UnauthorizedException ue) {
            logger.error("Create operation on this entity not allowed for this user");
            return handleAuthorizationException(ue);
        }
		BaseDataProviderResponse response = new BaseDataProviderResponse();
		EdmEntitySet entitySet = uriInfo.getStartEntitySet();
		EntityMetadata entityMetadata = new EntityMetadataV2(entitySet);
		CreateResponseImpl createResponse;
		if (uriInfo.getNavigationSegments() != null && !uriInfo.getNavigationSegments().isEmpty()) {
			logger.error("Create with Navigation Not implemented");
		} else {

			CreateRequestImpl createRequest = new CreateRequestImpl(null, entitySet.getName(), entityMetadata,
					context.getRequestHeaders(), context.getHttpMethod());
			EntityDataV2 entityData = new EntityDataV2(content.getProperties(), entityMetadata);
			createRequest.setData(entityData);
			try {
				createResponse = (CreateResponseImpl) ProcessorHelper.invokeOperation(createRequest, this.serviceName,
						ProcessorHelper.CREATE, timings);
				if (createResponse.getData() != null) {
					// response.setResultEntity(DataConversionUtility.convertToMap(createResponse.getData()));
					Map<String, Object> resultMap = DataConversionUtility.convertToMap(createResponse.getData());
					List<Map<String, Object>> resultList = new ArrayList<>();
					resultList.add(resultMap);

					/*
					 * Correction for UUID only to be applied in case of Extension Auto -Exposure
					 * has logic in ResultSetProcessor
					 */
					castStringToGuidWithCompelxProperty(getUUIDColumns((UriInfo) uriInfo), resultList,
							uriInfo.getTargetEntitySet().getEntityType());
					response.setResultEntity(resultList.get(0));

				}
				if ((ErrorResponseImpl) createResponse.getErrorResponse() != null) {
					response.setException(convertToODataApplicationException(
							(ErrorResponseImpl) createResponse.getErrorResponse(), ProcessorHelper.CREATE));
				}
				setResponseHeaders(response, createResponse.getHeaders());
				setStatisticsInContext(context, response, timings);
			} catch (ServiceSDKException e) {
				logger.error("Error in CreateEntity", e);
				if (e.getStatusCode() == 501) {
					ErrorResponseImpl errorResp = new ErrorResponseImpl(501, "", "Method Not Implemented", null, null);
					response.setException(new ODataApplicationException(errorResp.getMessage(), Locale.getDefault(),
							HttpStatusCodes.NOT_IMPLEMENTED, ""));
				} else
					throw new ODataException(e.getMessage(), e);
			}
		}

		return response;
	}

	public IDataProviderResponse updateEntity(PutMergePatchUriInfo uriInfo, ODataEntry content,
			String requestContentType, boolean merge, ODataContext context) throws ODataException {
		timings = new SAPStatistics();
        try {
            checkAuthorization(uriInfo, WRITE);
        } catch (UnauthorizedException ue) {
            logger.error("Update operation on this entity not allowed for this user");
            return handleAuthorizationException(ue);
        }
		BaseDataProviderResponse response = new BaseDataProviderResponse();
		EdmEntitySet entitySet = uriInfo.getStartEntitySet();
		EntityMetadata entityMetadata = new EntityMetadataV2(entitySet);
		Map<String, Object> keys = getKeys(uriInfo.getKeyPredicates());

		UpdateResponseImpl updateResponse = null;
		UpdateRequestImpl updateRequest = new UpdateRequestImpl(null, entitySet.getName(), entityMetadata,
				context.getRequestHeaders(), keys, context.getHttpMethod());
		try {
			EntityDataV2 entityData = new EntityDataV2(content.getProperties(), entityMetadata);
			updateRequest.setData(entityData);
			updateRequest.setData(content.getProperties());
			updateResponse = (UpdateResponseImpl) ProcessorHelper.invokeOperation(updateRequest, serviceName,
					ProcessorHelper.UPDATE, timings);
			if ((ErrorResponseImpl) updateResponse.getErrorResponse() != null) {
				response.setException(convertToODataApplicationException(
						(ErrorResponseImpl) updateResponse.getErrorResponse(), ProcessorHelper.UPDATE));
			}
			setResponseHeaders(response, updateResponse.getHeaders());
			setStatisticsInContext(context, response, timings);
		} catch (ServiceSDKException e) {
			log.error("Exception in update Entity", e);
			if (e.getStatusCode() == 501) {
				ErrorResponseImpl errorResp = new ErrorResponseImpl(501, "", "Method Not Implemented", null, null);
				response.setException(new ODataApplicationException(errorResp.getMessage(), Locale.getDefault(),
						HttpStatusCodes.NOT_IMPLEMENTED, ""));
			} else
				throw new ODataException(e.getMessage(), e);
		}

		return response;
	}

	public IDataProviderResponse deleteEntity(DeleteUriInfo uriInfo, ODataContext context) throws ODataException {
		timings = new SAPStatistics();
        try {
            checkAuthorization(uriInfo, WRITE);
        } catch (UnauthorizedException ue) {
            logger.error("Delete operation not allowed for this user");
            return handleAuthorizationException(ue);
        }
		BaseDataProviderResponse response = new BaseDataProviderResponse();
		EdmEntitySet entitySet = uriInfo.getStartEntitySet();
		EntityMetadata entityMetadata = new EntityMetadataV2(entitySet);
		Map<String, Object> keys = getKeys(uriInfo.getKeyPredicates());

		DeleteResponseImpl deleteResponse = null;
		DeleteRequestImpl deleteRequest = new DeleteRequestImpl(null, entitySet.getName(), entityMetadata,
				context.getRequestHeaders(), keys, context.getHttpMethod());
		try {
			deleteResponse = (DeleteResponseImpl) ProcessorHelper.invokeOperation(deleteRequest, serviceName,
					ProcessorHelper.DELETE, timings);
			if ((ErrorResponseImpl) deleteResponse.getErrorResponse() != null) {
				response.setException(convertToODataApplicationException(
						(ErrorResponseImpl) deleteResponse.getErrorResponse(), ProcessorHelper.DELETE));
			}
			setResponseHeaders(response, deleteResponse.getHeaders());
			setStatisticsInContext(context, response, timings);
		} catch (ServiceSDKException e) {
			log.error("Exception in delete Entity", e);
			if (e.getStatusCode() == 501) {
				ErrorResponseImpl errorResp = new ErrorResponseImpl(501, "", "Method Not Implemented", null, null);
				response.setException(new ODataApplicationException(errorResp.getMessage(), Locale.getDefault(),
						HttpStatusCodes.NOT_IMPLEMENTED, ""));
			} else
				throw new ODataException(e.getMessage(), e);
		}
		return response;
	}

	public List<IBatchResult> executeBatch(List<IBatchTask> batchTasks, ODataContext context) throws ODataException {
		isBatchRequest = true;
		List<IBatchResult> result = new ArrayList<IBatchResult>();
		for (IBatchTask batchTask : batchTasks) {
			BatchResult batchResult = new BatchResult();
			List<IRequest> requests = batchTask.getRequests();
			List<Integer> referencingRequestsData = batchTask.getReferencingRequestsData();
			if (batchTask.isChangeSet()) {
				List<IDataProviderResponse> responses = executeChangeSet(requests, context, referencingRequestsData);
				for (IDataProviderResponse response : responses) {
					batchResult.addResponse(response);
				}
				// logic for changeset
			} else {
				batchResult.addResponse(executeOperation(requests.get(0)));
			}
			result.add(batchResult);
		}
		isBatchRequest = false;
		return result;
	}

	private IDataProviderResponse executeOperation(IRequest request) {
		IParameter parameter = request.getParameter();
		ODataContext context = parameter.getContext();

		try {
			switch (request.getMethod()) {

			case READ_ENTITY:
				IReadEntityParameter readEntityParameter = (IReadEntityParameter) parameter;
				return readEntity(readEntityParameter.getUriInfo(), context);

			case READ_ENTITY_SET:
				IReadEntitySetParameter readEntitySetParameter = (IReadEntitySetParameter) parameter;
				return readEntitySet(readEntitySetParameter.getUriInfo(), context);
			case COUNT_ENTITY_SET:
				ICountEntitySetParameter countEntitySetParameter = (ICountEntitySetParameter) parameter;
                    try {
                        checkAuthorization( countEntitySetParameter.getUriInfo(), READ);
                    } catch (UnauthorizedException ue) {
                        logger.error("Count operation on this entity set not allowed for this user");
                        return handleAuthorizationException(ue);
                    }
				return countEntitySet(countEntitySetParameter.getUriInfo(), context);
			case READ_EXPANDED_ENTITY:
				IReadExpandedEntityParameter readExpandedEntityParameter = (IReadExpandedEntityParameter) parameter;
				return readExpandedEntity(readExpandedEntityParameter.getUriInfo(), context);
			case READ_EXPANDED_ENTITY_SET:
				IReadExpandedEntitySetParameter readExpandedEntitySetParameter = (IReadExpandedEntitySetParameter) parameter;
				return readExpandedEntitySet(readExpandedEntitySetParameter.getUriInfo(), context);
			default:
				throw new ODataNotImplementedException();
			}
		} catch (ODataException e) {
			BaseDataProviderResponse response = new BaseDataProviderResponse();
			response.setException(e);
			response.setStatusCode(HttpStatusCodes.INTERNAL_SERVER_ERROR);
			return response;
		}

	}

	public IDataProviderResponse executeFunctionImport(GetFunctionImportUriInfo uriInfo, ODataContext context)
			throws ODataException {
		logger.debug("Entering CXSDataProvider >> {executeFunctionImport}");
		timings = new SAPStatistics();
		try {
            checkAuthorization(uriInfo, FUNCTION);
        } catch (UnauthorizedException ue) {
            logger.error("Function Import operation not allowed for this user");
            return handleAuthorizationException(ue);
        }
		BaseDataProviderResponse response = new BaseDataProviderResponse();
		List<Map<String, Object>> resultEntities = new ArrayList<Map<String, Object>>();
		EdmFunctionImport function = uriInfo.getFunctionImport();
		OperationResponseImpl operationResponse = null;
		Map<String, Object> funcParams = new HashMap<String, Object>();
		Map<String, EdmLiteral> functionParameters = uriInfo.getFunctionImportParameters();
		try {
			if (functionParameters.isEmpty()) {
				Collection<String> params = function.getEntityContainer().getFunctionImport(function.getName())
						.getParameterNames();
				if (!(params == null || params.isEmpty())) {
					ErrorResponseImpl errorResp = new ErrorResponseImpl(400, "", "Invalid Function Import Parameter",
							null, null);
					response.setException(new ODataApplicationException(errorResp.getMessage(), Locale.getDefault(),
							HttpStatusCodes.BAD_REQUEST, ""));
				}
				if ((params == null || params.isEmpty()) && !uriInfo.getCustomQueryOptions().isEmpty()
						&& !uriInfo.getCustomQueryOptions().containsKey("sap-language") && !uriInfo.getCustomQueryOptions().containsKey("saml2")) {
					ErrorResponseImpl errorResp = new ErrorResponseImpl(400, "", "Invalid Function Import Parameter",
							null, null);
					response.setException(new ODataApplicationException(errorResp.getMessage(), Locale.getDefault(),
							HttpStatusCodes.BAD_REQUEST, ""));
				}
			}
			Iterator<String> iterator = functionParameters.keySet().iterator();
			while (iterator.hasNext()) {
				String key = iterator.next();
				EdmLiteral literal = functionParameters.get(key);
				EdmParameter param = function.getParameter(key);

				/*
				 * Here we should take care not to take the type of the literal, because in that
				 * case, based on the value of the literal, Olingo will give different types,
				 * for example, If the parameter was of type Edm.Int32 then if 10 was passed,
				 * Olingo gives Byte, 1000 short etc. Instead we do the object conversion using
				 * the Edm type of the parameter, thus making the logic independent of the
				 * current value of the parameter
				 */

				Object value = util.getFunctionImportParameterValue(literal, (EdmSimpleType) param.getType());
				funcParams.put(key, value);
			}

			EntityMetadata entityMetadata = function.getEntitySet() == null ? null
					: new EntityMetadataV2(function.getEntitySet());
			OperationRequest operationRequest = new OperationRequestImpl(null, function.getName(), entityMetadata,
					context.getRequestHeaders(), funcParams, function.getHttpMethod());
			String operation = operationRequest.getHttpMethod().equals("POST") ? Action : Function;
			operationResponse = (OperationResponseImpl) ProcessorHelper.invokeOperation(operationRequest, serviceName,
					operation, timings);

			if ((ErrorResponseImpl) operationResponse.getErrorResponse() != null) {
				response.setException(convertToODataApplicationException(
						(ErrorResponseImpl) operationResponse.getErrorResponse(), ProcessorHelper.FUNCTION));
			} else { // BCP ID 1970135453-2019
				response = getFunctionResponse(uriInfo, response, operationResponse);
			}
			setResponseHeaders(response, operationResponse.getHeaders());
			setStatisticsInContext(context, response, timings);
			MessageContainerImpl msgCon = (MessageContainerImpl) operationRequest.getMessageContainer();
			//adding message to response
			((BaseDataProviderResponse) response)
					.setMessageContainer(getIGWMessageContainerFromMessageContainer(msgCon));
		} catch (ServiceSDKException e) {
			log.error("Exception in execute FunctionImport", e);
			if (e.getStatusCode() == 501) {
				ErrorResponseImpl errorResp = new ErrorResponseImpl(501, "", "Method Not Implemented", null, null);
				response.setException(new ODataApplicationException(errorResp.getMessage(), Locale.getDefault(),
						HttpStatusCodes.NOT_IMPLEMENTED, ""));
			} else
				throw new ODataException(e.getMessage(), e);
		}

		logger.debug("Exiting CXSDataProvider >> {executeFunctionImport}");
		return response;
	}

	@Override
	public IDataProviderResponse readExpandedEntitySet(GetEntitySetUriInfo uriInfo, ODataContext context)
			throws ODataException {
		logger.debug("Entering CXSDataProvider >> {readExpandedEntitySet}");
		if (uriInfo.getExpand() != null) {
			expanding = true;
		}
		BaseDataProviderResponse response = (BaseDataProviderResponse) super.readExpandedEntitySet(uriInfo, context);
		logger.debug("Exiting CXSDataProvider >> {readExpandedEntitySet}");
		return response;
	}

	@Override
	public IDataProviderResponse readExpandedEntity(GetEntityUriInfo uriInfo, ODataContext context)
			throws ODataException {
		logger.debug("Entering CXSDataProvider >> {readExpandedEntity}");
		if (uriInfo.getExpand() != null) {
			expanding = true;
		}
		BaseDataProviderResponse response = (BaseDataProviderResponse) super.readExpandedEntity(uriInfo, context);
		logger.debug("Exiting CXSDataProvider >> {readExpandedEntitySet}");
		return response;
	}

	private void setResponseHeaders(BaseDataProviderResponse resp, Map<String, List<String>> headers) {
		Map<String, String> respHeaders = new HashMap<String, String>();
		if (headers != null && !headers.isEmpty()) {
			for (Map.Entry<String, List<String>> entry : headers.entrySet()) {
				respHeaders.put(entry.getKey(), entry.getValue().toString());
			}
		}
		resp.setHeaders(respHeaders);
	}

	private BaseDataProviderResponse getFunctionResponse(GetFunctionImportUriInfo uriInfo,
			BaseDataProviderResponse response, OperationResponseImpl operationResponse) throws EdmException {

		EdmFunctionImport functionImport = null;
		EdmTyped edmTyped = null;
		functionImport = uriInfo.getFunctionImport();
		edmTyped = functionImport.getReturnType();
		
		//Handling Functionimports without return parameters
        if(edmTyped == null) {
        	return response;
        }
        
		EdmMultiplicity multiplicity = edmTyped.getMultiplicity();
		if (multiplicity == null) {
			multiplicity = EdmMultiplicity.ONE;
		}

		if (edmTyped.getType().getKind() == EdmTypeKind.ENTITY) {
			if (multiplicity == EdmMultiplicity.ONE) {
				if (operationResponse.getEntityData() == null) {
					logger.error("EntityData should be set as response of function import");
					return response;
				} else {
					if (operationResponse.getEntityData().get(0) == null) {
						logger.error("Please set atleast one EntityData in the list");
						return response;
					}
					List<Map<String, Object>> resultList = new ArrayList<>();
					resultList.add(DataConversionUtility.convertToMap(operationResponse.getEntityData().get(0)));
					castStringToGuidWithCompelxProperty(getUUIDColumns((UriInfo) uriInfo), resultList,
							(EdmStructuralType) ((UriInfoImpl) uriInfo).getTargetType());
					response.setResultEntity(resultList.get(0));
					return response;
				}
			}
			if (multiplicity == EdmMultiplicity.MANY) {
				if (operationResponse.getEntityData() == null) {
					logger.error("List of EntityData is expected as response of function import");
					response.setResultEntities(Collections.EMPTY_LIST);
					return response;
				} else {
					if (((UriInfoImpl) uriInfo).getKeyPredicates() == null
							|| ((UriInfoImpl) uriInfo).getKeyPredicates().isEmpty()) {
						List<Map<String, Object>> resultEntities = null;
						resultEntities = (List<Map<String, Object>>) EntityDataResponseHandler
								.getResultEntityFromEntityData(operationResponse.getEntityData());
						response.setResultEntities(resultEntities);

						castStringToGuidWithCompelxProperty(getUUIDColumns((UriInfo) uriInfo), resultEntities,
								(EdmStructuralType) ((UriInfoImpl) uriInfo).getTargetType());

					} else {
						logger.error("Call not supported");
						return response;
					}
					return response;
				}
			}
		}
		if (edmTyped.getType().getKind() == EdmTypeKind.COMPLEX) {
			if (multiplicity == EdmMultiplicity.ONE) {
				if (operationResponse.getComplexData() == null) {
					logger.error("Complex Map Data should be set as response of function import");
					return response;
				} else {
					if (operationResponse.getComplexData().get(0) == null) {
						logger.error("Please set atleast one Complex Map in the list");
						return response;
					}

					List<Map<String, Object>> resultEntities = new ArrayList();
					resultEntities.add((Map<String, Object>) operationResponse.getComplexData().get(0));
					castStringToGuidWithCompelxProperty(getUUIDColumns((UriInfo) uriInfo), resultEntities,
							(EdmStructuralType) ((UriInfoImpl) uriInfo).getTargetType());
					response.setResultProperty(resultEntities.get(0));

					return response;
				}
			}
			if (multiplicity == EdmMultiplicity.MANY) {
				if (operationResponse.getComplexData() == null) {
					logger.error("List of Complex Map Data is expected as response of function import");
					return response;
				} else {

					List<Map<String, Object>> resultEntities = (List<Map<String, Object>>) operationResponse
							.getComplexData();
					response.setResultEntities(resultEntities);
					castStringToGuidWithCompelxProperty(getUUIDColumns((UriInfo) uriInfo), resultEntities,
							(EdmStructuralType) ((UriInfoImpl) uriInfo).getTargetType());

					return response;
				}
			}
		}

		if (edmTyped.getType().getKind() == EdmTypeKind.SIMPLE) {
			if (multiplicity == EdmMultiplicity.ONE) {
				if (operationResponse.getPrimitiveData() == null) {
					logger.error("Primitive Data should be set as response of function import");
					return response;
				} else {
					if (operationResponse.getPrimitiveData().get(0) == null) {
						logger.error("Please set atleast one Primitive Data in the list");
						return response;
					}
					response.setResultProperty(operationResponse.getPrimitiveData().get(0));
					return response;
				}
			}
			if (multiplicity == EdmMultiplicity.MANY) {
				if (operationResponse.getPrimitiveData() == null) {
					logger.error("List of Primitive Data is expected as response of function import");
					return response;
				} else {
					response.setResultProperty(operationResponse.getPrimitiveData());
					Object data = operationResponse.getPrimitiveData();
					response.setResultEntities((List<Map<String, Object>>) data);
					return response;
				}
			}
		}
		return response;
	}

	private static ODataApplicationException convertToODataApplicationException(ErrorResponseImpl e,
			String operation) {
		
		String msg = null;
	
		HttpStatusCodes status = HttpStatusCodes.fromStatusCode(e.getStatusCode()) == null
				? HttpStatusCodes.INTERNAL_SERVER_ERROR
				: HttpStatusCodes.fromStatusCode(e.getStatusCode());
		Throwable cause = e.getCause();
		if (e.getCause() == null) {
			msg = e.getMessage() == null ? "Error in " + operation + " operation"
					: e.getMessage();
			} else {
			msg = e.getMessage() != null ? e.getMessage()
					: e.getCause().getMessage();
			}
		Locale locale = Locale.getDefault();
		String errorCode = e.getErrorCode();
		List<Object> errorDetails = e.getErrorDetails();

		MessageContainer igwMsgCon = null;
		if (errorDetails != null && !errorDetails.isEmpty()) {
			igwMsgCon = new MessageContainer();

			for (Object m : errorDetails) {
				igwMsgCon.addMessage(getIGWMessageFromMessage((Message) m));
			}
		}
		ODataApplicationException exception;

		if (cause == null)
				if (errorCode == null)
					if (igwMsgCon == null)
						exception = new ODataApplicationException(msg, locale, status);
					else
						exception = new ODataApplicationExceptionWithMessageContainer(msg, locale, status, igwMsgCon);
				else if (igwMsgCon == null)
					exception = new ODataApplicationException(msg, locale, status, errorCode);
				else
					exception = new ODataApplicationExceptionWithMessageContainer(msg, locale, status, errorCode,
							igwMsgCon);
			else if (igwMsgCon == null)
				if (errorCode == null)
					exception = new ODataApplicationException(msg, locale, status, cause);
				else
					exception = new ODataApplicationException(msg, locale, status, errorCode, cause);
			else if (errorCode == null)
				exception = new ODataApplicationExceptionWithMessageContainer(msg, locale, status, cause, igwMsgCon);
			else
				exception = new ODataApplicationExceptionWithMessageContainer(msg, locale, status, errorCode, cause,
						igwMsgCon);
		

		return exception;
	}
		
	 
}
