/*******************************************************************************
 * (c) 201X SAP SE or an SAP affiliate company. All rights reserved.
 ******************************************************************************/
package com.sap.cloud.sdk.service.prov.v2.web;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.sdk.service.prov.annotation.repository.AnnotatedClassMethod;
import com.sap.cloud.sdk.service.prov.annotation.repository.AnnotationContainer;
import com.sap.cloud.sdk.service.prov.annotation.repository.AnnotationRepository;
import com.sap.cloud.sdk.service.prov.api.annotations.Function;
import com.sap.cloud.sdk.service.prov.api.operations.Create;
import com.sap.cloud.sdk.service.prov.api.operations.Delete;
import com.sap.cloud.sdk.service.prov.api.operations.Query;
import com.sap.cloud.sdk.service.prov.api.operations.Read;
import com.sap.cloud.sdk.service.prov.api.operations.Update;
import com.sap.cloud.sdk.service.prov.api.util.ClassHelper;
import com.sap.cloud.sdk.service.prov.v2.repository.DPCNameProviderImpl;
import com.sap.cloud.sdk.service.prov.v2.rt.api.dpc.name.provider.DPCNameFacade;
import com.sap.cloud.sdk.service.prov.v2.rt.core.RuntimeDelegate;
import com.sap.gateway.core.api.delegate.IServiceDelegate;
import com.sap.gateway.core.api.delegate.RuntimeDelegateFacade;
import com.sap.gateway.core.api.delegate.ServiceType;
import com.sap.gateway.core.api.exception.TechnicalException;
import com.sap.gateway.core.api.srvrepo.ServiceRepositoryProviderFacade;


public class ServiceInitializer<T> implements ServletContextListener {
	
	final static Logger log=LoggerFactory.getLogger(ServiceInitializer.class);
	
	private static final String GET_INSTANCE_METHOD = "getInstance";
	private static final String SERVICE_REPOSITORY_CLASS_NAME = "com.sap.cloud.sdk.service.prov.v2.rt.repository.ServiceRepositoryProvider";
	private static final String SERVICE_DELEGATE_CLASS_NAME = "com.sap.cloud.sdk.service.prov.v2.rt.repository.delegate.ServiceDelegate";
	private static final String RUNTIME_DELEGATE = "com.sap.cloud.sdk.service.prov.v2.rt.cdx.CDXRuntimeDelegate";
	private static final String DPC = "com.sap.cloud.sdk.service.prov.v2.data.provider.CXSDataProvider";
	private static final String MPC = "com.sap.cloud.sdk.service.prov.v2.rt.cdx.CDXEdmProvider";
	
	@Override
	public void contextInitialized(ServletContextEvent sce) {
		log.debug("Initializing Context");
		ServletContext context = sce.getServletContext();
		String packageName = context.getInitParameter("package");
		initializeServiceRepository();
		initializeRuntimeDelegate();
		registerSupportedODataAnnotations();
		registerUserProvidedAnnotations(packageName);
		log.debug("Successfully initialized context");
	}
	

	private void initializeRuntimeDelegate() {
		try{
			RuntimeDelegate delegate = (RuntimeDelegate)Class.forName(RUNTIME_DELEGATE).newInstance();
			delegate.setMPC(MPC);
			delegate.setDPC(DPC);
			RuntimeDelegateFacade.registerDelegate("com.sap.cloud.sdk.service.prov.v2.data.provider", "CXSDATAPROVIDER", delegate);
		} catch (IllegalAccessException | InstantiationException |ClassNotFoundException e) {
			log.error("Error Initializing Runtime Delegate",e);
		}
	}

	private void initializeServiceRepository() {
		try {
			DPCNameFacade.setDpcNameProvider(new DPCNameProviderImpl());
			Class<?> cls = Class.forName(SERVICE_DELEGATE_CLASS_NAME);
			Method method = cls.getDeclaredMethod(GET_INSTANCE_METHOD);
			IServiceDelegate idelegate = (IServiceDelegate) method.invoke(null);
			ServiceRepositoryProviderFacade.setServiceRepository(idelegate.getService(ServiceType.SERVICE_REPOSITORY, SERVICE_REPOSITORY_CLASS_NAME));
		} catch (IllegalAccessException | NoSuchMethodException | InvocationTargetException | TechnicalException | ClassNotFoundException e) {
			log.error("Error initializing Service Repository",e);
		}
	}
	
	private void registerSupportedODataAnnotations() {
		AnnotationContainer.getInstance().addAnnotation(Query.class);
		AnnotationContainer.getInstance().addAnnotation(Create.class);
		AnnotationContainer.getInstance().addAnnotation(Update.class);
		AnnotationContainer.getInstance().addAnnotation(Read.class);
		AnnotationContainer.getInstance().addAnnotation(Delete.class);
		AnnotationContainer.getInstance().addAnnotation(Function.class);
	}


	@SuppressWarnings("unchecked")
	private void registerUserProvidedAnnotations(String packageName) {
		logDebug("registering DPC Class started.....");
		List<Class<?>> classes = null;
		try {
			classes = ClassHelper.loadClasses(packageName,c -> {
			      Method[] m = c.getMethods();
			      return Arrays.stream(m)
			          .findFirst()
			          .isPresent();
			});
			classes.parallelStream().forEach((clazz) -> {
				List<Method> methods = Arrays.asList((((Class<T>)clazz).getDeclaredMethods()));
				List<Class<?>> v4annotationClasses = AnnotationContainer.getInstance().getAnnotationList();
				v4annotationClasses.stream().forEach(_v4AnnoClazz -> {_v4AnnoClazz = (Class<? extends Annotation>) _v4AnnoClazz;
				Class<? extends Annotation> tmp = (Class<? extends Annotation>) _v4AnnoClazz;
				methods.stream().filter((method)-> method.isAnnotationPresent(tmp)
						).forEach(method->{
							AnnotatedClassMethod<T> annotatedClassMethod = new AnnotatedClassMethod();
							annotatedClassMethod.setUserClass((Class<T>)clazz);
							annotatedClassMethod.setMethod(method);
							AnnotationRepository.getInstance().registerAnnotationClassMethod(tmp.getSimpleName(),annotatedClassMethod);
							if(log.isDebugEnabled()) {log.debug("Registering DPC for Annotation: "+tmp.getSimpleName()+". Method name is: "+annotatedClassMethod.getMethod().getName()+".");}
						}
					);
				});
			 }
		 );
		} catch (Exception e) {
			log.error("registering annotation failed: "+e.getMessage(), e);
		}
		logDebug("registering DPC Class done successfully");
		
	}

	@Override
	public void contextDestroyed(ServletContextEvent sce) {
	}
	
	private void logDebug(String logInfo){
			log.debug(logInfo);
	}
}
