package com.sap.xs.env;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.fasterxml.jackson.annotation.JsonAnyGetter;
import com.fasterxml.jackson.annotation.JsonAnySetter;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;

public class VcapApplication {
	public static final String VCAP_APPLICATION = "VCAP_APPLICATION";

	private static final Logger LOGGER = Logger.getLogger(VcapApplication.class.getName());
	private static final ObjectReader READER = new ObjectMapper().readerFor(VcapApplication.class);

	@JsonProperty("application_id")
	private String applicationId;

	@JsonProperty("application_name")
	private String applicationName;
	@JsonProperty("name")
	private String name;

	@JsonProperty("application_uris")
	private List<String> applicationUris;
	@JsonProperty("uris")
	private List<String> uris;

	@JsonProperty("application_version")
	private String applicationVersion;
	@JsonProperty("version")
	private String version;

	@JsonProperty("host")
	private String host;

	@JsonProperty("instance_id")
	private String instanceId;

	@JsonProperty("instance_index")
	private String instanceIndex;

	@JsonProperty("limits")
	private Limits limits;

	@JsonProperty("port")
	private String port;

	@JsonProperty("space_id")
	private String spaceId;

	@JsonProperty("space_name")
	private String spaceName;

	@JsonProperty("start")
	private String start;
	@JsonProperty("started_at")
	private String startedAt;

	@JsonProperty("started_at_timestamp")
	private String startedAtTimestamp;
	@JsonProperty("state_timestamp")
	private String stateTimestamp;

	@JsonProperty("users")
	private List<String> users;

	@JsonProperty("organization_name")
	private String organizationName;

	private Map<String, Object> any = new HashMap<>();

	/**
	 * @return map of properties that are not predefined
	 */
	@JsonAnyGetter
	public Map<String, Object> any() {
		return any;
	}

	/**
	 * @param name
	 *            name of property to set
	 * @param value
	 *            value of property to set
	 */
	@JsonAnySetter
	public void set(String name, Object value) {
		any.put(name, value);
	}

	/**
	 * Tries to build VcapApplication from the VCAP_APPLICATION environment
	 * variable. In case the environment variable is not set an empty instance
	 * is returned. To check if the environment variable exists use
	 * {@code System.getenv().containsKey(VCAP_APPLICATION)}.
	 *
	 * @return VcapApplication instance
	 * @throws IllegalArgumentException
	 *             In case the environment variable does not conform to
	 *             specification.
	 *
	 */
	public static VcapApplication fromEnvironment() {
		return VcapApplication.from(System.getenv(VCAP_APPLICATION));
	}

	/**
	 * Tries to build VcapApplication from the VCAP_APPLICATION system property.
	 * In case the system property is not set an empty instance is returned. To
	 * check if the system property exists use
	 * {@code System.getProperty(VCAP_APPLICATION) != null}.
	 *
	 * @return VcapApplication build from the VCAP_APPLICATION system property
	 * @throws IllegalArgumentException
	 *             In case the system property not conform to specification.
	 *
	 */
	public static VcapApplication fromSystemProperty() {
		return VcapApplication.from(System.getProperty(VCAP_APPLICATION));
	}

	/**
	 * @param application
	 *            the application json
	 * @return VcapApplication
	 * @throws IllegalArgumentException
	 *             In case application does not conform to specification.
	 *
	 */
	public static VcapApplication from(String application) {
		if (application == null || application.trim().isEmpty()) {
			return new VcapApplication();
		}

		if (LOGGER.isLoggable(Level.FINE)) {
			LOGGER.fine("Parsing VCAP_APPLICATION: " + application);
		}

		try {
			VcapApplication vcapApplication = READER.readValue(application);

			if (LOGGER.isLoggable(Level.FINE)) {
				LOGGER.fine("Parsed VCAP_APPLICATION: " + vcapApplication.toString());
			}

			return vcapApplication;
		} catch (IOException e) {
			LOGGER.log(Level.FINE, "", e);
			throw new IllegalArgumentException("Cannot parse VCAP_APPLICATION.");
		}
	}

	/**
	 * @return application id
	 */
	public String getApplicationId() {
		return applicationId;
	}

	/**
	 * @return application name
	 */
	public String getApplicationName() {
		return applicationName == null ? name : applicationName;
	}

	/**
	 * @return list of application uris
	 */
	public List<String> getApplicationUris() {
		return applicationUris == null ? uris : applicationUris;
	}

	/**
	 * @return application version
	 */
	public String getApplicationVersion() {
		return applicationVersion == null ? version : applicationVersion;
	}

	/**
	 * @return application host
	 */
	public String getHost() {
		return host;
	}

	/**
	 * @return application instance id
	 */
	public String getInstanceId() {
		return instanceId;
	}

	/**
	 * @return application instance index
	 */
	public String getInstanceIndex() {
		return instanceIndex;
	}

	/**
	 * @return application {@link Limits}
	 */
	public Limits getLimits() {
		return limits;
	}

	/**
	 * @return application port
	 */
	public String getPort() {
		return port;
	}

	/**
	 * @return application space id
	 */
	public String getSpaceId() {
		return spaceId;
	}

	/**
	 * @return application space name
	 */
	public String getSpaceName() {
		return spaceName;
	}

	/**
	 * @return application start time
	 */
	public String getStart() {
		return start == null ? startedAt : start;
	}

	/**
	 * @return timestamp of application start time
	 */
	public String getStartedAtTimestamp() {
		return startedAtTimestamp == null ? stateTimestamp : startedAtTimestamp;
	}

	/**
	 * @return list of application users
	 */
	public List<String> getUsers() {
		return users;
	}

	/**
	 * @return application organization name
	 */
	public String getOrganizationName() {
		return organizationName;
	}

	/**
	 * @param name
	 *            name of {@link VcapApplication} property
	 * @return value of the property
	 */
	public Object get(String name) {
		switch (name) {
		case "application_id":
			return getApplicationId();
		case "application_name":
		case "name":
			return getApplicationName();
		case "application_uris":
		case "uris":
			return getApplicationUris();
		case "application_version":
		case "version":
			return getApplicationVersion();
		case "host":
			return getHost();
		case "instance_id":
			return getInstanceId();
		case "instance_index":
			return getInstanceIndex();
		case "limits":
			return getLimits();
		case "port":
			return getPort();
		case "space_id":
			return getSpaceId();
		case "space_name":
			return getSpaceName();
		case "start":
		case "started_at":
			return getStart();
		case "started_at_timestamp":
		case "state_timestamp":
			return getStartedAtTimestamp();
		case "users":
			return getUsers();
		case "organization_name":
			return getOrganizationName();
		default:
			return any.get(name);
		}
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		sb.append("{");
		sb.append("\"application_id\": \"").append(getApplicationId()).append("\"");
		sb.append(", \"application_name\": \"").append(getApplicationName()).append("\"");
		sb.append(", \"application_uris\": ").append(StringUtil.listAsJson(getApplicationUris()));
		sb.append(", \"application_version\": \"").append(getApplicationVersion()).append("\"");
		sb.append(", \"host\": \"").append(getHost()).append("\"");
		sb.append(", \"instance_id\": \"").append(getInstanceId()).append("\"");
		sb.append(", \"instance_index\": ").append(getInstanceIndex());
		if (getLimits() != null) {
			sb.append(", \"limits\": ").append(getLimits());
		}
		sb.append(", \"port\": \"").append(getPort()).append("\"");
		sb.append(", \"space_id\": \"").append(getSpaceId()).append("\"");
		sb.append(", \"space_name\": \"").append(getSpaceName()).append("\"");
		sb.append(", \"start\": \"").append(getStart()).append("\"");
		sb.append(", \"started_at_timestamp\": \"").append(getStartedAtTimestamp()).append("\"");
		sb.append(", \"users\": ").append(StringUtil.listAsJson(getUsers()));
		sb.append(", \"organization_name\": \"").append(getOrganizationName()).append("\"");
		sb.append("}");

		return sb.toString();
	}
}
