package com.sap.xs.env;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;

public class VcapServices {

	public static final String VCAP_SERVICES = "VCAP_SERVICES";

	private static final Logger LOGGER = Logger.getLogger(VcapServices.class.getName());
	private static final ObjectReader READER = new ObjectMapper()
			.readerFor(new TypeReference<Map<String, List<Service>>>(){});

	private final List<Service> services = new ArrayList<>();

	public VcapServices() {}

	private VcapServices(Map<String, List<Service>> map) {
		map.values().forEach(services::addAll);
	}

	/**
	 * Tries to build VcapServices from the VCAP_SERVICES environment variable.
	 * In case the environment variable is not set an empty instance is
	 * returned. To check if the environment variable exists use
	 * {@code System.getenv().containsKey(VCAP_SERVICES)}.
	 *
	 * @return VcapServices build from the VCAP_SERVICES environment variable
	 * @throws IllegalArgumentException
	 *             In case the environment variable does not conform to
	 *             specification.
	 *
	 */
	public static VcapServices fromEnvironment() {
		return VcapServices.from(System.getenv(VCAP_SERVICES));
	}

	/**
	 * Tries to build VcapServices from the VCAP_SERVICES system property. In
	 * case the system property is not set an empty instance is returned. To
	 * check if the system property exists use
	 * {@code System.getProperty(VCAP_SERVICES) != null}.
	 *
	 * @return VcapServices build from the VCAP_SERVICES system property
	 * @throws IllegalArgumentException
	 *             In case the system property is null or does not conform to
	 *             specification.
	 *
	 */
	public static VcapServices fromSystemProperty() {
		return VcapServices.from(System.getProperty(VCAP_SERVICES));
	}

	/**
	 * @param services
	 *            the services json
	 * @return VcapServices
	 * @throws IllegalArgumentException
	 *             In case services does not conform to specification.
	 *
	 */
	public static VcapServices from(String services) {
		if (services == null || services.trim().isEmpty()) {
			return new VcapServices();
		}

		LOGGER.fine(()->"Parsing VCAP_SERVICES: " + StringUtil.hidePasswords(services));

		try {
			VcapServices vcapServices = new VcapServices(READER.readValue(services));
			LOGGER.fine(()->"Parsed VCAP_SERVICES: " + vcapServices.toString());
			return vcapServices;
		} catch (IOException e) {
			 // It is possible for this exception to contain sensitive data
			LOGGER.throwing("VcapServices", "from", e);
			throw new IllegalArgumentException("Cannot parse VCAP_SERVICES.");
		}
	}

	/**
	 * Gets service credentials.
	 *
	 * @param serviceName
	 *            name of the service which credentials you would like to get.
	 * @return credentials for the service with name <code>serviceName</code>.
	 *         If such service is not available in the service list, it returns
	 *         <code>null</code>.
	 */
	public Credentials getCredentials(String serviceName) {
		final Optional<Service> service = getService(serviceName);
		if (service.isPresent()) {
			return service.get().getCredentials();
		}
		return null;
	}

	/**
	 * Gets service with the specified name.
	 *
	 * @param serviceName
	 *            name of the service you would like to get.
	 * @return service with name <code>serviceName</code>. If such service is
	 *         not available in the service list, it returns <code>null</code>.
	 */
	public Optional<Service> getService(String serviceName) {
		return services.stream().filter(s -> serviceName.equals(s.getServiceInstanceName())).findFirst();
	}

	/**
	 *
	 * @return Returns string representation of service names
	 */
	public String getServiceNames() {
		return getServices().toString();
	}

	/**
	 *
	 * @return Returns list of service names
	 */
	public List<String> getServices() {
		List<String> names = new ArrayList<>();
		for (Service service : services) {
			names.add(service.getName());
		}
		return Collections.unmodifiableList(names);
	}

	/**
	 * Compares name, label, tags of each service to the given
	 * <code>filter</code> ,checks if <code>credentialsAttribute</code> starts
	 * with <code>credentialsAttributePrefix</code>. The method returns true if
	 * any services matches. If no service matches, returns false.
	 *
	 * @param filter
	 *            the Regular Expression to match against the name, label, and
	 *            tags of the services.
	 * @param credentialsAttribute
	 *            the credentials attribute.
	 * @param credentialsAttributePrefix
	 *            the prefix value of the credentials attribute for check.
	 * @return true if service matches criteria
	 */
	public boolean isServiceBound(String filter, String credentialsAttribute, String credentialsAttributePrefix) {
		return !findServices(filter, credentialsAttribute, credentialsAttributePrefix, false).isEmpty();
	}

	/**
	 * Compares <code>serviceAttribute</code> of each service to the given
	 * <code>filter</code>. The method returns true if any services matches. If
	 * no service matches, returns false.
	 *
	 * @param filter
	 *            the Regular Expression to match against the
	 *            <code>serviceAttribute</code> of the services.
	 * @param serviceAttribute
	 *            service attribute
	 * @return true if service matches criteria
	 */
	public boolean isServiceBound(String filter, ServiceAttribute serviceAttribute) {
		return !findServices(filter, serviceAttribute).isEmpty();
	}

	/**
	 * Compares name, label, tags of each service to the given
	 * <code>filter</code>, checks if <code>credentialsAttribute</code> starts
	 * with <code>credentialsAttributePrefix</code>. The method returns the
	 * first service that matches. If no service matches, returns
	 * <code>null</code>.
	 *
	 * @param filter
	 *            the Regular Expression to match against the name, label, and
	 *            tags of the services.
	 * @param credentialsAttribute
	 *            the credentials attribute.
	 * @param credentialsAttributePrefix
	 *            the prefix value of the credentials attribute for check.
	 *
	 * @return a Map with the first service that matches. If no service matches,
	 *         returns null.
	 */
	public Service findService(String filter, String credentialsAttribute, String credentialsAttributePrefix) {
		List<Service> result = findServices(filter, credentialsAttribute, credentialsAttributePrefix, false);

		if (result.isEmpty()) {
			return null;
		}

		return result.iterator().next();
	}

	/**
	 * Checks if there a service which name, label or tags contain the
	 * <code>filter</code> with the <code>requiredAttributes</code> If
	 * <code>requiredAttributes</code> is a list of group of attributes the
	 * service will match the criteria if at least one attribute of each group
	 * is available in the credentials of the service.
	 *
	 * @param filter
	 *            the Regular Expression to match against the name of the
	 *            services.
	 * @param requiredAttributes
	 *            List of keys or group of keys that should be part of the
	 *            Service's credentials
	 *
	 * @return <code>Service</code> if only one service that matches the
	 *         criteria is bound to the application. Otherwise <code>null</code>
	 */
	@SuppressWarnings("unchecked")
	public Service findOneService(String filter, List<String> ... requiredAttributes) {
		List<Service> filteredServices = findServices(filter);
		if (filteredServices.isEmpty() || filteredServices.size() > 1) {
			return null;
		} else {
			Service service = filteredServices.get(0);
			Credentials credentials = service.getCredentials();
			return (credentials != null && credentials.hasRequiredAttributes(requiredAttributes)) ? service : null;
		}
	}

	/**
	 * Checks if there are services whose name, label, or tags contain the <code>filter</code>.
	 * The <code>credentialsAttributes</code> represents a list of groups of attributes and at least one attribute
	 * of each group should be contained in the credentials of the service.
	 * The method returns all services that match. If no service matches, it returns <code>null</code>.
	 *
	 * @param filter
	 *            the Regular Expression to match against the name, label, and tags of the
	 *            services.
	 * @param credentialsAttributes
	 *            List of keys or group of keys that should be part of the
	 *            Service's credentials.
	 *
	 * @return a List with all services that match. If no service matches, returns <code>null</code>.
	 */
	@SuppressWarnings("unchecked")
	public List<Service> findServices(String filter, List<String> ... credentialsAttributes) {
		if (filter == null || filter.isEmpty()) {
			LOGGER.fine(()->"findServices: filter is null/empty.");
			return null;
		}

		List<Service> filteredServices = new ArrayList<>();
		for (Service service : services) {
			LOGGER.fine(()->"findServices \"" + filter + "\" checking service " + serviceLogString(service));
			if (containsMatcher(filter, service)) {
				Credentials credentials = service.getCredentials();
				if (credentials != null && credentials.hasRequiredAttributes(credentialsAttributes)) {
					filteredServices.add(service);
				}
			}
		}

		return filteredServices;
	}
	/**
	 * Compares name, label, tags of each service to the given
	 * <code>filter</code>, checks if <code>credentialsAttribute</code> starts
	 * with <code>credentialsAttributePrefix</code>. The method returns all
	 * services that match. If no service matches, returns <code>null</code>.
	 *
	 * @param filter
	 *            the Regular Expression to match against the name, label, and
	 *            tags of the services.
	 * @param credentialsAttribute
	 *            the credentials attribute.
	 * @param credentialsAttributePrefix
	 *            the prefix value of the credentials attribute for check.
	 *
	 * @return a Map with all services that matches. If no service matches,
	 *         returns null.
	 */
	public List<Service> findServices(String filter, String credentialsAttribute, String credentialsAttributePrefix) {
		return findServices(filter, credentialsAttribute, credentialsAttributePrefix, true);
	}

	/**
	 * This method finds a service by keyword in a specific service attribute.
	 *
	 * @param filter
	 *            the keyword to search for
	 * @param attribute
	 *            attribute of the service (name, tags, label, plan)
	 * @return a list of services that matches criteria. If such service is not
	 *         available, it returns an empty list.
	 */
	public ArrayList<Service> findServices(String filter, ServiceAttribute attribute) {
		Pattern pattern = Pattern.compile(filter);
		ArrayList<Service> result = new ArrayList<>();

		for (Service service : services) {
			LOGGER.fine(()->"findServices \"" + filter + "\" checking service " + serviceLogString(service));
			if (matcherAttribute(pattern, service, attribute)) {
				result.add(service);

			}
		}

		return result;
	}

	@Override
	public String toString() {
		StringJoiner sj = new StringJoiner(",", "[", "]");
		services.forEach(s -> sj.add(serviceLogString(s)));
		return sj.toString();
	}

	private List<Service> findServices(String filter) {
		ArrayList<Service> result = new ArrayList<>();

		if (filter == null || filter.isEmpty()) {
			LOGGER.fine(()->"findServices: filter is null/empty.");
			return result;
		}

		for (Service service : services) {
			LOGGER.fine(()->"findServices \"" + filter + "\" checking service " + serviceLogString(service));
			if (containsMatcher(filter, service)) {
				result.add(service);
			}
		}

		return result;
	}

	private boolean containsMatcher(String filter, Service service) {
		String name = service.getName();
		if (name != null && name.contains(filter)) {
			LOGGER.fine(()->"containsMatcher \"" + filter + "\" returns true. name matches.");
			return true;
		}

		String label = service.getLabel();
		if (label != null && label.contains(filter)) {
			LOGGER.fine(()->"containsMatcher \"" + filter + "\" returns true. label matches.");
			return true;
		}

		for (String tag : service.getTags()) {
			if (tag.contains(filter)) {
				LOGGER.fine(()->"containsMatcher \"" + filter + "\" returns true. tag matches.");
				return true;
			}
		}

		return false;
	}

	private List<Service> findServices(String filter, String credentialsAttribute, String credentialsAttributePrefix, boolean all) {
		Pattern pattern = Pattern.compile(filter);
		List<Service> result = new ArrayList<>();

		for (Service service : services) {
			LOGGER.fine(()->"findServices \"" + filter + "\" checking service " + serviceLogString(service));
			if (matcher(pattern, service) ||
					stringMatcher(service.getCredentials(), credentialsAttribute, credentialsAttributePrefix)) {
				result.add(service);
				if (!all) {
					break;
				}
			}
		}

		return result;
	}

	private boolean matcherAttribute(Pattern pattern, Service service, ServiceAttribute attribute) {
		if (attribute.equals(ServiceAttribute.NAME)) {
			String name = service.getName();
			if (name != null && pattern.matcher(name).matches()) {
				LOGGER.fine(()->"matcherAttribute \"" + pattern + "\" matched name \"" + name + "\".");
				return true;
			}
		}

		if (attribute.equals(ServiceAttribute.LABEL)) {
			String label = service.getLabel();
			if (label != null && pattern.matcher(label).matches()) {
				LOGGER.fine(()->"matcherAttribute \"" + pattern + "\" matched label \"" + label + "\".");
				return true;
			}
		}

		if (attribute.equals(ServiceAttribute.TAGS)) {
			for (String tag : service.getTags()) {
				if (pattern.matcher(tag).matches()) {
					LOGGER.fine(()->"matcherAttribute \"" + pattern + "\" matched tag \"" + tag + "\".");
					return true;
				}
			}
		}

		return false;
	}

	private boolean matcher(Pattern pattern, Service service) {
		String name = service.getName();
		if (name != null && pattern.matcher(name).matches()) {
			LOGGER.fine(()->"matcher \"" + pattern + "\" matched name \"" + name + "\".");
			return true;
		}

		String label = service.getLabel();
		if (label != null && pattern.matcher(label).matches()) {
			LOGGER.fine(()->"matcher \"" + pattern + "\" matched label \"" + label + "\".");
			return true;
		}

		for (String tag : service.getTags()) {
			if (pattern.matcher(tag).matches()) {
				LOGGER.fine(()->"matcher \"" + pattern + "\" matched tag \"" + tag + "\".");
				return true;
			}
		}

		return false;
	}

	private boolean stringMatcher(Credentials credentials, String attribute, String attributePrefix) {
		if (credentials == null || attribute == null)
			return false;

		Object attributeValue = credentials.get(attribute);
		if (attributeValue != null && attributeValue.toString().startsWith(attributePrefix)) {
			LOGGER.fine(()->"stringMatcher \"" + attribute + "\"/\"" + attributePrefix + "\" matched.");
			return true;
		}

		return false;
	}

	private String serviceLogString(Service service) {
		StringBuilder sb = new StringBuilder();

		sb.append("{");
		sb.append("\"name\":").append(service.getName()).append("\"");
		sb.append(", \"label\"=\"").append(service.getLabel()).append("\"");
		sb.append(", \"tags\":").append(StringUtil.listAsJson(service.getTags()));
		sb.append("}");

		return sb.toString();
	}
}
