/*
 * Decompiled with CFR 0.152.
 */
package com.sap.xs.env;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.sap.xs.env.Credentials;
import com.sap.xs.env.Service;
import com.sap.xs.env.ServiceAttribute;
import com.sap.xs.env.StringUtil;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.logging.Logger;
import java.util.regex.Pattern;

public class VcapServices {
    public static final String VCAP_SERVICES = "VCAP_SERVICES";
    private static final Logger LOGGER = Logger.getLogger(VcapServices.class.getName());
    private static final ObjectReader READER = new ObjectMapper().readerFor((TypeReference)new TypeReference<Map<String, List<Service>>>(){});
    private final List<Service> services = new ArrayList<Service>();

    public VcapServices() {
    }

    private VcapServices(Map<String, List<Service>> map) {
        map.values().forEach(this.services::addAll);
    }

    public static VcapServices fromEnvironment() {
        return VcapServices.from(System.getenv(VCAP_SERVICES));
    }

    public static VcapServices fromSystemProperty() {
        return VcapServices.from(System.getProperty(VCAP_SERVICES));
    }

    public static VcapServices from(String services) {
        if (services == null || services.trim().isEmpty()) {
            return new VcapServices();
        }
        LOGGER.fine(() -> "Parsing VCAP_SERVICES: " + StringUtil.hidePasswords(services));
        try {
            VcapServices vcapServices = new VcapServices((Map)READER.readValue(services));
            LOGGER.fine(() -> "Parsed VCAP_SERVICES: " + vcapServices.toString());
            return vcapServices;
        }
        catch (IOException e) {
            LOGGER.throwing("VcapServices", "from", e);
            throw new IllegalArgumentException("Cannot parse VCAP_SERVICES.");
        }
    }

    public Credentials getCredentials(String serviceName) {
        Optional<Service> service = this.getService(serviceName);
        if (service.isPresent()) {
            return service.get().getCredentials();
        }
        return null;
    }

    public Optional<Service> getService(String serviceName) {
        return this.services.stream().filter(s -> serviceName.equals(s.getServiceInstanceName())).findFirst();
    }

    public String getServiceNames() {
        return this.getServices().toString();
    }

    public List<String> getServices() {
        ArrayList<String> names = new ArrayList<String>();
        for (Service service : this.services) {
            names.add(service.getName());
        }
        return Collections.unmodifiableList(names);
    }

    public boolean isServiceBound(String filter, String credentialsAttribute, String credentialsAttributePrefix) {
        return !this.findServices(filter, credentialsAttribute, credentialsAttributePrefix, false).isEmpty();
    }

    public boolean isServiceBound(String filter, ServiceAttribute serviceAttribute) {
        return !this.findServices(filter, serviceAttribute).isEmpty();
    }

    public Service findService(String filter, String credentialsAttribute, String credentialsAttributePrefix) {
        List<Service> result = this.findServices(filter, credentialsAttribute, credentialsAttributePrefix, false);
        if (result.isEmpty()) {
            return null;
        }
        return result.iterator().next();
    }

    public Service findOneService(String filter, List<String> ... requiredAttributes) {
        List<Service> filteredServices = this.findServices(filter);
        if (filteredServices.isEmpty() || filteredServices.size() > 1) {
            return null;
        }
        Service service = filteredServices.get(0);
        Credentials credentials = service.getCredentials();
        return credentials != null && credentials.hasRequiredAttributes(requiredAttributes) ? service : null;
    }

    public List<Service> findServices(String filter, List<String> ... credentialsAttributes) {
        if (filter == null || filter.isEmpty()) {
            LOGGER.fine(() -> "findServices: filter is null/empty.");
            return null;
        }
        ArrayList<Service> filteredServices = new ArrayList<Service>();
        for (Service service : this.services) {
            Credentials credentials;
            LOGGER.fine(() -> "findServices \"" + filter + "\" checking service " + this.serviceLogString(service));
            if (!this.containsMatcher(filter, service) || (credentials = service.getCredentials()) == null || !credentials.hasRequiredAttributes(credentialsAttributes)) continue;
            filteredServices.add(service);
        }
        return filteredServices;
    }

    public List<Service> findServices(String filter, String credentialsAttribute, String credentialsAttributePrefix) {
        return this.findServices(filter, credentialsAttribute, credentialsAttributePrefix, true);
    }

    public ArrayList<Service> findServices(String filter, ServiceAttribute attribute) {
        Pattern pattern = Pattern.compile(filter);
        ArrayList<Service> result = new ArrayList<Service>();
        for (Service service : this.services) {
            LOGGER.fine(() -> "findServices \"" + filter + "\" checking service " + this.serviceLogString(service));
            if (!this.matcherAttribute(pattern, service, attribute)) continue;
            result.add(service);
        }
        return result;
    }

    public String toString() {
        StringJoiner sj = new StringJoiner(",", "[", "]");
        this.services.forEach(s -> sj.add(this.serviceLogString((Service)s)));
        return sj.toString();
    }

    private List<Service> findServices(String filter) {
        ArrayList<Service> result = new ArrayList<Service>();
        if (filter == null || filter.isEmpty()) {
            LOGGER.fine(() -> "findServices: filter is null/empty.");
            return result;
        }
        for (Service service : this.services) {
            LOGGER.fine(() -> "findServices \"" + filter + "\" checking service " + this.serviceLogString(service));
            if (!this.containsMatcher(filter, service)) continue;
            result.add(service);
        }
        return result;
    }

    private boolean containsMatcher(String filter, Service service) {
        String name = service.getName();
        if (name != null && name.contains(filter)) {
            LOGGER.fine(() -> "containsMatcher \"" + filter + "\" returns true. name matches.");
            return true;
        }
        String label = service.getLabel();
        if (label != null && label.contains(filter)) {
            LOGGER.fine(() -> "containsMatcher \"" + filter + "\" returns true. label matches.");
            return true;
        }
        for (String tag : service.getTags()) {
            if (!tag.contains(filter)) continue;
            LOGGER.fine(() -> "containsMatcher \"" + filter + "\" returns true. tag matches.");
            return true;
        }
        return false;
    }

    private List<Service> findServices(String filter, String credentialsAttribute, String credentialsAttributePrefix, boolean all) {
        Pattern pattern = Pattern.compile(filter);
        ArrayList<Service> result = new ArrayList<Service>();
        for (Service service : this.services) {
            LOGGER.fine(() -> "findServices \"" + filter + "\" checking service " + this.serviceLogString(service));
            if (!this.matcher(pattern, service) && !this.stringMatcher(service.getCredentials(), credentialsAttribute, credentialsAttributePrefix)) continue;
            result.add(service);
            if (all) continue;
            break;
        }
        return result;
    }

    private boolean matcherAttribute(Pattern pattern, Service service, ServiceAttribute attribute) {
        String label;
        String name;
        if (attribute.equals((Object)ServiceAttribute.NAME) && (name = service.getName()) != null && pattern.matcher(name).matches()) {
            LOGGER.fine(() -> "matcherAttribute \"" + pattern + "\" matched name \"" + name + "\".");
            return true;
        }
        if (attribute.equals((Object)ServiceAttribute.LABEL) && (label = service.getLabel()) != null && pattern.matcher(label).matches()) {
            LOGGER.fine(() -> "matcherAttribute \"" + pattern + "\" matched label \"" + label + "\".");
            return true;
        }
        if (attribute.equals((Object)ServiceAttribute.TAGS)) {
            for (String tag : service.getTags()) {
                if (!pattern.matcher(tag).matches()) continue;
                LOGGER.fine(() -> "matcherAttribute \"" + pattern + "\" matched tag \"" + tag + "\".");
                return true;
            }
        }
        return false;
    }

    private boolean matcher(Pattern pattern, Service service) {
        String name = service.getName();
        if (name != null && pattern.matcher(name).matches()) {
            LOGGER.fine(() -> "matcher \"" + pattern + "\" matched name \"" + name + "\".");
            return true;
        }
        String label = service.getLabel();
        if (label != null && pattern.matcher(label).matches()) {
            LOGGER.fine(() -> "matcher \"" + pattern + "\" matched label \"" + label + "\".");
            return true;
        }
        for (String tag : service.getTags()) {
            if (!pattern.matcher(tag).matches()) continue;
            LOGGER.fine(() -> "matcher \"" + pattern + "\" matched tag \"" + tag + "\".");
            return true;
        }
        return false;
    }

    private boolean stringMatcher(Credentials credentials, String attribute, String attributePrefix) {
        if (credentials == null || attribute == null) {
            return false;
        }
        Object attributeValue = credentials.get(attribute);
        if (attributeValue != null && attributeValue.toString().startsWith(attributePrefix)) {
            LOGGER.fine(() -> "stringMatcher \"" + attribute + "\"/\"" + attributePrefix + "\" matched.");
            return true;
        }
        return false;
    }

    private String serviceLogString(Service service) {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        sb.append("\"name\":").append(service.getName()).append("\"");
        sb.append(", \"label\"=\"").append(service.getLabel()).append("\"");
        sb.append(", \"tags\":").append(StringUtil.listAsJson(service.getTags()));
        sb.append("}");
        return sb.toString();
    }
}

