/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.spring.boot.mt.config;

import com.sap.cds.mtx.impl.Authenticator;
import com.sap.cloud.mt.subscription.DynamicHdiDeploymentParameters;
import com.sap.cloud.mt.subscription.HdiContainerManager;
import com.sap.cloud.mt.subscription.ProvisioningService;
import com.sap.cloud.mt.subscription.ServiceSpecification;
import com.sap.cloud.mt.subscription.SidecarAccess;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exits.UserProvidedSchemasExit;
import com.sap.cloud.mt.tools.api.RequestEnhancer;
import com.sap.cloud.mt.tools.api.ResilienceConfig;
import com.sap.cloud.sdk.cloudplatform.connectivity.AuthenticationType;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultDestinationLoader;
import com.sap.cloud.sdk.cloudplatform.connectivity.DefaultHttpDestination;
import com.sap.cloud.sdk.cloudplatform.connectivity.DestinationAccessor;
import com.sap.cloud.sdk.cloudplatform.connectivity.HttpDestination;
import com.sap.cloud.sdk.cloudplatform.security.BasicCredentials;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHeaders;

/**
 * Configuration bean that setups a tenant provider, a data source lookup and a routing data source bean
 */
public abstract class AbstractSubscriberConfig {
    private final DefaultDestinationLoader loader;

    protected AbstractSubscriberConfig() {
        this.loader = new DefaultDestinationLoader();
        DestinationAccessor.prependDestinationLoader(loader);
    }


    protected RequestEnhancer getAuthenticationEnhancer(Authenticator authenticator) {
        return request -> {
            if (authenticator != null && authenticator.getAuthorization().isPresent()) {
                request.addHeader(HttpHeaders.AUTHORIZATION, authenticator.getAuthorization().get());
            }
        };
    }

    protected SidecarAccess createSidecar(Config config, Authenticator authenticator) throws InternalError {
        ServiceSpecification sidecarSpecification = null;
        if (StringUtils.isNotBlank(config.getMtxUrl())) {
            sidecarSpecification = ServiceSpecification.Builder.create()
                    .requestEnhancer(getAuthenticationEnhancer(authenticator))
                    .polling(config.getMtxPolling())
                    .resilienceConfig(ResilienceConfig.builder().numOfRetries(config.getMaxRetries()).retryInterval(config.getRetryWaitTime()).build())
                    .build();
            addSidecarDestination(config.getMtxUrl());
            return new SidecarAccess(sidecarSpecification);
        }
        return null;
    }

    protected HdiContainerManager createHdiContainerManager(DynamicHdiDeploymentParameters hdiDeploymentParas, UserProvidedSchemasExit userProvidedSchemasExit) throws InternalError {
        if (hdiDeploymentParas != null && StringUtils.isNotBlank(hdiDeploymentParas.getUrl())) {
            ServiceSpecification serviceSpecification = ServiceSpecification.Builder.create()
                    .resilienceConfig(hdiDeploymentParas.getResilienceConfig())
                    .polling(hdiDeploymentParas.getPolling()).build();
            addHdiContainerManagerDestination(hdiDeploymentParas);
            return new HdiContainerManager(serviceSpecification, userProvidedSchemasExit);
        }
        return null;
    }

    protected ProvisioningService createProvisioningService(Config config, Authenticator authenticator) throws InternalError {
        if (StringUtils.isNotBlank(config.getProvisioningService().getUrl())) {
            ServiceSpecification serviceSpecification = ServiceSpecification.Builder.create()
                    .resilienceConfig(ResilienceConfig.builder().numOfRetries(config.getProvisioningService().getResilienceConfig().getNumOfRetries())
                            .retryInterval(config.getProvisioningService().getResilienceConfig().getRetryInterval()).build())
                    .requestEnhancer(request -> request.addHeader(HttpHeaders.AUTHORIZATION, authenticator.getAuthorization().orElse(null)))
                    .polling(config.getProvisioningService().getPolling())
                    .build();
            addProvisioningServiceDestination(config.getProvisioningService().getUrl());
            return new ProvisioningService(serviceSpecification);
        }
        return null;
    }

    protected void addSidecarDestination(String url) {
        loader.registerDestination(DefaultHttpDestination.builder(url).name(SidecarAccess.MTX_SIDECAR_DESTINATION).build());
    }

    protected void addHdiContainerManagerDestination(DynamicHdiDeploymentParameters parameters) {
        HttpDestination httpDestination = DefaultHttpDestination.builder(parameters.getUrl())
                .basicCredentials(new BasicCredentials(parameters.getUser(), parameters.getPassword()))
                .name(HdiContainerManager.HDI_DEPLOYER_DESTINATION).build();
        loader.registerDestination(httpDestination);
    }

    protected void addProvisioningServiceDestination(String url) {
        loader.registerDestination(DefaultHttpDestination.builder(url).name(ProvisioningService.MTX_PROVISIONING_SERVICE_DESTINATION).build());
    }
}