/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.spring.boot.mt.config;

import com.sap.cloud.mt.runtime.DataSourceLookup;
import com.sap.cloud.mt.runtime.DataSourceLookupBuilder;
import com.sap.cloud.mt.runtime.IdentityZoneDeterminer;
import com.sap.cloud.mt.runtime.TenantAwareDataSource;
import com.sap.cloud.mt.runtime.TenantProvider;
import com.sap.cloud.mt.subscription.InstanceLifecycleManager;
import com.sap.cloud.spring.boot.mt.lib.Const;
import com.sap.cloud.spring.boot.mt.lib.SpringEnvironmentAccess;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;

/**
 * Configuration bean that setups a tenant provider, a data source lookup and a routing data source bean
 */
@Configuration("ComSapMtBeanConfig")
@ConfigurationProperties("com.sap.mt")
@ConditionalOnProperty(name = Const.COM_SAP_MT_ENABLED, matchIfMissing = true)
public class BeanConfig {
    private Config config;

    public BeanConfig(Config config) {
        this.config = config;
    }

    /**
     * @return {@link TenantProvider}
     */
    @Bean("comSapMtTenantProvider")
    public TenantProvider tenantProvider(IdentityZoneDeterminer determiner) {
        return new TenantProvider(determiner);
    }


    /**
     * @param instanceLifecycleManager bean
     * @param environment              bean from which pool settings are read
     * @return {@link DataSourceLookup} responsible to create data sources for tenants
     */
    @Bean("comSapMtDataSourceLookup")
    DataSourceLookup dataSourceLookup(InstanceLifecycleManager instanceLifecycleManager, Environment environment) {
        DataSourceLookupBuilder builder = DataSourceLookupBuilder.create();
        return builder.instanceLifecycleManager(instanceLifecycleManager)
                .environmentAccess(new SpringEnvironmentAccess(environment))
                .poolProvider(config.getPoolProvider()).oneDataSourcePerDb(config.isOneDataSourcePerDb())
                .registerJmx(config.isTomcatJmx()).build();
    }

    /**
     * @param dataSourceLookup bean
     * @param tenantProvider   bean
     * @return {@link TenantAwareDataSource} that handles multi tenant specific data source access
     */
    @Bean("comSapMtTenantAwareDataSource")
    public TenantAwareDataSource tenantAwareDataSource(DataSourceLookup dataSourceLookup, TenantProvider tenantProvider) {
        return new TenantAwareDataSource(tenantProvider, dataSourceLookup);
    }
}