/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.spring.boot.mt.config;

import com.sap.cds.mtx.impl.Authenticator;
import com.sap.cloud.mt.subscription.DynamicHdiDeploymentParameters;
import com.sap.cloud.mt.subscription.InstanceLifecycleManager;
import com.sap.cloud.mt.subscription.SaasRegistry;
import com.sap.cloud.mt.subscription.SecurityChecker;
import com.sap.cloud.mt.subscription.Subscriber;
import com.sap.cloud.mt.subscription.SubscriberBuilder;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exits.DependencyExit;
import com.sap.cloud.mt.subscription.exits.Exits;
import com.sap.cloud.mt.subscription.exits.InitDbExit;
import com.sap.cloud.mt.subscription.exits.SubscribeExit;
import com.sap.cloud.mt.subscription.exits.UnSubscribeExit;
import com.sap.cloud.mt.subscription.exits.UserProvidedSchemasExit;
import com.sap.cloud.spring.boot.mt.lib.Const;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

/**
 * Configuration bean that setups a tenant provider, a data source lookup and a routing data source bean
 */
@Configuration("ComSapMtSubscriberConfig")
@Conditional(NoLiquibaseLifecycleManagementCondition.class)
@ConditionalOnProperty(name = Const.COM_SAP_MT_ENABLED, matchIfMissing = true)
public class SubscriberConfig extends AbstractSubscriberConfig {
    public SubscriberConfig() {
        super();
    }

    @Bean("comSapMtSubscriber")
    @Primary
    public Subscriber subscriber(InstanceLifecycleManager instanceLifecycleManager,
                                 @Autowired(required = false) UnSubscribeExit unSubscribeExit,
                                 @Autowired(required = false) SubscribeExit subscribeExit,
                                 @Autowired(required = false) DependencyExit dependencyExit,
                                 @Autowired(required = false) InitDbExit initDbExit,
                                 @Autowired(required = false) UserProvidedSchemasExit userProvidedSchemasExit,
                                 @Autowired(required = false) SaasRegistry saasRegistry,
                                 SecurityChecker securityChecker,
                                 @Autowired(required = false) DynamicHdiDeploymentParameters hdiDeploymentParas,
                                 Config config,
                                 @Autowired(required = false) Authenticator authenticator) throws InternalError {
        Exits exits = new Exits(unSubscribeExit, subscribeExit, dependencyExit, initDbExit, userProvidedSchemasExit, false);
        try {
            return SubscriberBuilder.create()
                    .sidecar(createSidecar(config, authenticator))
                    .instanceLifecycleManager(instanceLifecycleManager)
                    .exits(exits)
                    .saasRegistry(saasRegistry).securityChecker(securityChecker)
                    .hdiContainerManager(createHdiContainerManager(hdiDeploymentParas, userProvidedSchemasExit))
                    .baseUiUrl(config.getBaseUiUrl()).urlSeparator(config.getUrlSeparator())
                    .provisioningService(createProvisioningService(config, authenticator))
                    .withoutAuthorityCheck(true)
                    .hanaEncryptionMode(config.getHanaEncryptionMode())
                    .build();
        } catch (InternalError internalError) {
            throw new BeanCreationException(internalError.getMessage());
        }
    }

    @Bean("comSapMtSubscriberAsyncExits")
    public Subscriber subscriberAsyncExits(InstanceLifecycleManager instanceLifecycleManager,
                                           @Autowired(required = false) UnSubscribeExit unSubscribeExit,
                                           @Autowired(required = false) SubscribeExit subscribeExit,
                                           @Autowired(required = false) DependencyExit dependencyExit,
                                           @Autowired(required = false) InitDbExit initDbExit,
                                           @Autowired(required = false) UserProvidedSchemasExit userProvidedSchemasExit,
                                           @Autowired(required = false) SaasRegistry saasRegistry,
                                           SecurityChecker securityChecker,
                                           @Autowired(required = false) DynamicHdiDeploymentParameters hdiDeploymentParas,
                                           Config config,
                                           @Autowired(required = false) Authenticator authenticator) throws InternalError {
        Exits exits = new Exits(unSubscribeExit, subscribeExit, dependencyExit, initDbExit, userProvidedSchemasExit, true);
        try {
            return SubscriberBuilder.create()
                    .sidecar(createSidecar(config, authenticator))
                    .instanceLifecycleManager(instanceLifecycleManager)
                    .exits(exits)
                    .saasRegistry(saasRegistry).securityChecker(securityChecker)
                    .hdiContainerManager(createHdiContainerManager(hdiDeploymentParas, userProvidedSchemasExit))
                    .baseUiUrl(config.getBaseUiUrl()).urlSeparator(config.getUrlSeparator())
                    .provisioningService(createProvisioningService(config, authenticator))
                    .withoutAuthorityCheck(true)
                    .hanaEncryptionMode(config.getHanaEncryptionMode())
                    .build();
        } catch (InternalError internalError) {
            throw new BeanCreationException(internalError.getMessage());
        }
    }
}