/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.spring.boot.mt.config;

import com.sap.cds.mtx.impl.Authenticator;
import com.sap.cloud.mt.subscription.DynamicHdiDeploymentParameters;
import com.sap.cloud.mt.subscription.InstanceLifecycleManager;
import com.sap.cloud.mt.subscription.LiquibaseParameters;
import com.sap.cloud.mt.subscription.LiquibaseResourceAccessor;
import com.sap.cloud.mt.subscription.SaasRegistry;
import com.sap.cloud.mt.subscription.SecurityChecker;
import com.sap.cloud.mt.subscription.Subscriber;
import com.sap.cloud.mt.subscription.SubscriberBuilder;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exits.DependencyExit;
import com.sap.cloud.mt.subscription.exits.Exits;
import com.sap.cloud.mt.subscription.exits.InitDbExit;
import com.sap.cloud.mt.subscription.exits.SubscribeExit;
import com.sap.cloud.mt.subscription.exits.UnSubscribeExit;
import com.sap.cloud.mt.subscription.exits.UserProvidedSchemasExit;
import com.sap.cloud.spring.boot.mt.lib.Const;
import liquibase.integration.spring.SpringResourceAccessor;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.core.io.ResourceLoader;

/**
 * Configuration bean that setups a tenant provider, a data source lookup and a routing data source bean
 */
@Configuration("ComSapMtSubscriberWithLiquibaseConfig")
@Conditional(LiquibaseLifecycleManagementCondition.class)
@ConditionalOnProperty(name = Const.COM_SAP_MT_ENABLED, matchIfMissing = true)
public class SubscriberWithLiquibaseConfig extends AbstractSubscriberConfig {

    @Bean("comSapMtSubscriber")
    @Primary
    public Subscriber subscriber(InstanceLifecycleManager instanceLifecycleManager,
                                 @Autowired(required = false) UnSubscribeExit unSubscribeExit,
                                 @Autowired(required = false) SubscribeExit subscribeExit,
                                 @Autowired(required = false) DependencyExit dependencyExit,
                                 @Autowired(required = false) InitDbExit initDbExit,
                                 @Autowired(required = false) UserProvidedSchemasExit userProvidedSchemasExit,
                                 @Autowired(required = false) SaasRegistry saasRegistry,
                                 SecurityChecker securityChecker,
                                 @Autowired(required = false) DynamicHdiDeploymentParameters hdiDeploymentParas,
                                 @Autowired(required = false) LiquibaseResourceAccessor liquibaseResourceAccessor,
                                 ResourceLoader resourceLoader,
                                 Config config,
                                 LiquibaseConfig liquibaseConfig,
                                 @Autowired(required = false) Authenticator authenticator) throws InternalError {
        Exits exits = new Exits(unSubscribeExit, subscribeExit, dependencyExit, initDbExit, userProvidedSchemasExit, false);
        if (liquibaseResourceAccessor == null) {
            liquibaseResourceAccessor = (String tenant) -> new SpringResourceAccessor(resourceLoader);
        }
        LiquibaseParameters liquibaseParameters = new LiquibaseParameters(liquibaseConfig.getChangeLog(), liquibaseConfig.getContexts(), liquibaseResourceAccessor);
        try {
            return SubscriberBuilder.create().instanceLifecycleManager(instanceLifecycleManager).exits(exits)
                    .saasRegistry(saasRegistry).securityChecker(securityChecker)
                    .hdiContainerManager(createHdiContainerManager(hdiDeploymentParas, userProvidedSchemasExit))
                    .liquibaseParameters(liquibaseParameters)
                    .baseUiUrl(config.getBaseUiUrl()).urlSeparator(config.getUrlSeparator())
                    .withoutAuthorityCheck(true)
                    .hanaEncryptionMode(config.getHanaEncryptionMode())
                    .build();
        } catch (InternalError internalError) {
            throw new BeanCreationException(internalError.getMessage());
        }
    }

    @Bean("comSapMtSubscriberAsyncExits")
    public Subscriber subscriberWithAsyncExist(InstanceLifecycleManager instanceLifecycleManager,
                                               @Autowired(required = false) UnSubscribeExit unSubscribeExit,
                                               @Autowired(required = false) SubscribeExit subscribeExit,
                                               @Autowired(required = false) DependencyExit dependencyExit,
                                               @Autowired(required = false) InitDbExit initDbExit,
                                               @Autowired(required = false) UserProvidedSchemasExit userProvidedSchemasExit,
                                               @Autowired(required = false) SaasRegistry saasRegistry,
                                               SecurityChecker securityChecker,
                                               @Autowired(required = false) DynamicHdiDeploymentParameters hdiDeploymentParas,
                                               @Autowired(required = false) LiquibaseResourceAccessor liquibaseResourceAccessor,
                                               ResourceLoader resourceLoader,
                                               Config config,
                                               LiquibaseConfig liquibaseConfig,
                                               @Autowired(required = false) Authenticator authenticator) throws InternalError {
        Exits exits = new Exits(unSubscribeExit, subscribeExit, dependencyExit, initDbExit, userProvidedSchemasExit, true);
        if (liquibaseResourceAccessor == null) {
            liquibaseResourceAccessor = (String tenant) -> new SpringResourceAccessor(resourceLoader);
        }
        LiquibaseParameters liquibaseParameters = new LiquibaseParameters(liquibaseConfig.getChangeLog(), liquibaseConfig.getContexts(), liquibaseResourceAccessor);
        try {
            return SubscriberBuilder.create().instanceLifecycleManager(instanceLifecycleManager).exits(exits)
                    .saasRegistry(saasRegistry).securityChecker(securityChecker)
                    .hdiContainerManager(createHdiContainerManager(hdiDeploymentParas, userProvidedSchemasExit))
                    .liquibaseParameters(liquibaseParameters)
                    .baseUiUrl(config.getBaseUiUrl()).urlSeparator(config.getUrlSeparator())
                    .withoutAuthorityCheck(true)
                    .hanaEncryptionMode(config.getHanaEncryptionMode())
                    .build();
        } catch (InternalError internalError) {
            throw new BeanCreationException(internalError.getMessage());
        }
    }
}