/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.spring.boot.mt.controller;

import com.sap.cloud.mt.subscription.Subscriber;
import com.sap.cloud.mt.subscription.Tools;
import com.sap.cloud.mt.subscription.exceptions.AuthorityError;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.mt.subscription.exceptions.NotSupported;
import com.sap.cloud.mt.subscription.exceptions.ParameterError;
import com.sap.cloud.mt.subscription.json.ApplicationDependency;
import com.sap.cloud.mt.subscription.json.DeletePayload;
import com.sap.cloud.mt.subscription.json.SidecarUpgradePayload;
import com.sap.cloud.mt.subscription.json.SubscriptionPayload;
import com.sap.cloud.spring.boot.mt.lib.Const;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestHeader;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import static com.sap.cloud.mt.subscription.MtxTools.SAAS_REGISTRY_WAIT_TIME;
import static com.sap.cloud.mt.subscription.SecurityChecker.Authority.INIT_DB;
import static com.sap.cloud.mt.subscription.SecurityChecker.Authority.SUBSCRIBE;
import static com.sap.cloud.mt.subscription.SecurityChecker.Authority.UNSUBSCRIBE;


@RestController("comSapMtEndpoints")
@ConditionalOnProperty(name = Const.COM_SAP_MT_ENABLED, matchIfMissing = true)
public class Endpoints {
    private static final Logger logger = LoggerFactory.getLogger(Endpoints.class);
    private static final String PARAMETER_ERROR = "Parameter error";
    private static final String AUTHORITY_ERROR = "Authority error";
    private static final String INTERNAL_ERROR = "Internal error";
    private final Subscriber subscriber;
    private final Subscriber subscriberWithAsyncExits;

    public Endpoints(@Qualifier("comSapMtSubscriber") Subscriber subscriber, @Qualifier("comSapMtSubscriberAsyncExits") Subscriber subscriberWithAsyncExits) {
        this.subscriber = subscriber;
        this.subscriberWithAsyncExits = subscriberWithAsyncExits;
    }

    @GetMapping(value = "/mt_lib/callback/v1.0/dependencies")
    public List<ApplicationDependency> getDependencies() {
        logger.debug("Get dependencies called");
        try {
            subscriber.checkAuthority(SUBSCRIBE);
            return subscriber.getApplicationDependencies();
        } catch (AuthorityError authorityError) {
            logger.debug(AUTHORITY_ERROR, authorityError);
            throw new InsufficientAuthorizationException(authorityError.getMessage());
        }
    }

    @PutMapping(value = {"/mt_lib/callback/v1.0/tenants/{tenant_id}"})
    public ResponseEntity<String> subscribe(@PathVariable("tenant_id") String tenantId, @RequestBody(required = false) Map<String, Object> payload,
                                            @RequestHeader("Authorization") String jwt,
                                            @RequestHeader(value = "STATUS_CALLBACK", required = false) String saasRegistryUrl) {
        SubscriptionPayload subscriptionPayload = payload != null ? new SubscriptionPayload(payload) : null;
        logger.debug("Subscribe for tenant {} called", tenantId);
        try {
            if (saasRegistryUrl == null || saasRegistryUrl.isEmpty()) {
                subscriber.checkAuthority(SUBSCRIBE);
                String applicationUrl = subscriber.subscribe(tenantId, subscriptionPayload, jwt);
                logger.debug("Return to CIS with status CREATED. Application URL is {}", applicationUrl);
                return new ResponseEntity<>(applicationUrl, HttpStatus.CREATED);
            } else {
                logger.debug("Async processing requested. Callback is {}", saasRegistryUrl);
                subscriberWithAsyncExits.checkAuthority(SUBSCRIBE);
                CompletableFuture.supplyAsync(() -> {
                    try {
                        String applicationUrl = subscriberWithAsyncExits.subscribe(tenantId, subscriptionPayload, jwt);
                        Tools.waitSomeTime(SAAS_REGISTRY_WAIT_TIME);
                        subscriber.callSaasRegistry(true, "", applicationUrl, saasRegistryUrl);
                    } catch (Exception e) {
                        logger.error("Subscription failed", e);
                        reportError(saasRegistryUrl);
                    }
                    return null;
                });
                logger.debug("Return to CIS with status ACCEPTED.");
                return new ResponseEntity<>(subscriber.getSubscribeUrl(subscriptionPayload), HttpStatus.ACCEPTED);
            }
        } catch (InternalError internalError) {
            logger.error(INTERNAL_ERROR, internalError);
            throw new ServerError(internalError.getMessage());
        } catch (ParameterError parameterError) {
            logger.debug(PARAMETER_ERROR, parameterError);
            throw new BadRequest(parameterError.getMessage());
        } catch (AuthorityError authorityError) {
            logger.debug(AUTHORITY_ERROR, authorityError);
            throw new InsufficientAuthorizationException(authorityError.getMessage());
        }
    }

    @DeleteMapping(value = "/mt_lib/callback/v1.0/tenants/{tenant_id}")
    public ResponseEntity<Void> unsubscribe(@PathVariable("tenant_id") String tenantId, @RequestBody(required = false) Map<String, Object> payload,
                                            @RequestHeader("Authorization") String jwt,
                                            @RequestHeader(value = "STATUS_CALLBACK", required = false) String saasRegistryUrl) {
        var deletePayload = payload != null ? new DeletePayload(payload) : null;
        try {
            if (saasRegistryUrl == null || saasRegistryUrl.isEmpty()) {
                subscriber.checkAuthority(UNSUBSCRIBE);
                subscriber.unsubscribe(tenantId, deletePayload, jwt);
                logger.debug("Return to CIS with status NO_CONTENT");
                return new ResponseEntity<>(HttpStatus.NO_CONTENT);
            } else {
                subscriberWithAsyncExits.checkAuthority(UNSUBSCRIBE);
                logger.debug("Async processing requested. Callback is {}", saasRegistryUrl);
                CompletableFuture.supplyAsync(() -> {
                    try {
                        subscriberWithAsyncExits.unsubscribe(tenantId, deletePayload, jwt);
                        Tools.waitSomeTime(SAAS_REGISTRY_WAIT_TIME);
                        subscriber.callSaasRegistry(true, "", null, saasRegistryUrl);
                    } catch (Exception e) {
                        logger.error("Un-subscription failed", e);
                        reportError(saasRegistryUrl);
                    }
                    return null;
                });
                logger.debug("Return to CIS with status ACCEPTED");
                return new ResponseEntity<>(HttpStatus.ACCEPTED);
            }
        } catch (InternalError internalError) {
            logger.error(INTERNAL_ERROR, internalError);
            throw new ServerError(internalError.getMessage());
        } catch (ParameterError parameterError) {
            logger.debug(PARAMETER_ERROR, parameterError);
            throw new BadRequest(parameterError.getMessage());
        } catch (AuthorityError authorityError) {
            logger.debug(AUTHORITY_ERROR, authorityError);
            throw new InsufficientAuthorizationException(authorityError.getMessage());
        }
    }

    @PostMapping(value = "/mt_lib/v1.0/init_db")
    public void initDb(@RequestBody SidecarUpgradePayload upgradePayload) {
        try {
            subscriber.checkAuthority(INIT_DB);
            subscriber.setupDbTables(Arrays.asList(upgradePayload.tenants));
        } catch (InternalError internalError) {
            logger.error(INTERNAL_ERROR, internalError);
            throw new ServerError(internalError.getMessage());
        } catch (NotSupported notSupported) {
            logger.debug("Not Supported", notSupported);
            throw new BadRequest(notSupported.getMessage());
        } catch (ParameterError parameterError) {
            logger.debug(PARAMETER_ERROR, parameterError);
            throw new BadRequest(parameterError.getMessage());
        } catch (AuthorityError authorityError) {
            logger.debug(AUTHORITY_ERROR, authorityError);
            throw new InsufficientAuthorizationException(authorityError.getMessage());
        }
    }

    @PostMapping(value = "/mt_lib/v1.0/init_db_async")
    public String initDbAsync(@RequestBody SidecarUpgradePayload upgradePayload) {
        try {
            subscriber.checkAuthority(INIT_DB);
            return subscriber.setupDbTablesAsync(Arrays.asList(upgradePayload.tenants));
        } catch (InternalError internalError) {
            logger.error(INTERNAL_ERROR, internalError);
            throw new ServerError(internalError.getMessage());
        } catch (ParameterError parameterError) {
            logger.debug(PARAMETER_ERROR, parameterError);
            throw new BadRequest(parameterError.getMessage());
        } catch (AuthorityError authorityError) {
            logger.debug(AUTHORITY_ERROR, authorityError);
            throw new InsufficientAuthorizationException(authorityError.getMessage());
        }
    }

    @GetMapping(value = "/mt_lib/v1.0/init_db_async/status/{jobId}")
    public String getInitDbAsyncStatus(@PathVariable("jobId") String jobId) {
        try {
            subscriber.checkAuthority(INIT_DB);
            return subscriber.updateStatus(jobId);
        } catch (InternalError internalError) {
            logger.error(INTERNAL_ERROR, internalError);
            throw new ServerError(internalError.getMessage());
        } catch (ParameterError parameterError) {
            logger.debug(PARAMETER_ERROR, parameterError);
            throw new BadRequest(parameterError.getMessage());
        } catch (NotSupported notSupported) {
            logger.debug("Not Supported", notSupported);
            throw new BadRequest(notSupported.getMessage());
        } catch (com.sap.cloud.mt.subscription.exceptions.NotFound notFound) {
            logger.debug("Nout Found", notFound);
            throw new NotFound(notFound.getMessage());
        } catch (AuthorityError authorityError) {
            logger.debug("Authority Error", authorityError);
            throw new InsufficientAuthorizationException(authorityError.getMessage());
        }
    }

    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public static class ServerError extends RuntimeException {
        public ServerError(String message) {
            super(message);
        }
    }

    @ResponseStatus(HttpStatus.NOT_FOUND)
    public static class NotFound extends RuntimeException {
        public NotFound(String message) {
            super(message);
        }
    }

    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public static class BadRequest extends RuntimeException {
        public BadRequest(String message) {
            super(message);
        }
    }

    @ResponseStatus(HttpStatus.FORBIDDEN)
    public static class InsufficientAuthorizationException extends RuntimeException {
        public InsufficientAuthorizationException(String message) {
            super(message);
        }
    }

    private void reportError(String saasRegistryUrl) {
        try {
            Tools.waitSomeTime(SAAS_REGISTRY_WAIT_TIME);
            subscriber.callSaasRegistry(false, "An internal error occurred", null, saasRegistryUrl);
        } catch (InternalError ex) {
            logger.error("Couldn't call saas registry");
        }
    }

}
