/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/

package com.sap.cloud.spring.boot.mt.config;

import com.sap.cloud.mt.subscription.*;
import com.sap.cloud.mt.subscription.exceptions.InternalError;
import com.sap.cloud.spring.boot.mt.lib.Const;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

@Configuration("ComSapMtDbIdentifierConfig")
@ConfigurationProperties("com.sap.mt")
@ConditionalOnProperty(name = Const.COM_SAP_MT_ENABLED, matchIfMissing = true)
public class DbIdentifierConfig {
    private Set<String> hanaDatabaseIds = null;
    private List<PostgreSQLCredentials> postgres;
    private List<H2Credentials> h2;

    @Bean("comSapMtDbIdentifiers")
    public DbIdentifiers dbIdentifiers(Environment environment) {
        String uriStr = environment.getProperty("spring.datasource.url", String.class);
        List<DbCredentials> credentials = new ArrayList<>();
        if (postgres != null) {
            postgres.stream().forEach(p ->
            {
                try {
                    credentials.add(new DbCredentialsPostgreSQL(p.getUser(), p.getPassword(), p.getHost(), p.getPort(), p.getDatabase(),""));
                } catch(InternalError internalError) {
                    //as uri is not specified, won't be raised
                }
            });
            return new DbIdentifiersSql(credentials);
        } else if (h2 != null) {
            h2.stream().forEach(p -> {
                try {
                    credentials.add(new DbCredentialsH2(p.getUser(), p.getPassword(), p.getDatabase()));
                } catch(InternalError internalError) {
                    //as uri is not specified, won't be raised
                }
            });
            return new DbIdentifiersSql(credentials);
        } else if (!StringUtils.isEmpty(uriStr)) {
            String user = environment.getProperty("spring.datasource.username", String.class);
            String password = environment.getProperty("spring.datasource.password", String.class);
            if (StringUtils.isEmpty(user)) {
                throw new BeanCreationException("No DB user specified");
            }
            if (StringUtils.isEmpty(password)) {
                throw new BeanCreationException("No DB password specified");
            }
            if (uriStr.startsWith("jdbc:postgresql:")) {
                try {
                    credentials.add(new DbCredentialsPostgreSQL(user, password,"","","", uriStr));
                } catch(InternalError e) {
                    throw new BeanCreationException(e.getMessage());
                }
                return new DbIdentifiersSql(credentials);
            }
        }
        //Hana is default to stay compatible
        return new DbIdentifiersHana(hanaDatabaseIds);
    }

    public static class PostgreSQLCredentials {
        private String user;
        private String password;
        private String host;
        private String port;
        private String database;

        public void setUser(String user) {
            this.user = user;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public void setHost(String host) {
            this.host = host;
        }

        public void setPort(String port) {
            this.port = port;
        }

        public void setDatabase(String database) {
            this.database = database;
        }

        public String getUser() {
            return user;
        }

        public String getPassword() {
            return password;
        }

        public String getHost() {
            return host;
        }

        public String getPort() {
            return port;
        }

        public String getDatabase() {
            return database;
        }
    }

    public static class H2Credentials {
        private String user;
        private String password;
        private String database;

        public String getUser() {
            return user;
        }

        public void setUser(String user) {
            this.user = user;
        }

        public String getPassword() {
            return password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public String getDatabase() {
            return database;
        }

        public void setDatabase(String database) {
            this.database = database;
        }
    }

    public void setHanaDatabaseIds(Set<String> hanaDatabaseIds) {
        this.hanaDatabaseIds = hanaDatabaseIds;
    }

    public void setPostgres(List<PostgreSQLCredentials> postgres) {
        this.postgres = postgres;
    }

    public void setH2(List<H2Credentials> h2) {
        this.h2 = h2;
    }
}
