/******************************************************************************
 * © 2020 SAP SE or an SAP affiliate company. All rights reserved.            *
 ******************************************************************************/
package com.sap.cloud.spring.boot.mt.config;

import com.sap.cloud.mt.subscription.SecurityChecker;
import com.sap.cloud.mt.subscription.exceptions.AuthorityError;
import com.sap.cloud.security.adapter.spring.SpringSecurityContext;
import com.sap.cloud.security.token.AccessToken;
import com.sap.cloud.spring.boot.mt.lib.Const;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Configuration;

import java.util.function.Supplier;

@Configuration("comSapMtSecurityChecker")
@ConditionalOnClass(SpringSecurityContext.class)
@ConditionalOnProperty(name = Const.COM_SAP_MT_ENABLED, matchIfMissing = true)
@ConditionalOnMissingBean(SecurityChecker.class)
public class SecurityCheckerWithJavaSecurityImpl implements SecurityChecker {
    private final Config config;
    private final Supplier<AccessToken> accessTokenSupplier;

    public SecurityCheckerWithJavaSecurityImpl(Config config, @Qualifier("comSapMtAccessToken") Supplier<AccessToken> accessTokenSupplier) {
        this.config = config;
        this.accessTokenSupplier = accessTokenSupplier;
    }

    @Override
    public void checkSubscriptionAuthority() throws AuthorityError {
        if (config.isSecurityCheckOff()) return;
        try {
            if (!accessTokenSupplier.get().hasLocalScope(config.getSubscribeScope()))
                throw new AuthorityError("No authority to call endpoint");
        } catch (Exception e) {
            throw new AuthorityError(e);
        }
    }

    @Override
    public void checkInitDbAuthority() throws AuthorityError {
        if (config.isSecurityCheckOff()) return;
        try {
            if (!accessTokenSupplier.get().hasLocalScope(config.getInitDbScope()))
                throw new AuthorityError("No authority to call endpoint");
        } catch (Exception e) {
            throw new AuthorityError(e);
        }
    }
}
