/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import ch.qos.logback.core.spi.DeferredProcessingAware;

import java.io.IOException;

import net.logstash.logback.composite.AbstractFieldJsonProvider;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;


/**
 * A provider for extracting information about the application instance in Cloud Foundry, which is stored in the
 * VCAP_APPLICATION environment variable.
 */
public class CloudFoundryInfoJsonProvider<E extends DeferredProcessingAware> extends AbstractFieldJsonProvider<E>
{
	private static final String CF_VCAP_KEY = "VCAP_APPLICATION";
	private static final String CF_INSTANCE_INFO_KEY = "instance_index";
	private static final String DEFAULT_FIELD_NAME = "instance";
	private static final JsonFactory JSON_FACTORY = new JsonFactory(new ObjectMapper());

	private String instanceIndex;

	/**
	 * Creates a new instance assigning a default field name.
	 */
	public CloudFoundryInfoJsonProvider()
	{
		setFieldName(DEFAULT_FIELD_NAME);
	}

	@Override
	public void start()
	{
		final String cfInstanceInfo = getProperty(CF_VCAP_KEY, null);
		if (cfInstanceInfo != null)
		{
			try
			{
				final JsonParser jp = JSON_FACTORY.createParser(cfInstanceInfo);
				final JsonNode cfInfo = jp.readValueAsTree();
				final JsonNode instanceNode = cfInfo.get(CF_INSTANCE_INFO_KEY);
				if (instanceNode == null)
				{
					addError("Cannot find '" + CF_INSTANCE_INFO_KEY + "' attribute in cloudfoundry '" + CF_VCAP_KEY
							+ "' property with value:" + cfInstanceInfo);
				}
				else
				{
					instanceIndex = instanceNode.toString();
				}
			}
			catch (final IOException e)
			{
				addError("Cannot parse cloudfoundry '" + CF_VCAP_KEY + "' property with value: " + cfInstanceInfo, e);
			}
		}
		super.start();
	}

	private String getProperty(final String key, final String defaultValue)
	{
		String result = System.getProperty(key);
		if (result == null)
		{
			result = System.getenv(key);
		}
		if (result == null)
		{
			result = defaultValue;
		}
		return result;
	}

	@Override
	public void writeTo(final JsonGenerator generator, final E event) throws IOException
	{
		if (instanceIndex != null && !instanceIndex.isEmpty())
		{
			generator.writeStringField(getFieldName(), instanceIndex);
		}
	}
}
