/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import ch.qos.logback.core.Context;
import ch.qos.logback.core.spi.DeferredProcessingAware;

import java.io.IOException;

import net.logstash.logback.composite.AbstractFieldJsonProvider;
import net.logstash.logback.composite.JsonFactoryAware;
import net.logstash.logback.composite.JsonProvider;
import net.logstash.logback.composite.JsonProviders;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.MappingJsonFactory;


/**
 * A {@link JsonProvider} being a composite of many other {@link JsonProvider}'s, so chaining all write calls to a
 * sub-list of providers.
 */
public class CompositeJsonProvider<E extends DeferredProcessingAware> extends AbstractFieldJsonProvider<E> implements
		JsonFactoryAware
{
	private final JsonProviders<E> providers = new JsonProviders<E>();

	/**
	 * Adds a provider to the list of sub-providers.
	 * 
	 * @param provider provider to add to the list of providers to call
	 */
	public void addProvider(final JsonProvider<E> provider)
	{
		providers.addProvider(provider);
	}

	@Override
	public void start()
	{
		providers.start();
	}

	@Override
	public void stop()
	{
		providers.stop();
	}

	@Override
	public void setContext(final Context context)
	{
		providers.setContext(context);
	}

	@Override
	public void writeTo(final JsonGenerator generator, final E event) throws IOException
	{
		if (getFieldName() != null)
		{
			generator.writeObjectFieldStart(getFieldName());
		}
		providers.writeTo(generator, event);
		if (getFieldName() != null)
		{
			generator.writeEndObject();
		}
	}

	@Override
	public void setJsonFactory(final JsonFactory jsonFactory)
	{
		if (jsonFactory instanceof MappingJsonFactory)
		{
			providers.setJsonFactory((MappingJsonFactory) jsonFactory);
		}
	}
}
