/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Context;

import java.io.IOException;

import net.logstash.logback.composite.AbstractJsonProvider;
import net.logstash.logback.composite.JsonFactoryAware;
import net.logstash.logback.composite.JsonProvider;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;


/**
 * A {@link JsonProvider} chaining the write call to either the {@code thenProvider} or {@code elseProvider} dependent
 * on if the current log event has a marker attached configured with {@code ifMarker}. You can formulate the logic like
 * that: if event.marker contains ifMarker then call thenProvider else call elseProvider.
 */
public class ConditionalJsonProvider extends AbstractJsonProvider<ILoggingEvent> implements JsonFactoryAware
{
	private String ifMarker;
	private JsonProvider<ILoggingEvent> thenProvider;
	private JsonProvider<ILoggingEvent> elseProvider;

	/**
	 * The marker text to evaluate on.
	 * 
	 * @param ifMarker the textual marker to evaluate the event marker on
	 */
	public void setIfMarker(final String ifMarker)
	{
		this.ifMarker = ifMarker;
	}

	/**
	 * The provider to call in case the log event has the {@code ifMarker} assigned.
	 * 
	 * @param thenProvider provider to call if ifMarker applies
	 */
	public void setThenProvider(final JsonProvider<ILoggingEvent> thenProvider)
	{
		this.thenProvider = thenProvider;
	}

	/**
	 * The provider to call in case the log event has NOT the {@code ifMarker} assigned.
	 * 
	 * @param elseProvider provider to call if ifMarker does NOT apply
	 */
	public void setElseProvider(final JsonProvider<ILoggingEvent> elseProvider)
	{
		this.elseProvider = elseProvider;
	}

	@Override
	public void writeTo(final JsonGenerator generator, final ILoggingEvent event) throws IOException
	{
		if (ifMarker != null && event.getMarker() != null && thenProvider != null && event.getMarker().contains(ifMarker))
		{
			thenProvider.writeTo(generator, event);
		}
		else if (elseProvider != null)
		{
			elseProvider.writeTo(generator, event);
		}
	}

	@Override
	public void start()
	{
		if (thenProvider != null)
		{
			thenProvider.start();
		}
		if (elseProvider != null)
		{
			elseProvider.start();
		}
	}

	@Override
	public void stop()
	{
		if (thenProvider != null)
		{
			thenProvider.stop();
		}
		if (elseProvider != null)
		{
			elseProvider.stop();
		}
	}

	@Override
	public void setContext(final Context context)
	{
		if (thenProvider != null)
		{
			thenProvider.setContext(context);
		}
		if (elseProvider != null)
		{
			elseProvider.setContext(context);
		}
	}

	@Override
	public void setJsonFactory(final JsonFactory jsonFactory)
	{
		if (thenProvider instanceof JsonFactoryAware)
		{
			((JsonFactoryAware) thenProvider).setJsonFactory(jsonFactory);
		}
		if (elseProvider instanceof JsonFactoryAware)
		{
			((JsonFactoryAware) elseProvider).setJsonFactory(jsonFactory);
		}
	}
}
