/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.Map;
import java.util.concurrent.Callable;

import org.slf4j.MDC;


/**
 * Wrapper Callable which assures that the given delegate callable is ran with MDC content which is either given or
 * copied from the current Thread.
 */
public class DelegatingMDCCallable<V> implements Callable<V>
{

	private final Callable<V> delegate;
	private final Map<String, String> context;

	/**
	 * Constructor for the DelegatingMDCCallable which runs the delegate using given context as {@link org.slf4j.MDC}.
	 * 
	 * @param delegate the delegate callable which will be run with the given context
	 * @param context context as a Map<String, String> which will be set as the delegate callable MDC context
	 */
	public DelegatingMDCCallable(final Callable<V> delegate, final Map<String, String> context)
	{
		this.delegate = delegate;
		this.context = context;
	}

	/**
	 * Constructor for the DelegatingMDCCallable which runs the delegate using a {@link org.slf4j.MDC} copied from the
	 * current thread.
	 * 
	 * @param delegate delegate callable which will be run with the current thread's MDC context
	 */
	public DelegatingMDCCallable(final Callable<V> delegate)
	{
		this(delegate, MDC.getCopyOfContextMap());
	}

	@SuppressWarnings("PMD.SignatureDeclareThrowsException")
	@Override
	public V call() throws Exception
	{
		try
		{
			if (context != null)
			{
				MDC.setContextMap(context);
			}
			return delegate.call();
		}
		finally
		{
			MDC.clear();
		}
	}

	@Override
	public String toString()
	{
		return delegate.toString();
	}

	/**
	 * Creates an instance of DelegatingMDCCallable which runs the delegate using given context as {@link org.slf4j.MDC}.
	 * 
	 * @param delegate the delegate callable which will be run with the given context
	 * @param context context as a Map<String, String> which will be set as the delegate callable MDC context
	 * @param <V> the type of the returned object
	 *
	 * @return the wrapped callable
	 */
	public static <V> Callable<V> create(final Callable<V> delegate, final Map<String, String> context)
	{
		return context == null ? new DelegatingMDCCallable<V>(delegate)
				: new DelegatingMDCCallable<V>(delegate, context);
	}
}
