/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.Map;
import java.util.concurrent.Executor;

import org.slf4j.MDC;


/**
 * Wrapper Executor which assures that the given delegate executor is ran with MDC content which is either given or
 * copied from the current Thread.
 */
public class DelegatingMDCExecutor implements Executor
{
	private final Executor delegate;
	private final Map<String, String> context;

	/**
	 * Constructor for the DelegatingMDCRunnable which runs the delegate using given context as {@link org.slf4j.MDC}.
	 * 
	 * @param delegateExecutor the delegate executor which will be run with the given context
	 * @param context context as a Map<String, String> which will be set as the delegate runnable's MDC context
	 */
	public DelegatingMDCExecutor(final Executor delegateExecutor, final Map<String, String> context)
	{
		this.delegate = delegateExecutor;
		this.context = context;
	}

	/**
	 * Constructor for the DelegatingMDCExecutor which runs the delegate using a {@link org.slf4j.MDC} copied from the
	 * current thread.
	 * 
	 * @param delegate delegate runnable which will be run with the current thread's MDC context
	 */
	public DelegatingMDCExecutor(final Executor delegate)
	{
		this(delegate, MDC.getCopyOfContextMap());
	}

	@Override
	public final void execute(final Runnable task)
	{
		final Runnable wrappedTask = wrap(task);
		delegate.execute(wrappedTask);
	}

	protected final Executor getDelegateExecutor()
	{
		return delegate;
	}

	protected final Runnable wrap(final Runnable delegateToRun)
	{
		return DelegatingMDCRunnable.create(delegateToRun, context);
	}
}
