/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;


/**
 * Wrapper Executor service which assures that the given delegate executor service is ran with MDC content which is
 * either given or copied from the current Thread.
 * This class assures that the executed tasks are wrapped with the {@link DelegatingMDCCallable}.
 */
@SuppressWarnings("PMD.TooManyMethods")
public class DelegatingMDCExecutorService implements ExecutorService
{
	private final ExecutorService delegate;
	private final Map<String, String> context;

	/**
	 * Constructor for the DelegatingMDCExecutorService which runs the delegate using given context as
	 * {@link org.slf4j.MDC}.
	 * 
	 * @param delegateExecutorService the delegate executor service which will execute tasks with the given context
	 * @param context context as a Map<String, String> which will be set as the executed tasks's MDC context
	 */
	public DelegatingMDCExecutorService(final ExecutorService delegateExecutorService,
			final Map<String, String> context)
	{
		this.delegate = delegateExecutorService;
		this.context = context;
	}

	/**
	 * Constructor for the DelegatingMDCExecutorService which runs the delegate using an {@link org.slf4j.MDC} copied
	 * from the current thread.
	 * 
	 * @param delegate delegate executor service which will be run with the current thread's MDC context
	 */
	public DelegatingMDCExecutorService(final ExecutorService delegate)
	{
		this(delegate, null);
	}

	@Override
	public final void execute(final Runnable task)
	{
		final Runnable wrappedTask = wrap(task);
		delegate.execute(wrappedTask);
	}

	@Override
	public final void shutdown()
	{
		delegate.shutdown();
	}

	@Override
	public final List<Runnable> shutdownNow()
	{
		return delegate.shutdownNow();
	}

	@Override
	public final boolean isShutdown()
	{
		return delegate.isShutdown();
	}

	@Override
	public final boolean isTerminated()
	{
		return delegate.isTerminated();
	}

	@Override
	public final boolean awaitTermination(final long timeout, final TimeUnit unit) throws InterruptedException
	{
		return delegate.awaitTermination(timeout, unit);
	}

	@Override
	public final <T> Future<T> submit(final Callable<T> task)
	{
		final Callable<T> wrappedTask = wrap(task);
		return delegate.submit(wrappedTask);
	}

	@Override
	public final <T> Future<T> submit(final Runnable task, final T result)
	{
		final Runnable wrappedTask = wrap(task);
		return delegate.submit(wrappedTask, result);
	}

	@Override
	public final Future<?> submit(final Runnable task)
	{
		final Runnable wrappedTask = wrap(task);
		return delegate.submit(wrappedTask);
	}

	@Override
	@SuppressWarnings({"rawtypes", "unchecked"})
	public final List invokeAll(final Collection tasks) throws InterruptedException
	{
		final Collection wrappedTasks = createTasks(tasks);
		return delegate.invokeAll(wrappedTasks);
	}

	@Override
	@SuppressWarnings({"rawtypes", "unchecked"})
	public final List invokeAll(final Collection tasks, final long timeout, final TimeUnit unit)
			throws InterruptedException
	{
		final Collection wrappedTasks = createTasks(tasks);
		return delegate.invokeAll(wrappedTasks, timeout, unit);
	}

	@Override
	@SuppressWarnings({"rawtypes", "unchecked"})
	public final Object invokeAny(final Collection tasks) throws InterruptedException, ExecutionException
	{
		final Collection wrappedTasks = createTasks(tasks);
		return delegate.invokeAny(wrappedTasks);
	}

	@Override
	@SuppressWarnings({"rawtypes", "unchecked"})
	public final Object invokeAny(final Collection tasks, final long timeout, final TimeUnit unit)
			throws InterruptedException, ExecutionException, TimeoutException
	{
		final Collection wrappedTasks = createTasks(tasks);
		return delegate.invokeAny(wrappedTasks, timeout, unit);
	}

	private <T> Collection<Callable<T>> createTasks(final Collection<Callable<T>> tasks)
	{
		if (tasks == null)
		{
			return null;
		}
		final List<Callable<T>> results = new ArrayList<Callable<T>>(tasks.size());
		for (final Callable<T> task : tasks)
		{
			results.add(wrap(task));
		}
		return results;
	}

	protected final Runnable wrap(final Runnable delegateToRun)
	{
		return DelegatingMDCRunnable.create(delegateToRun, context);
	}

	protected final <T> Callable<T> wrap(final Callable<T> delegateToRun)
	{
		return DelegatingMDCCallable.create(delegateToRun, context);
	}
}
