/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.concurrent.Callable;

import com.netflix.hystrix.strategy.concurrency.HystrixConcurrencyStrategy;
import com.netflix.hystrix.strategy.concurrency.HystrixRequestContext;
import com.netflix.hystrix.strategy.concurrency.HystrixRequestVariable;
import com.netflix.hystrix.strategy.concurrency.HystrixRequestVariableDefault;
import com.netflix.hystrix.strategy.concurrency.HystrixRequestVariableLifecycle;


/**
 * A {@code HystrixConcurrencyStrategy} wrapping any {@link Callable} in a {@link DelegatingMDCCallable}, so that it
 * will be
 * executed with the correct {@link org.slf4j.MDC} assigned.
 */
public class DelegatingMDCHystrixConcurrencyStrategy extends HystrixConcurrencyStrategy
{

	/**
	 * Wraps the given {@code Callable} in a {@link DelegatingMDCCallable}.
	 *
	 * @param <T> the type of {@code Callable} to wrap
	 * @param callable {@code Callable} to wrap
	 * @return the wrapped {@code Callable}, which will propagate the {@link org.slf4j.MDC} upon execution
	 */
	@Override
	public <T> Callable<T> wrapCallable(final Callable<T> callable)
	{
		return new DelegatingMDCCallable<>(callable);
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * This implementation will return a {@code HystrixRequestVariable} based on the {@link HystrixRequestContext} that
	 * is bound to the current {@link Thread}.
	 * <p>
	 * If none is bound to the current thread, this implementation will return a dummy {@code HystrixRequestVariable}.
	 * This fallback behavior is provided as a convenience feature only. If you encounter any Hystrix related problems,
	 * please make sure that a {@link HystrixRequestContext} is bound to every {@code Thread} that uses Hystrix.
	 */
	@Override
	public <T> HystrixRequestVariable<T> getRequestVariable(final HystrixRequestVariableLifecycle<T> rv)
	{
		if (HystrixRequestContext.getContextForCurrentThread() != null)
		{
			return super.getRequestVariable(rv);
		}
		else
		{
			return new HystrixRequestVariableDefault<T>()
			{
				@Override
				public T get()
				{
					return initialValue();
				}

				// the below two methods are copied from default implementation
				@Override
				public T initialValue()
				{
					return rv.initialValue();
				}

				@Override
				public void shutdown(final T value)
				{
					rv.shutdown(value);
				}
			};
		}
	}
}
