/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.Map;

import org.slf4j.MDC;


/**
 * Wrapper Runnable which assures that the given delegate runnable is ran with MDC content which is either given or
 * copied from the current Thread.
 */
public class DelegatingMDCRunnable implements Runnable
{
	private final Runnable delegate;
	private final Map<String, String> context;

	/**
	 * Constructor for the DelegatingMDCRunnable which runs the delegate using given context as {@link org.slf4j.MDC}.
	 * 
	 * @param delegate the delegate runnable which will be run with the given context
	 * @param context context as a Map<String, String> which will be set as the delegate runnable's MDC context
	 */
	public DelegatingMDCRunnable(final Runnable delegate, final Map<String, String> context)
	{
		this.delegate = delegate;
		this.context = context;
	}

	/**
	 * Constructor for the DelegatingMDCRunnable which runs the delegate using a {@link org.slf4j.MDC} copied from the
	 * current thread.
	 * 
	 * @param delegate delegate runnable which will be run with the current thread's MDC context
	 */
	public DelegatingMDCRunnable(final Runnable delegate)
	{
		this(delegate, MDC.getCopyOfContextMap());
	}

	@Override
	public void run()
	{
		try
		{
			if (context != null)
			{
				MDC.setContextMap(context);
			}
			delegate.run();
		}
		finally
		{
			MDC.clear();
		}
	}

	@Override
	public String toString()
	{
		return delegate.toString();
	}


	/**
	 * Creates an instance of DelegatingMDCRunnable which runs the delegate using given context as {@link org.slf4j.MDC}.
	 * 
	 * @param delegate the delegate runnable which will be run with the given context
	 * @param context context as a Map<String, String> which will be set as the delegate runnable's MDC context
	 *
	 * @return the wrapped runnable
	 */
	public static Runnable create(final Runnable delegate, final Map<String, String> context)
	{

		return context == null ? new DelegatingMDCRunnable(delegate)
				: new DelegatingMDCRunnable(delegate, context);
	}
}
