/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.slf4j.MDC;

import rx.Scheduler;
import rx.Subscription;
import rx.functions.Action0;


/**
 * Wrapper Scheduler which assures that the given delegate scheduler runs with MDC content which is either given or
 * copied from the current Thread.
 */
public class DelegatingMDCScheduler extends Scheduler
{
	private final Scheduler delegate;
	private final Map<String, String> context;

	/**
	 * Constructor for the DelegatingMDCScheduler which runs the delegate using given context as {@link org.slf4j.MDC}.
	 * 
	 * @param delegate the delegate runnable which will be run with the given context
	 * @param context context as a Map<String, String> which will be set as the delegate runnable's MDC context
	 */
	public DelegatingMDCScheduler(final Scheduler delegate,
			final Map<String, String> context)
	{
		this.delegate = delegate;
		this.context = context;
	}

	/**
	 * Constructor for the DelegatingMDCRunnable which runs the delegate using a {@link org.slf4j.MDC} copied from the
	 * current thread.
	 * 
	 * @param delegate delegate runnable which will be run with the current thread's MDC context
	 */
	public DelegatingMDCScheduler(final Scheduler delegate)
	{
		this(delegate, null);
	}

	@Override
	public Worker createWorker()
	{
		return new DelegatingMDCWorker(delegate.createWorker(), context);
	}

	private static class DelegatingMDCWorker extends Worker
	{
		private final Worker delegate;
		private final Map<String, String> context;

		public DelegatingMDCWorker(final Worker delegate,
				final Map<String, String> context)
		{
			this.delegate = delegate;
			this.context = context;
		}

		@Override
		public Subscription schedule(final Action0 action)
		{
			return delegate.schedule(new DelegatingMDCAction(action, context));
		}

		@Override
		public Subscription schedule(final Action0 action, final long delayTime, final TimeUnit unit)
		{
			return delegate.schedule(new DelegatingMDCAction(action, context), delayTime, unit);
		}

		@Override
		public boolean isUnsubscribed()
		{
			return delegate.isUnsubscribed();
		}

		@Override
		public void unsubscribe()
		{
			delegate.unsubscribe();
		}
	}

	private static class DelegatingMDCAction implements Action0
	{
		private final Action0 delegate;
		private final Map<String, String> context;

		public DelegatingMDCAction(final Action0 delegate, final Map<String, String> context)
		{
			this.delegate = delegate;
			this.context = context == null ? MDC.getCopyOfContextMap() : context;
		}

		@Override
		public void call()
		{
			try
			{
				if (context != null)
				{
					MDC.setContextMap(context);
				}
				delegate.call();
			}
			finally
			{
				MDC.clear();
			}
		}
	}
}
