/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.function.Consumer;
import java.util.function.Supplier;

import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

/**
 * Represents a log action, which is a nullary function without a return value.
 * <p>
 * Also provides static utility methods that wrap {@code LogAction}s with preparation and clean-up functionality.
 */
@FunctionalInterface
public interface LogAction
{
	/**
	 * Performs this log action.
	 */
	void log();

	/**
	 * Wraps a given {@code LogAction} with {@code preparation} and {@code cleanUp} functionality, while transferring
	 * temporary state between the two.
	 * 
	 * @param <T> The type of temporary state to transfer between {@code preparation} and {@code cleanUp}.
	 * @param preparation A function that prepares the {@code actionToWrap} for logging.
	 *         May supply temporary state that is necessary for {@code cleanUp}, or return {@code null}, if no such state is
	 *         necessary.
	 *         May also be {@code null} itself, if no preparation is necessary at all.
	 * @param actionToWrap The {@code LogAction} to wrap.
	 * @param cleanUp A function that cleans up after the {@code actionToWrap} has been logged.
	 *         May consume temporary state that has been supplied upon {@code preparation}, but must be prepared to consume
	 *         {@code null} in case {@code preparation} failed.
	 *         May also be {@code null} itself, if no clean-up is necessary at all.
	 * @return The {@code actionToWrap} wrapped with {@code preparation} and {@code cleanUp} functionality.<br>
	 *         When the returned {@code LogAction} is performed, it is guaranteed to execute {@code preparation} functionality
	 *         before it performs the actual {@code actionToWrap}.
	 *         It then executes the {@code actionToWrap}, if, and only if, {@code preparation} functionality returned normally.
	 *         It is also guaranteed to execute the {@code cleanUp} functionality afterwards, regardless of failures in the
	 *         previous steps.
	 */
	static <T> LogAction wrap(
			final Supplier<? extends T> preparation,
			final LogAction actionToWrap,
			final Consumer<? super T> cleanUp)
	{
		return () -> {
			T state = null;
			try
			{
				if (preparation != null)
				{
					state = preparation.get();
				}
				actionToWrap.log();
			}
			finally
			{
				if(cleanUp != null)
				{
					cleanUp.accept(state);
				}
			}
		};
	}

	/**
	 * Wraps a given {@code LogAction} with a given diagnostic context, that is, a key-value pair on the {@link MDC}.
	 * 
	 * @param key The MDC key.
	 * @param value The MDC value.
	 * @param actionToWrap The {@code LogAction} to wrap.
	 * @return A {@code LogAction} that sets up the given diagnostic context before the {@code actionToWrap}
	 *         itself is performed, and restores the {@code MDC} to its previous state afterwards.
	 */
	static LogAction wrapWithDiagnosticContext(final String key, final String value, final LogAction actionToWrap)
	{
		// if no MDC value is given, just pass-through the LogAction
		if (value == null)
		{
			return actionToWrap;
		}

		// wrap with given diagnostic context
		return wrap(
				() -> {
					// remember a previously set MDC value and emit a warning, if it's present and differs from the given one
					final String oldValue = MDC.get(key);
					if (oldValue != null && !oldValue.equals(value))
					{
						LoggerFactory
								.getLogger(LogAction.class)
								.warn("The " + key + " value '" + oldValue + "' present in the MDC is overriden with '" + value
										+ "'. This may lead to inconsistencies in logging.");
					}
					MDC.put(key, value);
					return oldValue;
				},
				actionToWrap,
				oldValue -> {
					// remove the given MDC value, and restore the previously set one
					if (oldValue != null)
					{
						MDC.put(key, oldValue);
					}
					else
					{
						MDC.remove(key);
					}
				});
	}
}
