/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import org.slf4j.Logger;



/**
 * A wrapper around a {@code slf4j} {@link Logger} that adds convenience features for logging.
 * <p>
 * In particular this includes setting multiple tags, which are mapped to {@code slf4j} {@link org.slf4j.Marker}s,
 * and adding log parameters that are formatted in a consistent way.
 * <p>
 * Please note that this class is mutable and is not designed to be used in multi-threaded context. Do not try to reuse an instance
 * for multiple logging operations. A new instance should be obtained for each log operation.
 */
@SuppressWarnings("PMD.LoggerIsNotStaticFinal")
public class ParameterLogger extends SimpleLogger
{
	private Map<String, Object> params;

	/**
	 * No public constructor, as client code would look messy when chaining.
	 * <p>
	 * Use {@link #newInstance(Logger, String)} instead.
	 */
	protected ParameterLogger(final Logger log)
	{
		super(log);
	}

	/**
	 * Creates a new instance of {@code ParameterLogger}.
	 *
	 * @param log the {@code slf4j Logger} to wrap.
	 * @param tags the tags, which will be used as a {@link org.slf4j.Marker} for the messages logged by this
	 *           {@code ParameterLogger}.
	 *           May be left empty, and specified later using {@link #tags}.
	 * @return new ParameterLogger instance.
	 */
	public static ParameterLogger newInstance(final Logger log, final String... tags)
	{
		final ParameterLogger instance = new ParameterLogger(log);
		instance.tags(tags);
		return instance;
	}

	/**
	 * Adds a custom parameter, in form of a name-value-pair, to this {@code ParameterLogger}.
	 * <p>
	 * It will be included in the parameters list in the messages logged by this {@code ParameterLogger}.
	 * <p>
	 * Custom parameters set through this method will override any parameters implicitly set by other method of this
	 * class. Use a {@code null} value to suppress an implicit value with the given key.
	 *
	 * @param name the name of the custom parameter
	 * @param value the value of the custom parameter
	 * @return {@code this}, for chaining
	 */
	public ParameterLogger parameter(final String name, final Object value)
	{
		if (params == null)
		{
			params = new LinkedHashMap<>();
		}
		params.put(name, value);
		return this;
	}

	/**
	 * Hook method that merges all logging parameters of this {@link ParameterLogger} right before a log message is
	 * emitted.
	 * <p>
	 * Subclasses can override this method to add their own logging parameters. However they should always invoke
	 * {@code super.mergeParams()} to take into account the custom parameters managed by super-classes, like
	 * {@code ParameterLogger}.
	 * 
	 * @param parameterHolder the Map to merge the parameters on.
	 * @return the merged parameters.
	 */
	protected Map<String, Object> mergeParameters(final Map<String, Object> parameterHolder)
	{
		if (params != null)
		{
			for (final Entry<String, Object> param : params.entrySet())
			{
				if (param.getValue() != null)
				{
					parameterHolder.put(param.getKey(), param.getValue());
				}
				else
				{
					parameterHolder.remove(param.getKey());
				}
			}
		}
		return parameterHolder;
	}

	@Override
	protected String formatMessage(final String message)
	{
		return ParameterMessageFormatter.format(message, mergeParameters(new LinkedHashMap<>()));
	}
}
