/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.Map;
import java.util.Map.Entry;


/**
 * Utility class that helps formatting log messages in a consistent manner.
 */
public final class ParameterMessageFormatter
{
	/** This class shall not be instantiated, thus its only constructor is private. */
	private ParameterMessageFormatter()
	{
		// nothing to do, only prevents instantiation
	}

	/**
	 * Format the given parameters according to the following pattern:
	 * <p>
	 * <code>
	 * [<em>paramName1</em>=<em>paramValue1</em>, <em>paramValue2</em>=<em>paramValue2</em>, <em>...</em>]
	 * <em>message</em>
	 * </code>
	 *
	 * @param message a custom message
	 * @param params a map representing additional parameters
	 * @return the formatted message
	 */
	public static String format(final String message, final Map<String, Object> params)
	{
		final StringBuilder b = new StringBuilder();
		appendParams(b, params);
		b.append(message);
		return b.toString();
	}

	/**
	 * Helper method to add additional parameters to a message.
	 *
	 * @param b a builder representing the message.
	 * @param params a map representing additional parameters
	 */
	private static void appendParams(final StringBuilder b, final Map<String, Object> params)
	{
		if (params != null && !params.isEmpty())
		{
			b.append('[');
			for (final Entry<String, Object> param : params.entrySet())
			{
				b.append(param.getKey()).append('=');

				final Object value = param.getValue();
				if (value != null)
				{
					b.append(param.getValue());
				}
				// append list-item-seperator
				b.append(", ");
			}
			// compensate for last list-item-separator
			b.delete(b.length() - 2, b.length());
			b.append("] ");
		}
	}
}
