/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;



/**
 * A wrapper around a {@code slf4j} {@link Logger} that adds convenience features for logging.
 * <p>
 * In particular this includes setting multiple tags, which are mapped to {@code slf4j} {@link Marker}s.
 */
@SuppressWarnings("PMD.LoggerIsNotStaticFinal")
public class SimpleLogger
{
	private final Logger log;

	private final Set<String> theTags = new LinkedHashSet<String>();

	/**
	 * No public constructor, as client code would look messy when chaining.
	 * <p>
	 * Use {@link #newInstance(Logger, String)} instead.
	 */
	protected SimpleLogger(final Logger log)
	{
		this.log = log;
	}

	/**
	 * Creates a new instance of {@code SimpleLogger}.
	 *
	 * @param log the {@code slf4j Logger} to wrap.
	 * @param tags the tags, which will be used as a {@link Marker} for the messages logged by this
	 *           {@code SimpleLogger}.
	 *           May be left empty, and specified later using {@link #tags}.
	 * @return new SimpleLogger instance.
	 */
	public static SimpleLogger newInstance(final Logger log, final String... tags)
	{
		final SimpleLogger instance = new SimpleLogger(log);
		instance.tags(tags);
		return instance;
	}

	/**
	 * Adds tags to this {@code SimpleLogger}.
	 * 
	 * @param tags additional tags to be used as a {@link Marker} for the messages logged by this {@code SimpleLogger}.
	 *           These will be added to tags that have been added before.
	 *           Duplicates and {@code null} values will be silently ignored.
	 * @return {@code this}, for chaining
	 */
	public SimpleLogger tags(final String... tags)
	{
		Stream
				.of(tags)
				.filter(Objects::nonNull)
				.forEach(tag -> theTags.add(tag));
		return this;
	}

	/**
	 * Adds topics to this {@code SimpleLogger}.
	 * 
	 * @deprecated The term <em>topic</em> has been replaced by the term <em>tag</em>, which is more consistent with the
	 *           YaaS Logging patterns. Consequently use the {@link #theTags} method instead.
	 * @param topic the tag to be used as a {@link Marker} for the messages logged by this {@code SimpleLogger}.
	 *           {@code null} values will be silently ignored.
	 * @return {@code this}, for chaining
	 */
	@Deprecated
	public SimpleLogger addTopic(final String topic)
	{
		return tags(topic);
	}

	/**
	 * Logs a message at debug level, using the wrapped {@link Logger}.
	 * <p>
	 * Uses the {@link #formatMessage(String)} to build the actual log message.
	 *
	 * @see #newInstance(Logger, String...)
	 * @see #theTags
	 *
	 * @param message the custom message to include into the log message
	 * @param arguments additional arguments, passed through to the wrapped {@link Logger}
	 */
	public void debug(final String message, final Object... arguments)
	{
		if (log.isDebugEnabled())
		{
			wrapLogAction(() -> log.debug(createMarker(), formatMessage(message), arguments)).log();
		}
	}

	/**
	 * Logs a message at info level, using the wrapped {@link Logger}.
	 * <p>
	 * Uses the {@link #formatMessage(String)} to build the actual log message.
	 *
	 * @see #newInstance(Logger, String...)
	 * @see #theTags
	 *
	 * @param message the custom message to include into the log message
	 * @param arguments additional arguments, passed through to the wrapped {@link Logger}
	 */
	public void info(final String message, final Object... arguments)
	{
		if (log.isInfoEnabled())
		{
			wrapLogAction(() -> log.info(createMarker(), formatMessage(message), arguments)).log();
		}
	}

	/**
	 * Logs a message at warn level, using the wrapped {@link Logger}.
	 * <p>
	 * Uses the {@link #formatMessage(String)} to build the actual log message.
	 *
	 * @see #newInstance(Logger, String...)
	 * @see #theTags
	 *
	 * @param message the custom message to include into the log message
	 * @param arguments additional arguments, passed through to the wrapped {@link Logger}
	 */
	public void warn(final String message, final Object... arguments)
	{
		if (log.isWarnEnabled())
		{
			wrapLogAction(() -> log.warn(createMarker(), formatMessage(message), arguments)).log();
		}
	}

	/**
	 * Logs a message at error level, using the wrapped {@link Logger}.
	 * <p>
	 * Uses the {@link #formatMessage(String)} to build the actual log message.
	 *
	 * @see #newInstance(Logger, String...)
	 * @see #theTags
	 *
	 * @param message the custom message to include into the log message
	 * @param arguments additional arguments, passed through to the wrapped {@link Logger}
	 */
	public void error(final String message, final Object... arguments)
	{
		if (log.isErrorEnabled())
		{
			wrapLogAction(() -> log.error(createMarker(), formatMessage(message), arguments)).log();
		}
	}

	/**
	 * Hook method that allows sub-classes to perform additional actions before or after the actual logAction is invoked,
	 * or even block it altogether.
	 * 
	 * @param logAction the action to be performed by this {@link SimpleLogger} to emit the log event.
	 * @return the {@code logAction} itself (default implementation).
	 */
	protected LogAction wrapLogAction(final LogAction logAction)
	{
		return logAction;
	}

	/**
	 * Allows to specify custom message formatting.
	 * 
	 * @param message the raw message
	 * @return the formatted message
	 */
	protected String formatMessage(final String message)
	{
		// do nothing by default
		return message;
	}

	/**
	 * Prepares a {@code Marker} based on all configured {@code tags}.
	 * <p>
	 * The {@code Marker} and all referenced {@code Marker}s will be detached, in order to avoid side-effects on shared
	 * {@code Markers}.
	 * 
	 * @return the {@code Marker}.
	 */
	protected Marker createMarker()
	{
		Marker marker = null;
		for (final String tag : theTags)
		{
			if (marker == null)
			{
				marker = MarkerFactory.getDetachedMarker(tag);
			}
			else
			{
				marker.add(MarkerFactory.getDetachedMarker(tag));
			}
		}
		return marker;
	}
}
