/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging;

import ch.qos.logback.classic.spi.ILoggingEvent;

import java.io.IOException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import net.logstash.logback.composite.JsonWritingUtils;
import net.logstash.logback.composite.loggingevent.MdcJsonProvider;

import com.fasterxml.jackson.core.JsonGenerator;


/**
 * Extension of {@link net.logstash.logback.composite.loggingevent.MdcJsonProvider} supporting type conversion for
 * included elements.
 */
public class TypedMdcJsonProvider extends MdcJsonProvider
{
	private final Map<String, Class<?>> includeMdcKeys = new LinkedHashMap<String, Class<?>>();

	@Override
	public void writeTo(final JsonGenerator generator, final ILoggingEvent event) throws IOException
	{
		Map<String, String> mdcProperties = event.getMDCPropertyMap();
		if (mdcProperties != null && !mdcProperties.isEmpty())
		{
			if (!getIncludeMdcKeyNames().isEmpty())
			{
				final Map<String, String> usedMdcProperties = new LinkedHashMap<String, String>();
				for (final String includeKey : getIncludeMdcKeyNames())
				{
					usedMdcProperties.put(includeKey, mdcProperties.get(includeKey));
				}
				mdcProperties = usedMdcProperties;
			}
			if (!getExcludeMdcKeyNames().isEmpty())
			{
				mdcProperties = new LinkedHashMap<String, String>(mdcProperties);
				mdcProperties.keySet().removeAll(getExcludeMdcKeyNames());
			}

			final Map<String, Object> fieldsToWrite = new LinkedHashMap<>();
			for (final Map.Entry<String, String> mdcProperty : mdcProperties.entrySet())
			{
				final Class<?> targetClass = includeMdcKeys.get(mdcProperty.getKey());
				fieldsToWrite.put(mdcProperty.getKey(), convertValue(mdcProperty.getValue(), targetClass));
			}

			if (getFieldName() != null)
			{
				generator.writeObjectFieldStart(getFieldName());
			}

			JsonWritingUtils.writeMapEntries(generator, fieldsToWrite);
			if (getFieldName() != null)
			{
				generator.writeEndObject();
			}
		}
	}

	private Object convertValue(final String value, final Class<?> targetClass)
	{
		if (value != null && targetClass != null && Integer.class.isAssignableFrom(targetClass))
		{
			try
			{
				return Integer.valueOf(value);
			}
			catch (final NumberFormatException e)
			{
				// ignore silently - a log here will spoil current log entry
				// do not return value as it is in wrong format
				return null;
			}
		}
		return value;
	}

	/**
	 * Adds an MDC key having a specific type attached. The passed value needs to be conform to that pattern:
	 * "key,full class name" like "myKey,java.lang.Integer".
	 * 
	 * @param includedMdcKeyName MDC key to include in document including optional type of value
	 */
	@Override
	public void addIncludeMdcKeyName(final String includedMdcKeyName)
	{
		final String[] splittedName = includedMdcKeyName.split(",");
		if (splittedName.length > 2)
		{
			addError("Invalid pattern " + includedMdcKeyName
					+ " used for specifying an MDC conversion key, use [key, full class name]");
		}
		else if (splittedName.length == 2)
		{
			final String key = splittedName[0].trim();
			try
			{
				final Class<?> clazz = Class.forName(splittedName[1].trim());
				super.addIncludeMdcKeyName(key);
				this.includeMdcKeys.put(key, clazz);
			}
			catch (final ClassNotFoundException e)
			{
				addError("Class " + splittedName[1] + " specified for MDC conversion cannot be found");
			}
		}
		else if (splittedName.length == 1)
		{
			final String key = splittedName[0].trim();
			super.addIncludeMdcKeyName(key);
			this.includeMdcKeys.put(key, String.class);
		}
	}

	/**
	 * Sets all MDC keys to include in the generated JSON document.
	 * 
	 * @param includeMdcKeyNames MDC keys to include
	 */
	@Override
	public void setIncludeMdcKeyNames(final List<String> includeMdcKeyNames)
	{
		super.setIncludeMdcKeyNames(Collections.<String>emptyList());
		for (final String name : includeMdcKeyNames)
		{
			addIncludeMdcKeyName(name);
		}
	}
}
