/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging.audit;

import com.sap.cloud.yaas.servicesdk.patternsupport.traits.SessionAwareTrait;
import com.sap.cloud.yaas.servicesdk.patternsupport.traits.YaasAwareTrait;

import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;


/**
 * An Audit Logger that takes information from a HttpServletRequest into account, and fills log parameters accordingly.
 * 
 * @see #addRequest(HttpServletRequest)
 * 
 * <p>
 * Please note that this class is mutable and is not designed to be used in multi-threaded context. Do not try to reuse an instance
 * for multiple logging operations. A new instance should be obtained for each log operation.
 */
public class ServletAuditLogger extends SimpleAuditLogger
{
	private HttpServletRequest request;

	/**
	 * No public constructor, as client code would look messy when chaining.
	 * <p>
	 * Use {@link #newInstance(Logger, String)} instead.
	 */
	protected ServletAuditLogger(final Logger log)
	{
		super(log);
	}

	/**
	 * Creates a new instance of {@code ServletAuditLogger}.
	 *
	 * @param log the {@code slf4j Logger} to wrap.
	 * @param tags the tags, which will be used as a {@link org.slf4j.Marker} for the messages logged by this
	 *           {@code SimpleAuditLogger}.
	 *           May be left empty, and specified later using {@link #tags}.
	 * @return new ServletAuditLogger instance.
	 */
	public static ServletAuditLogger newInstance(final Logger log, final String... tags)
	{
		final ServletAuditLogger instance = new ServletAuditLogger(log);
		instance.tags(tags);
		return instance;
	}

	/**
	 * Adds a {@code HttpServletRequest} to this {@code AuditLogger}.
	 * <p>
	 * Its method, uri, hybris-scopes, hybris-user, and hybris-session will be included in the log parameters when a
	 * message is logged.
	 *
	 * @param setRequest the {@code Request} to set. {@code null} will be ignored
	 * @return {@code this}, for chaining
	 */
	public ServletAuditLogger addRequest(final HttpServletRequest setRequest)
	{
		if (setRequest != null)
		{
			this.request = setRequest;
		}
		return this;
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * This implementation fills in log parameters for method, uri, hybris-scopes, hybris-user, and hybris-session from
	 * the {@link HttpServletRequest}.
	 * 
	 * @see #addRequest(HttpServletRequest)
	 */
	@Override
	protected Map<String, Object> mergeParameters(final Map<String, Object> paramsHolder)
	{
		if (request != null)
		{
			paramsHolder.put(PARAM_METHOD, request.getMethod());
			paramsHolder.put(PARAM_URI, request.getServletPath() + request.getPathInfo());
			paramsHolder.put(PARAM_SCOPES, request.getHeader(YaasAwareTrait.Headers.SCOPES));
			paramsHolder.put(PARAM_USER, request.getHeader(YaasAwareTrait.Headers.USER));
			paramsHolder.put(PARAM_SESSION, request.getHeader(SessionAwareTrait.Headers.SESSION_ID));
		}
		return super.mergeParameters(paramsHolder);
	}
}
