/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging.audit;

import java.util.Map;

import org.slf4j.Logger;

import com.sap.cloud.yaas.servicesdk.logging.ParameterLogger;



/**
 * A wrapper around a {@code slf4j} {@link Logger} that adds convenience features for audit logging.
 * <p>
 * In particular this includes setting multiple tags, which are mapped to {@code slf4j} {@link org.slf4j.Marker}s,
 * and adding log parameters that are formatted in a consistent way.
 * <p>
 * Note that subclasses provide even more convenient methods to draw useful log parameters from different sources, like
 * HTTP headers and similar.
 */
@SuppressWarnings("PMD.LoggerIsNotStaticFinal")
public class SimpleAuditLogger extends ParameterLogger
{
	/**
	 * The name used for formatting the URI parameter in the log output.
	 * <p>
	 * Use this constant explicitly when setting a URI parameter manually, utilizing {@link #addParam(String, Object)}.
	 */
	public static final String PARAM_URI = "uri";

	/**
	 * The name used for formatting the HTTP method parameter in the log output.
	 * <p>
	 * Use this constant explicitly when setting an HTTP method parameter manually, utilizing
	 * {@link #addParam(String, Object)}.
	 */
	public static final String PARAM_METHOD = "method";

	/**
	 * The name used for formatting the {@code hybris-scopes} HTTP header parameter in the log output.
	 * <p>
	 * Use this constant explicitly when setting a {@code hybris-scopes} HTTP header parameter manually, utilizing
	 * {@link #addParam(String, Object)}.
	 */
	public static final String PARAM_SCOPES = "scopes";

	/**
	 * The name used for formatting the {@code hybris-user} HTTP header parameter in the log output.
	 * <p>
	 * Use this constant explicitly when setting a {@code hybris-user} HTTP header parameter manually, utilizing
	 * {@link #addParam(String, Object)}.
	 */
	public static final String PARAM_USER = "user";

	/**
	 * The name used for formatting the {@code hybris-session} HTTP header parameter in the log output.
	 * <p>
	 * Use this constant explicitly when setting a {@code hybris-session} HTTP header parameter manually, utilizing
	 * {@link #addParam(String, Object)}.
	 */
	public static final String PARAM_SESSION = "session";

	private static final String AUDIT_MARKER = "AUDIT";

	/**
	 * No public constructor, as client code would look messy when chaining.
	 * <p>
	 * Use {@link #newInstance(Logger, String)} instead.
	 */
	protected SimpleAuditLogger(final Logger log)
	{
		super(log);
		tags(AUDIT_MARKER);
	}

	/**
	 * Creates a new instance of {@code SimpleAuditLogger}.
	 *
	 * @param log the {@code slf4j Logger} to wrap.
	 * @param tags the tags, which will be used as a {@link org.slf4j.Marker} for the messages logged by this
	 *           {@code SimpleAuditLogger}.
	 *           May be left empty, and specified later using {@link #tags}.
	 * @return new SimpleAuditLogger instance.
	 */
	public static SimpleAuditLogger newInstance(final Logger log, final String... tags)
	{
		final SimpleAuditLogger instance = new SimpleAuditLogger(log);
		instance.tags(tags);
		return instance;
	}

	/**
	 * Adds a custom parameter, in form of a name-value-pair.
	 * 
	 * @deprecated Use {@link #parameter} instead.
	 * @param name The name.
	 * @param value The value.
	 * @return {@code this}, for chaining
	 */
	@Deprecated
	public SimpleAuditLogger addParam(final String name, final Object value)
	{
		parameter(name, value);
		return this;
	}

	/**
	 * Legacy variant of the of the {@code mergeParameters} method, with the same semantics.
	 * 
	 * @deprecated Override {@link #mergeParameters} instead.
	 * @param paramsHolder the Map to merge the parameters on.
	 * @return the merged parameters.
	 */
	@Deprecated
	protected Map<String, Object> mergeParams(final Map<String, Object> paramsHolder)
	{
		return paramsHolder;
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * This implementation merely ensures that the legacy-variant of this method, {@link #mergeParams}, also gets considered.
	 */
	@Override
	protected Map<String, Object> mergeParameters(final Map<String, Object> parameterHolder)
	{
		return super.mergeParameters(mergeParams(parameterHolder));
	}
}
