/*
 * © 2015 SAP SE or an SAP affiliate company.
 * All rights reserved.
 * Please see http://www.sap.com/corporate-en/legal/copyright/index.epx for additional trademark information and
 * notices.
 */
package com.sap.cloud.yaas.servicesdk.logging.metric;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.sap.cloud.yaas.servicesdk.logging.LogAction;
import com.sap.cloud.yaas.servicesdk.logging.MDCFields;
import com.sap.cloud.yaas.servicesdk.logging.SimpleLogger;


/**
 * A wrapper around a {@code slf4j} {@link Logger} that adds convenience features for logging metrics using the "METRIC"
 * log pattern.
 * <p>
 * In particular this includes setting the metric specific data in the {@link org.slf4j.MDC} like the metric type and
 * the metric key.
 * Assigning the metric to a specific tenant and client is supported as well.
 * Furthermore, it will set the required "METRIC" {@link org.slf4j.Marker} while supporting further tags,
 * which are mapped to {@code Marker}s as well.
 * <p>
 * Please note that this class is mutable and is not designed to be used in multi-threaded context. Do not try to reuse an instance
 * for multiple logging operations. A new instance should be obtained for each log operation.
 */
@SuppressWarnings("PMD.TooManyMethods")
public final class MetricLogger extends SimpleLogger
{
	/**
	 * Possible {@value #KEY_TYPE}s of the value of a metric.
	 */
	public static enum MetricType
	{
		/** Indicates a numeric metric value. */
		NUMBER("number"),
		/** Indicates a textual metric value. */
		STRING("string");

		private String value;

		private MetricType(final String value)
		{
			this.value = value;
		}

		/**
		 * Returns the String representation of this {@code MetricType}, as used in the "METRIC" log pattern.
		 * 
		 * @return the String representation.
		 */
		@Override
		public String toString()
		{
			return value;
		}
	}

	/** The name of the marker to mark a log as a metric. */
	public static final String METRIC_TAG = "METRIC";

	/** The name of the marker to mark a metric as a business metric. */
	public static final String BUSINESS_METRIC_TAG = "BUSINESS";

	/** The MDC key for setting the metric name. **/
	public static final String KEY_METRIC = "metric";
	/** The MDC key for setting the metric type. **/
	public static final String KEY_TYPE = "type";

	private static final Logger DEFAULT_LOGGER = LoggerFactory.getLogger(MetricLogger.class);

	private MetricType theType = MetricType.NUMBER;
	private String theMetric = "unknown";
	private String theTenant;
	private String theClient;
	private String theClientOwner;
	private String theServiceOwner;

	/**
	 * No public constructor, as client code would look messy when chaining.
	 * <p>
	 * Use {@link #newInstance(Logger, String)} instead.
	 */
	protected MetricLogger(final Logger log)
	{
		super(log);
		tags(METRIC_TAG);
	}

	/**
	 * Creates a new instance of {@code MetricLogger}.
	 * 
	 * @param tags the tags, which will be used as a {@link org.slf4j.Marker} for the messages logged by this
	 *           {@code SimpleAuditLogger}.
	 *           May be left empty, and specified later using {@link #tags}.
	 * @return new MetricLogger instance.
	 */
	public static MetricLogger newInstance(final String... tags)
	{
		return newInstance(DEFAULT_LOGGER, tags);
	}

	/**
	 * Creates a new instance of {@code MetricLogger}.
	 * 
	 * @param log the {@code slf4j Logger} to wrap.
	 * @param tags the tags, which will be used as a {@link org.slf4j.Marker} for the messages logged by this
	 *           {@code SimpleAuditLogger}.
	 *           May be left empty, and specified later using {@link #tags}.
	 * @return new MetricLogger instance.
	 */
	public static MetricLogger newInstance(final Logger log, final String... tags)
	{
		final MetricLogger instance = new MetricLogger(log);
		instance.tags(tags);
		return instance;
	}

	/**
	 * Creates a new instance of {@code MetricLogger} for creating business metrics, including the tag
	 * {@value #BUSINESS_METRIC_TAG}.
	 *
	 * @param tags the tags, which will be used as a {@link org.slf4j.Marker} for the messages logged by this
	 *           {@code SimpleAuditLogger}.
	 *           May be left empty, and specified later using {@link #tags}.
	 * @return new MetricLogger instance.
	 */
	public static MetricLogger newBusinessInstance(final String... tags)
	{
		return newBusinessInstance(DEFAULT_LOGGER, tags);
	}

	/**
	 * Creates a new instance of {@code MetricLogger} for creating business metrics, including the tag
	 * {@value #BUSINESS_METRIC_TAG}.
	 *
	 * @param log the {@code slf4j Logger} to wrap.
	 * @param tags the tags, which will be used as a {@link org.slf4j.Marker} for the messages logged by this
	 *           {@code SimpleAuditLogger}.
	 *           May be left empty, and specified later using {@link #tags}.
	 * @return new MetricLogger instance.
	 */
	public static MetricLogger newBusinessInstance(final Logger log, final String... tags)
	{
		final MetricLogger result = newInstance(log, BUSINESS_METRIC_TAG);
		result.tags(tags);
		return result;
	}

	/**
	 * Sets the metric type of this {@code MetricLogger}.
	 * <p>
	 * It will be used in the MDC context when a metric value is logged to identify the metric type.
	 * The default of {@link MetricType#NUMBER} will be used by new instances of {@code MetricLogger}, if not set
	 * otherwise.
	 * 
	 * @param type the metric type to set. {@code null} will be ignored
	 * @return {@code this}, for chaining
	 */
	public MetricLogger type(final MetricType type)
	{
		if (type != null)
		{
			theType = type;
		}
		return this;
	}

	/**
	 * Convenience method that sets the metric type to be {@link MetricType#NUMBER}, which is also the default for new
	 * instances of {@code MetricLogger}.
	 * 
	 * @return {@code this}, for chaining
	 */
	public MetricLogger numeric()
	{
		return type(MetricType.NUMBER);
	}

	/**
	 * Convenience method that sets the metric type to be {@link MetricType#STRING}.
	 * 
	 * @return {@code this}, for chaining
	 */
	public MetricLogger textual()
	{
		return type(MetricType.STRING);
	}

	/**
	 * Adds a metric name to this {@code MetricLogger}.
	 * <p>
	 * It will be used in the MDC context when a metric value is logged to identify the metric.
	 * 
	 * @param metric the metric name to set. {@code null} will be ignored
	 * @return {@code this}, for chaining
	 */
	public MetricLogger metric(final String metric)
	{
		if (metric != null)
		{
			theMetric = metric;
		}
		return this;
	}

	/**
	 * Assigns the metrics to be logged to a specific tenant.
	 * <p>
	 * Its value will be used in the MDC context when a metric value is logged, to identify the related tenant.
	 *
	 * @param tenant the tenant to set. {@code null} will be ignored
	 * @return {@code this}, for chaining
	 */
	public MetricLogger tenant(final String tenant)
	{
		if (tenant != null)
		{
			theTenant = tenant;
		}
		return this;
	}

	/**
	 * Assigns the metrics to be logged to a specific client.
	 * <p>
	 * Its value will be used in the MDC context when a metric value is logged, to identify the related client.
	 *
	 * @param client the client to set. {@code null} will be ignored
	 * @return {@code this}, for chaining
	 */
	public MetricLogger client(final String client)
	{
		if (client != null)
		{
			theClient = client;
		}
		return this;
	}

	/**
	 * Assigns the metrics to be logged to a specific client's owner project.
	 * <p>
	 * Its value will be used in the MDC context when a metric value is logged, to identify
	 * the owner project of the client making the call.
	 *
	 * @param clientOwner the identifier of the client's owner to set. {@code null} will be ignored
	 * @return {@code this}, for chaining
	 */
	public MetricLogger clientOwner(final String clientOwner)
	{
		if (clientOwner != null)
		{
			theClientOwner = clientOwner;
		}
		return this;
	}

	/**
	 * Assigns the metrics to be logged to a specific client's owner project.
	 * <p>
	 * Its value will be used in the MDC context when a metric value is logged, to identify
	 * the owner project of the client making the call.
	 *
	 * @param serviceOwner the identifier of the serveice's owner to set. {@code null} will be ignored
	 * @return {@code this}, for chaining
	 */
	public MetricLogger serviceOwner(final String serviceOwner)
	{
		if (serviceOwner != null)
		{
			theServiceOwner = serviceOwner;
		}
		return this;
	}

	/**
	 * Logs a metric at info level, using the wrapped {@link Logger}.
	 *
	 * @see #newInstance(Logger, String...)
	 * @see #tags
	 *
	 * @param metricValue the metric value to log
	 */
	public void log(final Object metricValue)
	{
		info(String.valueOf(metricValue));
	}

	@Override
	protected LogAction wrapLogAction(final LogAction logAction)
	{
		return
				LogAction.wrapWithDiagnosticContext(
					MDCFields.CLIENT,
					theClient,
				LogAction.wrapWithDiagnosticContext(
					MDCFields.CLIENT_OWNER,
					theClientOwner,
				LogAction.wrapWithDiagnosticContext(
					MDCFields.SERVICE_OWNER,
					theServiceOwner,
				LogAction.wrapWithDiagnosticContext(
					MDCFields.TENANT,
					theTenant,
				LogAction.wrapWithDiagnosticContext(
					KEY_METRIC,
					theMetric,
				LogAction.wrapWithDiagnosticContext(
					KEY_TYPE,
					theType.toString(),
				logAction))))));
	}
}
