/**
 * Copyright (C) 2013-2014 Sappenin Inc. (developers@sappenin.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.sappenin.objectify.translate;

import com.google.appengine.api.datastore.EmbeddedEntity;
import com.googlecode.objectify.impl.Path;
import com.googlecode.objectify.impl.TypeUtils;
import com.googlecode.objectify.impl.translate.CreateContext;
import com.googlecode.objectify.impl.translate.LoadContext;
import com.googlecode.objectify.impl.translate.NullSafeTranslator;
import com.googlecode.objectify.impl.translate.SaveContext;
import com.googlecode.objectify.impl.translate.SkipException;
import com.googlecode.objectify.impl.translate.Translator;
import com.googlecode.objectify.impl.translate.TranslatorFactory;
import com.googlecode.objectify.impl.translate.TypeKey;
import com.googlecode.objectify.repackaged.gentyref.GenericTypeReflector;
import com.googlecode.objectify.util.DatastoreUtils;
import com.googlecode.objectify.util.GenericUtils;
import com.sappenin.objectify.translate.util.BigDecimalCodec;

import java.lang.reflect.Type;
import java.math.BigDecimal;

/**
 * <p> This a more advanced strategy for storing java.math.BigDecimal in the datastore than the standard Objectify
 * translators. This implementation encodes the {@link java.math.BigDecimal} value and stores the result as a String.
 * This is appropriate for monetary and other values of ~475 digits (the end encoded value length is number dependent).
 * This strategy offers the following advantages over encoding a java.math.BigDecimal as a Long: <ul> <li>Offers support
 * for arbitrary precision numbers</li> <li>Supports very large numbers (~475 digits long)</li> <li>Encodes numbers in
 * lexigraphical order, which allows for native sorting in Datastore queries. </ul> </p> <p> For more details, read this
 * blog entry: "http://softwareblog.sappenin.com/2014/05/best-practices-for-storing-bigdecimal.html" </p> <p> This
 * translator is not installed by default, but can be installed as follows: </p>
 * <p/>
 * <pre>
 * ObjectifyService.factory().getTranslators().add(new BigDecimalStringTranslatorFactory());
 * </pre>
 *
 * @author David Fuelling
 */
public class BigDecimalEmbeddedEntityTranslatorFactory implements TranslatorFactory<BigDecimal, EmbeddedEntity>
{

	@Override
	public Translator<BigDecimal, EmbeddedEntity> create(TypeKey<BigDecimal> tk, CreateContext ctx, Path path)
	{

		if (java.math.BigDecimal.class.isAssignableFrom(GenericTypeReflector.erase(tk.getType())))
		{
			return new BigDecimalTranslator(tk, ctx, path);
		}
		else
		{
			return null;
		}
	}

	/**
	 * Translator which knows what to do with a {@link java.math.BigDecimal}.<br/> <br/> This class utilizes an
	 * optional
	 * {@link com.sappenin.objectify.annotation.BigDecimal} annotation which allows for fine-grained control over the
	 * field names used to store the java.math.BigDecimal information, as well as indexing of each sub-field. See the
	 * Javadoc of that annotation for more details.
	 *
	 * @author David Fuelling
	 */
	static class BigDecimalTranslator extends NullSafeTranslator<BigDecimal, EmbeddedEntity>
	{
		private final Translator<Object, Object> componentTranslator;

		private boolean storeDisplayableAmount;

		private boolean indexDisplayableAmount;

		private boolean indexAmount;

		private String encodedAmountFieldName = "encodedAmount";

		private String displayableAmountFieldName = "displayableAmount";

		/**
		 * Required Args Constructor.
		 *
		 * @param tk
		 * @param ctx
		 * @param path
		 */
		public BigDecimalTranslator(TypeKey<BigDecimal> tk, final CreateContext ctx, final Path path)
		{
			// Look for an @BigDecimal Annotation, if present.
			com.sappenin.objectify.annotation.BigDecimal bigDecimalAnnotation = TypeUtils
					.getAnnotation(tk.getAnnotations(), com.sappenin.objectify.annotation.BigDecimal.class);

			Type componentType = GenericUtils.getMapValueType(tk.getType());
			componentTranslator = ctx.getFactory().getTranslators().get(new TypeKey<>(componentType, tk), ctx, path);

			if (bigDecimalAnnotation != null)
			{
				storeDisplayableAmount = bigDecimalAnnotation.storeDisplayableAmount();
				indexAmount = bigDecimalAnnotation.indexEncodedAmount();

				indexDisplayableAmount = bigDecimalAnnotation.indexDisplayableAmount();
				encodedAmountFieldName = bigDecimalAnnotation.encodedAmountFieldName();
				displayableAmountFieldName = bigDecimalAnnotation.displayableAmountFieldName();
			}
		}

		@Override
		protected BigDecimal loadSafe(EmbeddedEntity node, LoadContext ctx, Path path) throws SkipException
		{

			BigDecimal returnable = null;

			// Get the amount as a java.math.BigDecimal
			if (node.hasProperty(encodedAmountFieldName))
			{
				Object encodedAmountFieldValue = node.getProperty(encodedAmountFieldName);

				// Object amountValue = amountNode.getPropertyValue();
				if ((encodedAmountFieldValue != null) && (encodedAmountFieldValue.toString().length() > 0))
				{
					// //////////
					// Get the CurrencyUnit, defaulting to USD
					// //////////
					BigDecimal bdValue = null;
					try
					{
						bdValue = BigDecimalCodec.decodeAsBigDecimal(encodedAmountFieldValue.toString());
					}
					catch (Exception e)
					{
						System.err.print("Unable to Decode java.math.BigDecimal from encoded string \""
								+ encodedAmountFieldValue + "\"");
						e.printStackTrace();
					}

					returnable = bdValue;
				}
			}

			return returnable;

		}

		@Override
		protected EmbeddedEntity saveSafe(BigDecimal pojo, boolean index, SaveContext ctx, Path path)
				throws SkipException
		{

			if (pojo == null)
			{
				throw new SkipException();
			}
			else
			{
				EmbeddedEntity emb = new EmbeddedEntity();

				// /////////////////////////
				// Handle the EncodedAmount
				// /////////////////////////

				// Path encodedAmountPath = path.extend(encodedAmountFieldName);
				// Encode the Amount value as a String
				{
					String encodedAmountValue = BigDecimalCodec.encode(pojo);
					Path propPath = path.extend(encodedAmountFieldName);
					Object value = componentTranslator.save(encodedAmountValue, indexAmount, ctx, propPath);
					DatastoreUtils.setContainerProperty(emb, encodedAmountFieldName, value, indexAmount, ctx,
							propPath);
				}

				// /////////////////////////
				// Handle the DisplayableAmount
				// /////////////////////////

				// DisplayableAmountPath (This is never loaded)
				if (storeDisplayableAmount)
				{
					String displayableValue = pojo.toString();
					Path propPath = path.extend(displayableAmountFieldName);
					Object value = componentTranslator.save(displayableValue, indexDisplayableAmount, ctx, propPath);
					DatastoreUtils
							.setContainerProperty(emb, displayableAmountFieldName, value, indexDisplayableAmount, ctx,
									propPath);
				}

				return emb;
			}
		}
	}

}
