/**
 * Copyright (C) 2013-2014 Sappenin Inc. (developers@sappenin.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.sappenin.objectify.translate;

import com.google.appengine.api.datastore.EmbeddedEntity;
import com.googlecode.objectify.impl.Path;
import com.googlecode.objectify.impl.TypeUtils;
import com.googlecode.objectify.impl.translate.CreateContext;
import com.googlecode.objectify.impl.translate.LoadContext;
import com.googlecode.objectify.impl.translate.NullSafeTranslator;
import com.googlecode.objectify.impl.translate.SaveContext;
import com.googlecode.objectify.impl.translate.SkipException;
import com.googlecode.objectify.impl.translate.Translator;
import com.googlecode.objectify.impl.translate.TranslatorFactory;
import com.googlecode.objectify.impl.translate.TypeKey;
import com.googlecode.objectify.repackaged.gentyref.GenericTypeReflector;
import com.googlecode.objectify.util.DatastoreUtils;
import com.googlecode.objectify.util.GenericUtils;
import com.sappenin.objectify.annotation.Money;
import com.sappenin.objectify.translate.util.BigDecimalCodec;
import org.joda.money.BigMoney;
import org.joda.money.BigMoneyProvider;
import org.joda.money.CurrencyUnit;

import java.lang.reflect.Type;
import java.math.BigDecimal;

/**
 * Factory for creating a {@link JodaMoneyTranslator} which can store component properties of a {@link
 * org.joda.money.BigMoneyProvider} as a "value" and a "currencyUnit" (both of which are Strings), similar to how an
 * Embedded class would be stored. The {@code value} is an encoded-String version of a {@link java.math.BigDecimal} that
 * supports lexicographical sorting and large-digit number sets. <p> For more details, read this blog entry:
 * "http://softwareblog.sappenin.com/2014/05/best-practices-for-storing-joda-money.html" </p>
 *
 * @author David Fuelling
 */
public class JodaMoneyEmbeddedEntityTranslatorFactory implements TranslatorFactory<BigMoneyProvider, EmbeddedEntity>
{
	@Override
	public Translator create(final TypeKey tk, final CreateContext ctx, final Path path)
	{

		@SuppressWarnings("unchecked")
		final Class<BigMoneyProvider> clazz = (Class<BigMoneyProvider>) GenericTypeReflector.erase(tk.getTypeAsClass
				());

		if (org.joda.money.BigMoney.class.isAssignableFrom(clazz))
		{
			return new JodaMoneyTranslator(tk, ctx, path, true);
		}
		else if (org.joda.money.Money.class.isAssignableFrom(clazz))
		{
			return new JodaMoneyTranslator(tk, ctx, path, false);
		}
		else
		{
			return null;
		}
	}

	/**
	 * Translator which knows what to do with a Joda {@link org.joda.money.BigMoneyProvider}, which is an interface
	 * implemented by both {@link org.joda.money.BigMoney} and {@link org.joda.money.Money} objects.<br/> <br/> This
	 * class utilizes an optional {@link com.sappenin.objectify.annotation.Money} annotation which allows for
	 * fine-grained control over the field names used to store Money information,
	 * as well as indexing of each sub-field.
	 * See the Javadoc of that annotation for more details.
	 *
	 * @author David Fuelling
	 */
	static class JodaMoneyTranslator extends NullSafeTranslator<BigMoneyProvider, EmbeddedEntity>
	{
		private final Translator<Object, Object> componentTranslator;

		// If false, use Money; Otherwise, use bigMoney
		private boolean isBigMoney = false;

		private String encodedAmountFieldName = "encodedAmount";

		private String displayableAmountFieldName = "displayableAmount";

		private String currencyCodeFieldName = "currencyCode";

		private boolean storeDisplayableAmount;

		private boolean indexDisplayableAmount;

		private boolean indexAmount;

		private boolean indexCurrencyCode;

		/**
		 * Required Args Constructor.
		 *
		 * @param tk
		 * @param ctx
		 * @param path
		 * @param isBigMoney
		 */
		public JodaMoneyTranslator(final TypeKey tk, final CreateContext ctx, final Path path,
				final boolean isBigMoney)
		{

			Type componentType = GenericUtils.getMapValueType(tk.getType());
			componentTranslator = ctx.getFactory().getTranslators().get(new TypeKey<>(componentType, tk), ctx, path);

			this.isBigMoney = isBigMoney;

			// Look for an @BigDecimal Annotation, if present.
			Money moneyAnnotation = TypeUtils.getAnnotation(tk.getAnnotations(), Money.class);
			if (moneyAnnotation != null)
			{
				storeDisplayableAmount = moneyAnnotation.storeDisplayableAmount();
				indexCurrencyCode = moneyAnnotation.indexCurrencyCode();
				indexAmount = moneyAnnotation.indexEncodedAmount();

				indexDisplayableAmount = moneyAnnotation.indexDisplayableAmount();
				encodedAmountFieldName = moneyAnnotation.encodedAmountFieldName();
				displayableAmountFieldName = moneyAnnotation.displayableAmountFieldName();
				currencyCodeFieldName = moneyAnnotation.currencyCodeFieldName();
			}
		}

		@Override
		protected BigMoneyProvider loadSafe(EmbeddedEntity node, LoadContext ctx, Path path) throws SkipException
		{

			BigMoneyProvider returnable = null;

			// Get the amount as a java.math.BigDecimal
			if (node.hasProperty(encodedAmountFieldName))
			{
				BigDecimal bdValue = null;
				CurrencyUnit currencyUnit = CurrencyUnit.USD;

				// //////////
				// Populate bdValue (the BigDecimal Value)
				// //////////
				Object encodedAmountFieldValue = node.getProperty(encodedAmountFieldName);
				if ((encodedAmountFieldValue != null) && (encodedAmountFieldValue.toString().length() > 0))
				{
					try
					{
						bdValue = BigDecimalCodec.decodeAsBigDecimal(encodedAmountFieldValue.toString());
					}
					catch (Exception e)
					{
						System.err.print("Unable to Decode java.math.BigDecimal from encoded string \""
								+ encodedAmountFieldValue + "\"");
						e.printStackTrace();
					}
				}

				// //////////
				// Populate currencyUnit (the CurrencyUnit for the returnable
				// Money object)
				// //////////
				Object currencyCodeFieldValue = node.getProperty(currencyCodeFieldName);
				if ((currencyCodeFieldValue != null) && (currencyCodeFieldValue.toString().length() > 0))
				{
					currencyUnit = CurrencyUnit.getInstance(currencyCodeFieldValue.toString());
				}

				// /////////////////
				// Populate the BigMoney with data from above.
				// /////////////////

				if (isBigMoney)
				{
					returnable = BigMoney.of(currencyUnit, bdValue);
				}
				else
				{
					returnable = org.joda.money.Money.of(currencyUnit, bdValue);
				}
			}

			return returnable;
		}

		@Override
		protected EmbeddedEntity saveSafe(BigMoneyProvider pojo, boolean index, SaveContext ctx, Path path)
				throws SkipException
		{

			if (pojo == null)
			{
				throw new SkipException();
			}
			else
			{
				EmbeddedEntity emb = new EmbeddedEntity();

				final BigMoney bigMoney = pojo.toBigMoney();

				// /////////////////////////
				// Handle the EncodedAmount Path
				// /////////////////////////

				// Encode the Amount value as a String
				{
					String encodedAmountValue = BigDecimalCodec.encode(bigMoney.getAmount());
					Path propPath = path.extend(encodedAmountFieldName);
					Object value = componentTranslator.save(encodedAmountValue, indexAmount, ctx, propPath);
					DatastoreUtils.setContainerProperty(emb, encodedAmountFieldName, value, indexAmount, ctx,
							propPath);
				}

				// /////////////////////////
				// Handle the CurrencyCode
				// /////////////////////////
				// Encode the Amount value as a String
				{
					String currencyCodeValue = bigMoney.getCurrencyUnit().getCurrencyCode();
					Path propPath = path.extend(this.currencyCodeFieldName);
					Object value = this.componentTranslator
							.save(currencyCodeValue, this.indexCurrencyCode, ctx, propPath);
					DatastoreUtils
							.setContainerProperty(emb, this.currencyCodeFieldName, value, this.indexCurrencyCode, ctx,
									propPath);
				}

				// /////////////////////////
				// Handle the Displayable Amount
				// /////////////////////////
				if (storeDisplayableAmount)
				{

					String displayableValue = bigMoney.toString();
					Path propPath = path.extend(this.displayableAmountFieldName);
					Object value = this.componentTranslator
							.save(displayableValue, this.indexDisplayableAmount, ctx, propPath);
					DatastoreUtils.setContainerProperty(emb, this.displayableAmountFieldName, value,
							this.indexDisplayableAmount, ctx, propPath);
				}

				return emb;
			}
		}
	}

}
