/**
 * Copyright (C) 2013-2014 Sappenin Inc. (developers@sappenin.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.sappenin.objectify.translate;

import com.google.common.base.Optional;
import com.googlecode.objectify.impl.Path;
import com.googlecode.objectify.impl.translate.CreateContext;
import com.googlecode.objectify.impl.translate.LoadContext;
import com.googlecode.objectify.impl.translate.SaveContext;
import com.googlecode.objectify.impl.translate.SkipException;
import com.googlecode.objectify.impl.translate.Translator;
import com.googlecode.objectify.impl.translate.TranslatorFactory;
import com.googlecode.objectify.impl.translate.TypeKey;
import com.googlecode.objectify.impl.translate.ValueTranslatorFactory;
import com.googlecode.objectify.repackaged.gentyref.GenericTypeReflector;

/**
 * An implementation of {@link ValueTranslatorFactory} for storing values of type {@link Optional}.
 */
public class OptionalStringTranslatorFactory implements TranslatorFactory<Optional, String>
{

	@Override
	public Translator<Optional, String> create(TypeKey<Optional> tk, CreateContext ctx, Path path)
	{

		if (Optional.class.isAssignableFrom(GenericTypeReflector.erase(tk.getType())))
		{
			return new OptionalStringTranslator();
		}
		else
		{
			return null;
		}
	}

	/**
	 * An instance of {@link com.googlecode.objectify.impl.translate.Translator} that converts an {@link
	 * com.google.common.base.Optional} to a {@link java.lang.String} and back.
	 */
	public static class OptionalStringTranslator implements Translator<Optional, String>
	{

		@Override
		final public Optional load(String node, LoadContext ctx, Path path) throws SkipException
		{

			return Optional.fromNullable(node);
		}

		@Override
		public String save(Optional pojo, boolean index, SaveContext ctx, Path path) throws SkipException
		{

			if (pojo == null)
			{
				return null;
			}
			else
			{
				Object internalValue = pojo.orNull();
				return internalValue == null ? null : internalValue.toString();
			}
		}

	}

}
